; file: readdotcube = read (part of) a DOT data file with DOT filename
;       convention containing NXxNYxNT into a cube in memory
; init: sometime 2006, Tim van Werkhoven @ Utrecht
; last: Jul 19 2013  Rob Rutten  LA Deil

FUNCTION readdotcube,file,itstart=itstart,itend=itend,byt=byt

;+
; readdotcube.pro reads a DOT cube file into memory
; The file may be float/integer/byte.  Call example:
;   ha=readdotcube('AR0789-ha-wing-20050713-cube-1074x814x332-dt30',$
;                   itstart=10,itend=90,/byt)
; The type and size are extracted from the filename; this routine
; expects a cube to be named <anything>-<xsize>x<ysize>x<time>-<anything>
; Author: Tim van Werkhoven, Utrecht University, The Netherlands
; 26-06-2006: possible conflict with SSW strsplit solved using strtok
; 26-06-2006: now works with idl 5.3 or higher instead of 6.0 and higher;
;   replaced COUNT in STRTOK with a more universal alternative
; Jan 27 2013 RJR: changed to assoc; added itstart, itend for partial read
; Jul 19 2013 RJR: added option to deliver the cube bytescaled
;-

; answer to x=readddotcube() query
if (n_params() lt 1) then begin
  print,' cube = readddotcube(file, [itstart=itstart, itend=itend,byt=byt])
  print,' readddotcube finds the dimensions from the filesize and 
  print,' the dimensions specified in the filename.'
  print,' This works when filesize = n*product(dimensions)'
  print,' with n = 1,2,4 (for byte, int, float)'
 return,-1
endif

; default keywords
if (n_elements(itstart) eq 0) then itstart=0
if (n_elements(itend) eq 0) then itend=0
if (n_elements(byt) eq 0) then byt=0

; split the filename by '-'
filespl = STRTOK(file,'-',/EXTRACT)
; get the filesize to guess filetype
filesize = (FILE_INFO(file)).size

found=-1
; begin at the end and try to find input file size
for i=n_elements(filespl)-1,0,-1 do begin
 ; split each filenamepart by 'x'
 insize = STRTOK(filespl[i],'x',/EXTRACT)
 ; count the number of matches
 c = (SIZE(insize))[1]
 ; if there are 2 or more matches, we probably have the file size
 if (c gt 2) then begin    ;RR Dec 29 2010 from 1 to 2 to permit 2x2
;;  print,' Probably found size, using this string:',filespl[i]
  found=c
  break
 endif
endfor

if (found eq -1) then begin 
 print,' ==== No size found in filename :('
 return,-1
endif

insize = FIX(insize)
type = filesize/PRODUCT(insize)

if (type ne ROUND(type)) then begin
 print,' ==== WARNING'
 print,' ==== Filesize divided by product of dimensions defined in filename'
 print,'      is not a natural number, so something is wrong :('
 return,-1
endif

; now we have the dimensions
;; print,' ==== Guess for filetype: '+STRTRIM(type,2)+' bytes/pixel'
nx=insize[0]
ny=insize[1]
nt=insize[2]
if (itend eq 0) then itend=nt-1
nseg=itend-itstart+1
if (fix(type) eq 4) then datatype='float'
if (fix(type) eq 2) then datatype='integer'
if (fix(type) eq 1) then datatype='byte'

; open input file for assoc
get_lun, unit_in
openr,unit_in,file
if (datatype eq 'float') then inarr=fltarr(nx,ny)
if (datatype eq 'integer') then inarr=intarr(nx,ny)
if (datatype eq 'byte') then inarr=bytarr(nx,ny)
inassoc=assoc(unit_in,inarr) ; DOT header = 0 

; define output cube
if (datatype eq 'float') then cube=fltarr(nx,ny,nseg)
if (datatype eq 'integer') then cube=intarr(nx,ny,nseg)
if (datatype eq 'byte') then cube=bytarr(nx,ny,nseg)

; now read and fill the cube
for it=itstart,itend do cube[*,*,it-itstart]=inassoc[it]

; bytscale if requested
if (byt eq 1) then cube=bytscl(cube)

return,cube

end

