; file: plotconfusogram = plot phase difference, power and coherence spectra
; init: Aug 23 2010 
; last: Jan 31 2021  Rob Rutten  Deil

;+
;  plot Fourier phase difference, power, and coherece spectra.
;  Format described in Krijger et al. 2001A&A...379.1052K.  
;
; CALL:
;   plotconfusogram,cube1,cube2,cadence,plotfilename,$
;    apod=apod, pxdetrend=pxdetrend, meandetrend=meandetrend,$
;    navercoh=navercoh, fmax=fmax
;
; INPUTS:
;   cube1: (x,y,t) data cube, any type
;   cube2: (x,y,t) data cube, any type, delayed signal = pos phase difference
;   cadence: [regular] sampling cadence in sec
;   plotfilename: postscript output file name
;     optional keywords: 
;   apod: fractional extent of apodization edges; default = 0.1
;   pxdetrend=1: subtract linear trend with time per pixel (a bit slow)
;   pxdetrend=2: subtract linear trend with time per pixel (better but slower)
;   meandetrend: subtract linear trend with time for the image means 
;   navercoh: nr frequency bins for coherence averaging; default >= 5
;   fmax: frequency axis max as fraction of Nyquist frequency; default 1.0
;
; MODIFICATION HISTORY: started as "ruttenplot" by Bruce Lites in 1994
;  at HAO following White & Athay 1979ApJS...39..317W and continued by
;  Thijs Krijger and Alfred de Wijn in their PhD theses at Utrecht.  
;
;  Aug 2010: Rob Rutten (RR) assembly of Alfred de Wijn's routines.
;-

;-----------------------------------------------------------------------
function wrap,data,minval,maxval
    ; wraps values from outside into the range [minval,maxval]
    ; for example: wrap(200,-180,+180) = -160
    ;RR Aug 23 2010: found in Thijs Krijger's IDL
  range=maxval-minval
  ndata=data
  while (max(ndata) gt maxval)  do begin
    ndata(where(ndata gt maxval))=ndata(where(ndata gt maxval))-range
  endwhile
  while (min(ndata) lt minval)do begin
    ndata(where(ndata lt minval))=ndata(where(ndata lt minval))+range
  endwhile
  return,ndata    
end

;-----------------------------------------------------------------------
function avgstd, array, stdev=stdev
    ; get array average + standard deviation
    ;RR Aug 23 2010 found in Mabula Haverkamp's IDL, later also AdW
    ;RR not the same as avg.pro in ssw
    ;RR not the same as avg.pro in Pit Suetterlin DOT software
    ;RR so renamed to avgstd
  avrg = total(array)/n_elements(array)
  stdev = sqrt(total((array-avrg)^2)/n_elements(array))
  return, avrg
end

;-----------------------------------------------------------------------
function linear, x, p
  ; used in trend subtraction 
  ;RR Aug 23 2010 from Alfred de Wijn drawconfuse.pro
  ymod = p[0] + x * p[1]
  return, ymod
end

;-----------------------------------------------------------------------
function apodtempcube,cube,apod,meandetrend,pxdetrend
    ; apodizes the temporal (x,y,*) columns of an (x,y,t) data cube
    ; optional detrending, either mean image sequence or per (x,y) pixel 

  ; get cube dimensions
  sizecube=size(cube)
  nx=sizecube[1]
  ny=sizecube[2]
  nt=sizecube[3]

  ; initialize
  apocube=cube
  tf=findgen(nt) + 1
  col=fltarr(nt)
  apodt = fltarr(nt)+1
  if (apod ne 0) then begin
    apodrim = apod*nt
    apodt[0] = (sin(!pi/2.*findgen(apodrim)/apodrim))^2
    apodt = apodt*shift(rotate(apodt,2),1)   ;RR had ik nooit verzonnen
   endif 

  ; meandetrend: get spatially-averaged trend 
  fit=0  
  if (meandetrend) then begin
    avgseq=fltarr(nt)
    for it=0,nt-1 do avgseq[it]=total(cube[*,*,it])
    avgseq=avgseq/(double(nx)*double(ny)) 
    meanfitp = mpfitfun('linear',tf,avgseq,fltarr(nt)+1,[1000.,0.],/quiet)
      ;RR AdW; ssw; Levenberg-Marquardt least-squares fit to IDL function
      ;RR fltarr(nt)+1 = 1 sigma error = set to 1; what units?
      ;RR below: subtract mean so that meanfit is a modulation with mean=zero
    meanfit=meanfitp[0]+tf*meanfitp[1]-total(avgseq)/double(nt)
  endif 
  
  ; apodize per [x,y] temporal column = time sequence per px
  for ix=long(0),long(nx)-1 do begin  
    for iy=long(0),long(ny)-1 do begin
      col=cube[ix,iy,*]
      meancol=avgstd(col)
      if (meandetrend) then col=col-meanfit
      ;RR poly_fit for speed
      if (pxdetrend eq 1) then begin
        pxfitp=poly_fit(tf,col,1) 
        col=col-pxfitp[0]-tf*pxfitp[1]+meancol  
      endif
      ;RR pxfitp from ssw: Alfred's choice, probably better but slower
      if (pxdetrend eq 2) then begin     ;RR slow
         pxfitp = mpfitfun('linear',tf,col,fltarr(nt)+1,[meancol, 0.],/quiet)
        col=col-pxfitp[0]-tf*pxfitp[1]+meancol  
      endif 
      apocube[ix,iy,*]=(col-meancol)*apodt+meancol
   endfor
   ; printout to show (slow) progress
   if (pxdetrend ne 0) then $ 
      writeu,-1,string(format='(%"\r == detrend next row... ",i5,"/",i5)',$
        ix,nx)
  endfor
  return,apocube
end

;-----------------------------------------------------------------------
pro coherence, cube1, cube2, navercoh, coherence, sigma
    ; compute coherence spectrum with rms deviation
    ;RR Aug 23 2010 from Alfred de Wijn computeconfuse.pro
  common fourier,nt,nx,ny,cad
  nmsk = long(nx) * ny ;RR added long (had nx=ny=256 > integer word length)
  kolom = fltarr(nt)   ;RR Alfred is Dutch after all
  tf = findgen(nt) + 1
  teller = fltarr(nmsk,nt/2+1) ; and again
  noemer = fltarr(nmsk,nt/2+1)
  for i=long(0), nmsk-1 do begin
    x = i mod nx
    y = i/nx
    kolom = cube1[x,y,*]
    fft1 = (fft(kolom,-1))[0:nt/2]
    kolom = cube2[x,y,*]
    fft2 = (fft(kolom,-1))[0:nt/2]
    pow1 = float(fft1*conj(fft1))
    pow2 = float(fft2*conj(fft2))
    crpow = fft1*conj(fft2)
    pow1 = smooth(pow1, navercoh, /edge_truncate)
    pow2 = smooth(pow2, navercoh, /edge_truncate)
    crpow = smooth(crpow, navercoh, /edge_truncate)
    teller[i,*] = (float(crpow)^2 + imaginary(crpow)^2)
    noemer[i,*] = (pow1 * pow2)
  endfor

  ; compute average coherence
  coherence = fltarr(nt/2+1)
  for i=0, nt/2 do coherence[i] = avgstd(sqrt(teller[*,i]/noemer[*,i]))
  ; compute error in the average coherence
  sigma = fltarr(nt/2+1)
  for i=0, nt/2 do sigma[i] = $
    sqrt(avgstd((sqrt(teller[*,i]/noemer[*,i])-coherence[i])^2))
end

;-----------------------------------------------------------------------
pro power, cube, meanpower, sigma
  ; compute (x,y) averaged power spectrum with rms deviations
  ;RR Aug 23 2010: from Alfred de Wijn computeconfuse.pro
  common fourier,nt,nx,ny,cad
  tf = findgen(nt) + 1
  nmsk = long(nx) * ny  ;RR inserted long
  kolom = fltarr(nt)
  powcube = fltarr(nmsk,nt/2+1)
  for i=long(0), nmsk-1 do begin
    x = i mod nx   ;RR wat doettie hier?
    y = i/nx
    kolom = cube[x,y,*]
    fftcube = (fft(kolom,-1))[0:nt/2]
    powcube[i,*] = float(fftcube*conj(fftcube))
  endfor

  ; compute mean power and error
  meanpower = fltarr(nt/2+1)
  sigma = fltarr(nt/2+1)
  for i=0, nt/2 do begin
    meanpower[i] = avgstd(powcube[*,i], stdev=tmpsig)
    sigma[i] = tmpsig
  endfor
end

;-----------------------------------------------------------------------
pro dphi, cube1, cube2, nbins, meandphi, sigma, gray, dphi_length, dphi_noise
  ; compute phase difference spectra
  ;RR Aug 23 2010: from Alfred de Wijn computeconfuse.pro
  common fourier,nt,nx,ny,cad
  dtime = 0.
  sigmadt = 0.
  sigmac = 0.
  ;RR Aug 23 2010 parameters above were for non-sychronous TRACE timing
;;  frequencies = 1./(cad*2)*findgen(nx)/(nx-1)  ; wrong Ding Yuan 9 Apr 2012
  frequencies = 1./(cad*2)*findgen(nt/2+1)/(nt/2)  
  deltaphioffset = 2*!pi*dtime*frequencies
  tf = findgen(nt) + 1
  nmsk = long(nx) * ny ;RR added long
  kolom = fltarr(nt)
  teller = fltarr(nmsk,nt/2+1)
  noemer = fltarr(nmsk,nt/2+1)
  dphi_cube = fltarr(nmsk,nt/2+1)
  dphi_wght = fltarr(nmsk,nt/2+1)
  for i=long(0), nmsk-1 do begin
    x = i mod nx
    y = i/nx
    kolom = cube1[x,y,*]
    fft1 = (fft(kolom,-1))[0:nt/2]
    kolom = cube2[x,y,*]
    fft2 = (fft(kolom,-1))[0:nt/2]
    powcube1 = float(fft1*conj(fft1))
    powcube2 = float(fft2*conj(fft2))
    crpow = fft1*conj(fft2)             ;RR cross power
    teller[i,*] = imaginary(crpow)
    noemer[i,*] = float(crpow)
    dphi_cube[i,*] = wrap(atan(imaginary(crpow),float(crpow))$
                     +deltaphioffset,-!pi,!pi)   
      ;RR atan always [-pi,+pi]; wrap needed for offset addition
    dphi_wght[i,*] = sqrt(powcube1*powcube2)
  endfor

  ; make the grayscale
  ;RR nt/2+1 = full Fourier scale up to Nykvist
  gray = fltarr(nt/2+1,nbins)
  binsize = 2*!pi/(nbins)
  for i=0, nt/2 do begin
    tmpdphi = dphi_cube[*,i]
    tmpwght = dphi_wght[*,i]
    for j=0, nbins-1 do begin
      waar = where(tmpdphi ge -!pi+j*binsize and $
                   tmpdphi lt -!pi+(j+1)*binsize)
      if waar[0] ne -1 then gray[i,j] = total(tmpwght[waar])
    endfor
  endfor
  meandphi = fltarr(nt/2+1)
  sigma = fltarr(nt/2+1)
  for i=0, nt/2 do begin
    avgtel = avgstd(teller[*,i],stdev=stdtel)
    avgnoem = avgstd(noemer[*,i],stdev=stdnoem)
    meandphi[i] = wrap(atan(avgtel,avgnoem)+deltaphioffset[i],-!pi,!pi)
    sigma[i] = sqrt(total((meandphi[i]-dphi_cube[*,i])^2*dphi_wght[*,i])/$
                total(dphi_wght[*,i]))
  endfor
end

;-----------------------------------------------------------------------
pro plotconfuse, coh, sigmacoh, meandphi, sigmadphi, dphi_gray, $
	pow1, sigmapow1, pow2, sigmapow2, navercoh, fmax, plotfilename
  ; plots confusogram
  ;RR Aug 23 2010 from Alfred de Wijn drawconfuse.pro
  common fourier,nt,nx,ny,cad

  ; set plot extent
  sizedphi = size(dphi_gray)
  plotrange = [fix(sizedphi[1]*fmax),sizedphi[2]]
  plotgray =  dphi_gray[0:plotrange[0]-1,0:plotrange[1]-1]
  sizeplotgray=size(plotgray)
  xas = fmax*1./(cad*2)*findgen(plotrange[0])/(plotrange[0]-1)*1e3 
  xrange = [min(xas),max(xas)]

  ;RR parameters controlling the plot areas
  delta1 = 0.20  ;RR changed 0.15>0.20 to get y-labels off the axes
  delta2 = 0.05
  delta3 = 0.033
  aspect = 3./2
  ticklen = -0.02
  x = 1 - delta1 - delta2 - delta3
  y = (aspect - 2*delta1 - delta2 - x)
  p1 = [delta1, delta1/aspect, 1-delta3, (delta1+y)/aspect]
  p2 = [delta1, (delta1+y+delta2)/aspect, 1-delta3, $
        (delta1+y+delta2+x)/aspect]
;RR  xticks = 8    ;RR Alfred had 8 for 30s cadence; let IDL choose instead
;RR  xtickpos = findgen(xticks+1)*2   ;RR idem
  emptyxticks = replicate(' ',60) ;RR max value to be sure to blank all
  if (xrange[1] le 10) then begin ;RR I wonder how to automate this  
    topxtickpos = [20,10,5,4,3,2,1]
    topxticksn = ['20','10','5','4','3','2','1']
  endif else if (xrange[1] le 20) then begin
    topxtickpos = [10,5,4,3,2,1.5,1,0.5]
    topxticksn = ['10','5','4','3','2','1.5','1','0.5']
  endif else if (xrange[1] le 50) then begin
    topxtickpos = [5,2,1,0.5,0.2,0.1]
    topxticksn = ['5','2','1','0.5','0.2','0.1']
  endif else begin
    topxtickpos = [2,1,0.5,0.2,0.1]
    topxticksn = ['2','1','0.5','0.2','0.1']
  endelse
  topxticks=n_elements(topxtickpos)-1
  topxtickv = 1e3/topxtickpos/60  ;RR period, from mHz to min

  ; plot startup:  what windows?
  windowsdevice='X'
  if (!d.name eq 'WIN') then windowsdevice='WIN'

  ;RR inserted Alfred's startplot parameters instead of calling startplot
  set_plot, 'ps'
  !p.font=1
  !p.thick=3
  !x.thick=3
  !y.thick=3
  !z.thick=3
  device, filename=plotfilename,xsize=8.8, ysize=8.8*aspect, $
    set_font='Times', /tt_font, font_size=13, bits_per_pixel=8

  ; lower panel = power+coherence spectra
  powyrange = [0,1]
  plot, xrange, powyrange, /nodata, xstyle=9, ystyle=1, position=p1, $
    xticklen=ticklen/(p1[3]-p1[1])*(p1[2]-p1[0])/aspect, yticklen=ticklen,$
     /noerase, xminor=1, yminor=1, yticks=4 
;RR    xminor=1, xticks=xticks, xtickv=xtickpos, yminor=1, yticks=4
  axis, /xaxis, xticks=topxticks, xtickv=topxtickv, xtickname=emptyxticks, $
    xticklen=ticklen/(p1[3]-p1[1])*(p1[2]-p1[0])/aspect, yticklen=ticklen, $
    /noerase, xminor=1
  xyouts, delta1/4, (delta1+y/2)/aspect, 'power and coherence', $
    /normal, orientation=90, alignment=0.5
  xyouts, delta1+x/2, 0.05, 'frequency [mHz]', /normal, alignment=0.5
  oplot, xas, pow1/pow1[1]  ; solid
  oplot, xas, pow2/pow2[1], linestyle=2 ; dashed

   ;RR normalized by power at f=1 (feels trend removal)
  oplot, xas, coh, linestyle=3 ; dash-dotted

    ;RR error estimate for navercoh = nr smooth bins in frequency
  oplot, xrange, [1,1]/sqrt(navercoh), linestyle=1 ; dotted

  ; phase difference grayscale plot
  grayscale = fltarr((sizeplotgray[1]-1)*2,sizeplotgray[2])
  grayscale[0,*] = 1-plotgray[0,*]/max(plotgray[0,*])
  for i=1,sizeplotgray[1]-2 do begin
    grayscale[2*i-1,*] = 1-plotgray[i,*]/max(plotgray[i,*])
    grayscale[2*i,*] = 1-plotgray[i,*]/max(plotgray[i,*])
  endfor
  grayscale[(sizeplotgray[1]-1)*2-1,*] = 1-plotgray[sizeplotgray[1]-1,*]/$
    max(plotgray[sizeplotgray[1]-1,*])
  tv, bytscl(grayscale, top=180)+75, $  ;RR Alfred top=128)+127
    p2[0], p2[1], xsize=p2[2]-p2[0], ysize=p2[3]-p2[1], /normal
  plot, xrange, [-1,1]*180, /nodata, xstyle=9, ystyle=1, position=p2, $
      /noerase, $
     xticklen=ticklen/(p2[3]-p2[1])*(p2[2]-p2[0])/aspect,$
     xminor=1, xtickname=emptyxticks,$  ;RR xticks=xticks, xtickv=xtickpos, $
     yticklen=ticklen, yminor=1, ytickv=indgen(9)/8.*360-180, yticks=8
  axis, /xaxis, xticks=topxticks, xtickv=topxtickv, xtickname=topxticksn, $
     xticklen=ticklen/(p2[3]-p2[1])*(p2[2]-p2[0])/aspect, yticklen=ticklen, $
     /noerase, xminor=1
  xyouts, delta1+x/2, (aspect-delta1/2)/aspect, 'period [min]', /normal, $
    alignment=0.5
  xyouts, delta1/4, (delta1+delta2+y+x/2)/aspect, $
     'phase difference [degrees]',/normal,orientation=90, alignment=0.5

  ; overplot mean phase difference curve
  oplot, xas[1:*], meandphi[1:*]*180/!pi

  ; overplot dotted phase difference zero line
  oplot, xrange, [0,0], linestyle=1

  ; done
  device, /close
  set_plot, windowsdevice

  ; convert the IDL ad so that gv shows the plotfilename as window label
  if (windowsdevice eq 'X') then begin
    spawn,'cat '+plotfilename+$
      '| sed "s|Graphics produced by IDL|'+plotfilename+$
      '|" >  idltemp.ps; mv idltemp.ps '+plotfilename
  endif

end

; ====================== MAIN ROUTINE ================================

pro plotconfusogram,cube1,cube2,cadence,plotfilename,$
  apod=apod,pxdetrend=pxdetrend,meandetrend=meandetrend,$
  navercoh=navercoh,fmax=fmax
    ; plots the confusogram
    ;RR from Alfred de Wijn's "pro doallconfuse" in drawconfuse.pro
if (n_elements(apod) ne 0) then apod=apod else apod=0.1
if (n_elements(pxdetrend) ne 0) then pxdetrend=pxdetrend else pxdetrend=0
if not keyword_set(meandetrend) then meandetrend=0
if (n_elements(navercoh) ne 0) then navercoh=navercoh else navercoh=0
if (n_elements(fmax) ne 0) then fmax=fmax else fmax=1.0

if (n_params() lt 4) then begin
  print, 'plotconfusogram,cube1,cube2,cadence,plotfilename,'
  print, '  [,apod=apod, pxdetrend=pxdetrend, meandetrend=meandetrend,'
  print, '   mavercoh=navercoh, fmax=fmax]'
  print, '  cubes must be equal size and type; cadence in sec'
  return
endif

common fourier,nt,nx,ny,cad

; define common parameters
sizecube=size(cube1)
nx=sizecube[1]
ny=sizecube[2]
nt=sizecube[3]
cad=float(cadence)

; apodize both cubes
apocube1=apodtempcube(cube1,apod,meandetrend,pxdetrend)
apocube2=apodtempcube(cube2,apod,meandetrend,pxdetrend)

; compute power spectra
power,apocube1,powc1,sigmapowc1
power,apocube2,powc2,sigmapowc2

; compute coherence spectrum
if (navercoh eq 0) then navercoh=max([nt/10+(nt/20 eq nt/20.),5])
  ;RR 1/10th uneven, minimum 5 
  ;RR Alfred de Wijn used 3, Thijs Krijger 5, Bruce Lites 9
coherence, apocube1, apocube2, navercoh, cohc1c2, sigmacohc1c2

; compute phase difference spectra
dphi, apocube1, apocube2, 359, dphic1c2, sigmadphic1c2, grayc1c2
  ;RR number = number of bins in y direction (phase diff bins)
  ;RR Alfred had nbins=119

; plot confusogram
plotconfuse, cohc1c2, sigmacohc1c2, dphic1c2, sigmadphic1c2, grayc1c2, $
  powc1, sigmapowc1, powc2, sigmapowc2, navercoh, fmax, plotfilename

; done
print, ' === wrote ', plotfilename

end
; ======================== main test Hyper-C

cd,'/media/rutten/RRHOME/alldata/ALMA/2018-04-12-ca/sdo/target/cubes'
trimbox=[54,67,183,194] ; found per showex almaaia.fits (command line)
reformcubefile,'almaaia.fits','almatrim.fits',trimbox=trimbox
reformcubefile,'aia304.fits','aia304trim.fits',trimbox=trimbox
reformcubefile,'gongaia.fits','gongtrim.fits',trimbox=trimbox

cube1=readfits('aia304trim.fits')
cube2=readfits('almatrim.fits')
plotfilename='/tmp/aia304_alma_confusogram.ps'

;; cube1=readfits('almatrim.fits')
;; cube2=readfits('gongtrim.fits')
;; plotfilename='/tmp/alma_gong_confusogram.ps'

;; cube1=readfits('aia304.fits')
;; cube2=readfits('gongaia.fits')
;; plotfilename='/tmp/aia304_gong_confusogram.ps'

cadence=12 ; AIA cadence
plotconfusogram,cube1,cube2,cadence,plotfilename,$
      apod=apod,pxdetrend=pxdetrend,meandetrend=meandetrend,$
      navercoh=navercoh,fmax=fmax

spawn,'gv '+plotfilename
spawn,'rm -f *trim.fits'

end


