FUNCTION LMBD_MATRIX,TAU,MU

;; TO DETERMINE THE LAMBDA MATRIX
;; BASED ON FEAUTRIER FUNCTION OF HAN UITENBROEK
;       Evaluate monochromatic intensities I(+), I(-) and emergent
;       intensity I(surf) along a ray with given optical depth scale
;       TAU and source function S. Second order Feautrier method
;       formulated as in Appendix A of:
;                       G.B. Rybicki & D.G Hummer, A&A 245, 171-181.
;;      Second-order boundary conditions, see for example
;                       L. Auer, ApJ 150, L53-55, 1967

;; INPUT:
;       TAU   --  optical depth scale
;       S     --  monochromatic source function
;       MU    --  point in chosen quadrature

;; OUTPUT:
;       matrix for lambda transformation

;; CALLING SEQUENCE:
;  RESULT = LMBD_MATRIX ( TAU , MU )

;       TAU_L --  line optical depth
;       R0    --  upper boundary condition: I(-)= R0*I(+),   at tau(0)
;       RN    --  lower boundary condition: I(+)= RN*I(-),   at tau(nd-1)

 tau_l=tau/mu
 R0=0
 RN=1

 nd   = n_elements(tau_l)
 dtau = shift(tau_l,-1)-tau_l    ; Don't need dtau(ndep-1)
 dts  = dtau+ shift(dtau,1)      ; Don't need dts(0)

 a = 2.0 / (dts*shift(dtau,1))
 c = 2.0 / (dts*dtau)
 abc = 1.0 + fltarr(nd)

;;  UPPER BOUNDARY CONDITION:  I(-) = R0*I(+)
 ff0  = (1.0 - r0) / (1.0 + r0)
 abc0 = 1.0 + 2.0*ff0/dtau(0)
 c0 = 2.0 / dtau(0)^2

;;  LOWER BOUNDARY CONDITION:  I(+) = RN*I(-)
 ffn  = (1.0 - rn) / (1.0+rn)
 abcn = 1.0 + 2.0*ffn / dtau(nd-2)
 an = 2.0 / dtau(nd-2)^2

;;  START THE ELIMINATION
 f = fltarr(nd)
 z_mat  = fltarr(nd,nd)
 f(0) = abc0 / c0
 z_mat(0,0) = 1. / (abc0+c0)
 FOR k=1,nd-2 DO BEGIN
  f(k) = (abc(k) + a(k)*f(k-1)/(1.0 + f(k-1))) / c(k)
  z_mat(0:k,k) = 1/( c(k)*(1+f(k)) )*[ a(k)*z_mat(0:k-1,k-1) , 1 ]
 ENDFOR

 mat=fltarr(nd,nd)
 mat(*,nd-1) = $
 1/ ( abcn + an*(f(nd-2)/(1.0 + f(nd-2))))*[an*z_mat(0:nd-2,nd-2) ,1]

;;  NOW BACKSUBSTITUTE
 FOR  k=nd-2,0,-1 DO BEGIN
  mat(*,k)=1/(1.+f(k))*mat(*,k+1)+z_mat(*,k)
 ENDFOR

RETURN,mat
END
