FUNCTION FEAUTRIER,TAU,SOURCE,R0,H0,RN,HN
;+
; NAME:
;       FEAUTRIER
; PURPOSE:
;       Evaluate monochromatic intensities I(+), I(-) and emergent
;       intensity I(surf) along a ray with given optical depth scale
;       TAU and source function S. Second order Feautrier method
;       formulated as in Appendix A of:
;                       G.B. Rybicki & D.G Hummer, A&A 245, 171-181.
; CATEGORY:
; CALLING SEQUENCE:
;       Isurf = FEAUTRIER(tau, source, r0, h0, rn, hn)
; INPUTS:
;       TAU     --  optical depth scale
;       SOURCE  --  monochromatic source function
;       R0,H0   --  upper boundary condition: I(-)= R0*I(+)+ H0,   at tau(0)
;       RN,HN   --  lower boundary condition: I(+)= RN*I(-)+ HN,   at tau(nd-1)
; KEYWORD PARAMETERS:
; OUTPUTS:
;       Intensities I(+) and I(-) along the ray, and emergent intensity I(surf)
; MODIFICATION HISTORY:
;       Han Uitenbroek, September 1991
;       modified, February 1998
;-

  nd   = n_elements(tau)
  dtau = shift(tau,-1)-tau        ; Don't need dtau(ndep-1)
  dts  = dtau+ shift(dtau,1)      ; Don't need dts(0)

  a = 2.0 / (dts*shift(dtau,1))
  c = 2.0 / (dts*dtau)
  y = source
  kk = 1.0 + fltarr(nd)

  ;;  Upper boundary condition:  I(-) = R0*I(+) + H0

  c0 = 2.0 / dtau(0)^2
  y0 = source(0) + 2.0*h0 / ((1.0 + r0)*dtau(0))
  kk0 = 1.0 + ( 2.0 / dtau(0) ) * (1.0 - r0) / (1.0 + r0)

  ;;  Lower boundary condition:  I(+) = Rn*I(-) + Hn

  an = 2.0 / dtau(nd-2)^2
  yn = source(nd-1) + 2.0*hn / ((1.0 + rn)*dtau(nd-2))
  kkn = 1.0 + ( 2.0 / dtau(nd-2) ) * (1.0 - rn) / (1.0 + rn)

  ;;  Start the elimination

  f = fltarr(nd)  &  z=f
  f(0) = kk0 / c0
  z(0) = y0 / (kk0+c0)
  FOR j=1,nd-2 DO BEGIN
    f(j) = (kk(j) + a(j)*f(j-1)/(1.0 + f(j-1))) / c(j)
    z(j) = (y(j) + a(j)*z(j-1))/(c(j) * (1.0 + f(j)))
  ENDFOR

  ;;  Now backsubstitute

  p = fltarr(nd)  &  q = p  &  pms = p
  p(nd-1) = (yn + an*z(nd-2)) / (kkn + an*(f(nd-2)/(1.0 + f(nd-2))))
  FOR  j=nd-2,0,-1  DO  p(j) = p(j+1) / (1.0+f(j)) + z(j)

  pms(0) = p(0) - source(0)
  q(0) = (p(0) * (1.0 - r0) - h0)/(1.0 + r0)
  qmid = q
  qmid(0) = qmid(0) + 0.5*dtau(0)*pms(0)
  FOR j=1,nd-2 DO BEGIN
    IF (abs(a(j)) gt 1.0) THEN $
     pms(j) = p(j) - source(j) $
    ELSE $
     pms(j) = c(j)*(p(j+1) - p(j)) - a(j)*(p(j) - p(j-1))

    qmid(j) = qmid(j-1) + 0.5*dts(j)*pms(j)
  ENDFOR
  FOR j=1,nd-2 DO q(j) = ( dtau(j)*qmid(j-1) + dtau(j-1)*qmid(j) )/dts(j)
  pms(nd-1) = p(nd-1)-source(nd-1)
  q(nd-1) = -  ((1.0 - rn) / (1.0+rn)) * p(nd-1) + hn/(1.0 + r0)

  iplus  = p + q
  iminus = p - q
  isurf  = (iplus(0)-source(0))*exp(-tau(0)) + source(0)
RETURN, Isurf
END

