; file: sst_reform5d.pro
; init: Jun 23 2022  Rob Rutten  Deil
; last: Aug 18 2022  Rob Rutten  Deil
; todo: also grey triangles per channel when they are not NaN

;+
pro sst_reform5d,infile,outfile,$
  noflt2int=noflt2int,taionly=taionly,nrtimes=nrtimes,verbose=verbose
 ;
 ; sanitize a SSTRED "5D" fits file into a SSTRR 3D fits file
 ; with naxis3 = nrtimes*nrwavs*nrstokes (as classic LP files)
 ; including de-NaN into grey and optional float-to-integer conversion
 ; to halve the silly SSTRED float file size.
 ;
 ; INPUTS:
 ;   infile = string 'path/sst5dfile,fits'
 ;   outfile = string 'path/reformedfile.fits'
 ;
 ; OPTIONAL INPUTS:
 ;   noflt2int = 1/0: do NOT change intensities to integers (max 32000)
 ;   taionly = 1/0: only write ssttimetai file (outfile_tai.dat)
 ;   verbose = 1/0 more/less printout in terminal
 ;
 ; OUTPUTS:
 ;   outfile: dimension-stripped deNaNned SSTRED "5D" fits file in 3D
 ;   outfile_tai.dat: list of exposure TAI times from infile header
 ;   nrtimes: number of time steps to showex multi-wav per nt_mw
 ;   
 ; HISTORY:
 ;   Jun 26 2022 RR: start
 ;   Aug 13 2022 RR: write ssttimetai = outfile_tai.dat 
;-

; answer no-parameter query
if (n_params() lt 2) then begin
  sp,sst_reform5d
  return
endif

; default keywords
if (n_elements(noflt2int) eq 0) then noflt2int=0
if (n_elements(taionly) eq 0) then taionly=0
if (n_elements(verbose) eq 0) then verbose=0

; set wall-clock timer (seconds)
timestart=systime(1) 

; set endian for this computer 
;RR l = little, b = big; fits files are big; linux machines are little
if ((byte(1L, 0, 1))[0] eq 1) then endian = 'l' else endian='b'

; check infile and outfile have fits extension
dummy=cgrootname(infile,extension=ext) ;RR needs coyotelib
if (ext ne 'fits') then begin
  print,' ##### sst_reform5d.pro aborts: infile not extension fits'
  return
endif
dummy=cgrootname(outfile,extension=ext) ;RR needs coyotelib
if (ext ne 'fits') then begin
  print,' ##### sst_reform5d.pro aborts: outfile not extension fits'
  return
endif

; get data type and cube dimensions
header_in=headfits_rr(infile) ; _rr admits spaces in Mac directories
header_insize=(1+fix(n_elements(header_in)/36.))*2880
bitpix_in=fxpar(header_in,'bitpix')
endian_in='b'  ; fits files are always big endian
nx_in=fxpar(header_in,'naxis1')
ny_in=fxpar(header_in,'naxis2')
ndims=fxpar(header_in,'naxis')

; file must be SSTRED 5D
if (ndims eq 5) then begin  
  nw_in=fxpar(header_in,'naxis3') ; nr wavelengths (CRISP, CHROMIS, wide band)
  ns_in=fxpar(header_in,'naxis4') ; nr Stokes (I,Q,U,V)
  nt_in=fxpar(header_in,'naxis5') ; nr time steps
endif else begin
  print,' ##### # sst_reform5d.pro abort: not 5D SSTRED file'
  return
endelse
nz_in=nw_in*ns_in*nt_in  ; total 3rd assoc dimension
nrchan=nz_in/nt_in       ; nr of "channels" = nrwavs*nstokes
nrtimes=nt_in            ; nr time steps
nrwavs=nw_in             ; nr wavelengths
nrstokes=ns_in           ; nr Stokes parameters

; get the exposure times in TAI from header 
datetimestart=fxpar(header_in,'DATE-BEG') ; '2020-08-13Tx07:58:16.12499'
datetimeend=fxpar(header_in,'DATE-END')   ; end of scan so + cadence
cadence_sst=fxpar(header_in,'CADAVG')     ; average cadence
taistart=anytim2tai(datetimestart)
timearr_sst=+indgen(nrtimes)*cadence_sst+taistart  ; TAI array
taiend=timearr_sst[nrtimes-1]                      ; start of last F-P scan
  ; check
if (abs(anytim2tai(datetimeend)-taiend) gt cadence_sst/2) then begin
  print,' ##### sst_findlocation abort: SSTRED endtime conflict'
  STOP
endif

; get file path and extension for outfile_tai.dat and write that, maybe quit
outonly=cgrootname(outfile,director=outpath,extension=outext) ;RR coyotelib
ssttimetai=outpath+outonly+'_tai.dat'
writecol,ssttimetai,timearr_sst,fmt='(5E21.12)'
if (taionly eq 1) then return

; define input assoc frame size 
if (bitpix_in eq 8) then inarr=bytarr(nx_in,ny_in)
if (bitpix_in eq 16) then inarr=intarr(nx_in,ny_in)
if (bitpix_in eq -32) then inarr=fltarr(nx_in,ny_in)

; define output assoc frame size 
nx_out=nx_in
ny_out=ny_in
nz_out=nz_in

; open infile for assoc
get_lun, unit_in
openr,unit_in,infile,swap_endian=(endian ne endian_in)
inassoc=assoc(unit_in,inarr,header_insize)

; output header
header_out=header_in
sxaddpar,header_out,'naxis',3,'RR 3D assocable, not SSTRED 5D' 
sxaddpar,header_out,'naxis1',nx_out
sxaddpar,header_out,'naxis2',ny_out
sxaddpar,header_out,'naxis3',nz_out,'RR nrtimes*nrwavs*nrstokes SSTRED'
sxdelpar,header_out,'naxis4'  ; take off
sxdelpar,header_out,'naxis5'  ; take off
sxaddpar,header_out,'nrchan',nrchan,'RR nr channels SSTRED'
sxaddpar,header_out,'nrtimes',nrtimes,'RR nr timesteps SSTRED'
sxaddpar,header_out,'nrwavs',nrwavs,'RR nr wavelengths SSTRED'
sxaddpar,header_out,'nrstokes',nrstokes,'RR nr Stokes params SSTRED'
sxaddpar,header_out,'starttim',datetimestart,'RR start date-time UT SSTRED'
sxaddpar,header_out,'cadence',cadence_sst,'RR cadence in s SSTRED'

; size header_out
sizeheader_out=size(header_out)  
; fits header = Nx36 "card images" = Nx2880 bytes
header_outsize=(1+fix(sizeheader_out[1]/36.))*2880

; open outfile for assoc
endian_out='b'
bitpix_out=bitpix_in
get_lun, unit_out
openw,unit_out,outfile,swap_endian=(endian ne endian_out)
if (bitpix_out eq -32) then outassoc=assoc(unit_out,fltarr(nx_out,ny_out),$
                                           header_outsize)
if (bitpix_out eq 16) then outassoc=assoc(unit_out,intarr(nx_out,ny_out),$
                                          header_outsize)
if (bitpix_out eq 8) then outassoc=assoc(unit_out,bytarr(nx_out,ny_out),$
                                         header_outsize)
if (header_outsize ne 0) then begin
  rec=assoc(unit_out, bytarr(header_outsize))
  rec[0]=byte(header_out)
endif

; loop over "channels" = all wavs and Stokes (set nangrey per channel)
; --------------------------------------------------------------------
for ichan=0,nrchan-1 do begin

; patience
  if (verbose and nrchan gt 10 and ichan/10. eq ichan/10) then $
    print,' ----- progress: next channel ='+trimd(ichan)+' of'+trimd(nrchan)
  
; set nangrey per wav from midtime center part (maintain intensity range)
  iz=(nrtimes/2)*nrchan+ichan
  immid=reform(inassoc[iz])
  imcut=immid[nx_in/2-nx_in/4:nx_in/2+nx_in/4,$  
              ny_in/2-ny_in/4:ny_in/2+ny_in/4]
  nangrey=avg(imcut)
  
; loop over time steps for this channel
  for it=0,nrtimes-1 do begin
    iz=it*nrchan+ichan
    im=inassoc[iz]

; de-NAN making NaNs grey (same grey value per channnel) (bad SSTRED practice)
    if (ichan eq 0 and it eq 0) then begin
      if (product(finite(im)) eq 0) then $
        print,' ----- NaN in first image; all such become grey'
    endif
    denanarray,im,replace=nangrey
    outassoc[iz]=im
  endfor
endfor

; done
free_lun,unit_in,unit_out

; optional noflt2int conversion
if (noflt2int eq 0) then begin
  if (verbose) then print,' ----- progress: start flt2int, takes [very] long'
  tmprescalefile='tmp_sst_reform5d_flt2int.fits'
  spawn,'mv '+outfile+' '+tmprescalefile
  reformcubefile,tmprescalefile,outfile,/flt2int,/intrescale
  spawn,'rm -f '+tmprescalefile
endif

; print elapsed time
timedone=systime(1)
timelaps=ntostr((timedone-timestart)/60.,format='(F11.1)')
if (verbose) then print,' ===== sst_reform5D took '+timelaps+' minutes'

end

; =============== test per IDLWAVE S-c ==============================

cd,'/media/rutten/RRHOME/alldata/SST/2020-08-13-AR-5D/sst/crisp'

infile='nb_8542_2020-08-13T07:58:01_scans=0-82_stokes_corrected_im.fits'
outfile='nb_8542_sstrr.fits'

sst_reform5d,infile,outfile,nrtimes=nrtimes,/verbose ;; ,/taionly
showex,outfile,nt_mw=nrtimes

end



