; file: sst_mwfits2fitscube.pro
; init: Jun 23 2022  Rob Rutten  Deil from earlier trial crispfits2fits.pro
; last: Aug 18 2022  Rob Rutten  Deil
; todo: rescaling for filterlike weighted summing range of wav indices

;+
pro sst_mwfits2fitscube,infile,outfile,$
  iwav=iwav,istokes=istokes,$
  xrange=xrange,yrange=yrange,trange=trange,$
  intfile=intfile,intcube=intcube,$
  nx_out=nx_out,ny_out=ny_out,nt_out=nt_out,$
  verbose=verbose

 ;   write integer (or byte) single-wav [partial] SST cubefile 
 ;   from a SSTRR 3D or SSTRED "5D" multi-wavelength CRISP or CHROMIS file
 ;
 ; INPUTS:
 ;   infile = string 'path/sstmwfile,fits', either SSTRED or SSTRR product
 ;   outfile = string 'path/outfile.fits'
 ;
 ; OPTIONAL INPUTS:
 ;   iwav = index selected wavelength (start at 0, default 0)
 ;     may also be 2-elem vector defining summation range 
 ;   istokes = index Stokes param (0,1,2,3 = I,Q,U,V, default 0)  
 ;   xrange=xrange,yrange=yrange,trange=trange: limit cube ranges
 ;   intfile = 1/0: apply flt2int over range of whole input file 
 ;   intcube = 1/0: apply flt2int over range output cubefile
 ; 
 ; OUTPUTS:
 ;   fitscube file [x,y,t] for the selected wavelength(s)
 ;   file outfile_tai.dat list = exposure TAI times from infile header
 ;
 ; OPTIONAL OUTPUTS:
 ;   nx_out, ny_out, nt_out: output cube dimensions for further use
 ;
 ; PROBLEMS?
 ;   tests with SSTRED "5D" files showed some having NaN values for
 ;   the image area outside the detector frame.  These obstruct any
 ;   IDL program.  Remedy: use my sst_reform5D.pro to clean them away.
 ; 
 ; HISTORY:
 ;   Jun 23 2022: start for problematic SSTRED "5D" fits files
 ;   Jun 28 2022: also for "SSTRR 3D" files made with sst_reform5d.pro
 ;   Aug 13 2022:  write ssttimetai = outfile_tai.dat 
;-

; answer no-parameter query
if (n_params() lt 2) then begin
  sp,sst_mwfits2fitscube
  return
endif

; default keywords
if (n_elements(iwav) eq 0) then iwav=0
if (n_elements(istokes) eq 0) then istokes=0
if (n_elements(intfile) eq 0) then intfile=0
if (n_elements(intcube) eq 0) then intcube=0
if (n_elements(xrange) eq 0) then xrange=[0,-1]
if (n_elements(yrange) eq 0) then yrange=[0,-1]
if (n_elements(trange) eq 0) then trange=[0,-1]
if (n_elements(verbose) eq 0) then verbose=0

; check intcube vs intfile)
if (intfile*intcube eq 1) then begin
  print,' ##### sst_mwfits2fitscube abort: both intfile and intcube set'
  return
endif

; get and check iwav dimension
wavdim=n_elements(iwav)
if (wavdim gt 2) then begin
  print,' ##### sst_mwfits2fitscube abort: iwav not constant or index range'
  return
endif

; set endian for this computer 
;RR l = little, b = big; fits files are big; linux machines are little
if ((byte(1L, 0, 1))[0] eq 1) then endian = 'l' else endian='b'

; get data type and cube dimensions
header_in=headfits_rr(infile) ; _rr admits spaces in Mac directories
header_insize=(1+fix(n_elements(header_in)/36.))*2880
bitpix_in=fxpar(header_in,'bitpix')
endian_in='b'  ; fits files are always big endian
nx_in=fxpar(header_in,'naxis1')
ny_in=fxpar(header_in,'naxis2')
ndims=fxpar(header_in,'naxis')


if (ndims eq 3) then begin
; sanitized SSTRR file made with sst_reform5d.pro from SSTRED 5D file
  nrchan_in=fxpar(header_in,'nrchan')
  if (nrchan_in eq 0) then begin
    print,' ##### sst_mwfits2fitscube abort: 3D file not from sst_reform5d'
    return
  endif
  if (verbose) then print,' ----- infile is SSTRR 3D format'
  nw_in=fxpar(header_in,'nrwavs')
  ns_in=fxpar(header_in,'nrstokes')
  nt_in=fxpar(header_in,'naxis3')/nrchan_in  ; nr time steps
  if (fxpar(header_in,'nrtimes') ne nt_in) then begin
    print,' ##### sst_mwfits2fitscube abort: '+$
      'nr time steps not equal to SSTRR header nrtimes'
    return
  endif
  if (nrchan_in ne nw_in*ns_in) then begin
    print,' ##### sst_mwfits2fitscube abort: '+$
      'nr channels not equal nr wavs * nr stokes'
    return
  endif    
endif else  if (ndims eq 5) then begin  
; original SSTRED 5D file
  if (verbose) then print,' ----- infile is SSTRED "5D" format'
  nw_in=fxpar(header_in,'naxis3') ; nr wavelengths (CRISP, CHROMIS, wide band)
  ns_in=fxpar(header_in,'naxis4') ; nr Stokes (I,Q,U,V)
  nt_in=fxpar(header_in,'naxis5') ; nr time steps
  nrchan_in=nw_in*ns_in
endif else begin
  print,' ##### sst_mwfits2fitscube abort: not SSTRR 3D nor SSTRED 5D file'
  return
endelse
nz_in=nrchan_in*nt_in  ; total 3rd assoc dimension
nrtimes=nt_in         ; true number time steps

; get the exposure times in TAI from header for outfile_tai.dat
datetimestart=fxpar(header_in,'DATE-BEG') ; '2020-08-13T07:58:16.12499'
datetimeend=fxpar(header_in,'DATE-END')   ; end of scan so + cadence
cadence_sst=fxpar(header_in,'CADAVG')     ; average cadence
  ; check
if (datetimestart eq 0 or datetimeend eq 0 or cadence_sst eq 0) then begin
  print,' ##### sst_mwfits2fitscube abort: no SSTRED timing in header_in'
  STOP
endif  
taistart=anytim2tai(datetimestart)
timearr_sst=+indgen(nrtimes)*cadence_sst+taistart  ; TAI array
taiend=timearr_sst[nrtimes-1]                      ; start of last F-P scan
  ; check
if (abs(anytim2tai(datetimeend)-taiend) gt cadence_sst/2) then begin
  print,' ##### sst_mwfits2fitscube: SSTRED endtime conflict'
  STOP
endif

; get file path and extension for outfile_tai.dat and write that  
outonly=cgrootname(outfile,director=outpath,extension=outext) ;RR coyotelib
ssttimetai=outpath+outonly+'_tai.dat'
writecol,ssttimetai,timearr_sst,fmt='(5E21.12)'

; define input assoc frame size 
if (bitpix_in eq 8) then inarr=bytarr(nx_in,ny_in)
if (bitpix_in eq 16) then inarr=intarr(nx_in,ny_in)
if (bitpix_in eq -32) then inarr=fltarr(nx_in,ny_in)

; define output assoc frame size 
if (xrange[1] eq -1) then xrange[1]=nx_in-1
if (yrange[1] eq -1) then yrange[1]=ny_in-1
if (trange[1] eq -1) then trange[1]=nt_in-1
nx_out=xrange[1]-xrange[0]+1
ny_out=yrange[1]-yrange[0]+1
nt_out=trange[1]-trange[0]+1

; open infile for assoc
get_lun, unit_in
openr,unit_in,infile,swap_endian=(endian ne endian_in)
inassoc=assoc(unit_in,inarr,header_insize)

; output header (either integer or byte, no float!)
header_out=header_in
endian_out='b'

; modify dimensions (sxaddpar can add but also modify)
sxaddpar,header_out,'naxis',3  ; my "fitscube" = 3D
sxaddpar,header_out,'naxis1',nx_out
sxaddpar,header_out,'naxis2',ny_out
sxaddpar,header_out,'naxis3',nt_out,'RR nr times'
sxdelpar,header_out,'naxis4'   ; take out 
sxdelpar,header_out,'naxis5'   ; take out
sxaddpar,header_out,'nrchan_in',1,'RR nr channels'
sxaddpar,header_out,'nrwavs',1,'RR nr wavelengths or nr intervals'
sxaddpar,header_out,'starttim',datetimestart,'RR start date-time UT'
sxaddpar,header_out,'cadence',cadence_sst,'RR cadence in s'

; modify bitpix = word type = input type or integer
bitpix_out=bitpix_in
if (intfile eq 1) then bitpix_out=16
sxaddpar,header_out,'bitpix',bitpix_out

; size header_out
sizeheader_out=size(header_out)  
; fits header = Nx36 "card images" = Nx2880 bytes
header_outsize=(1+fix(sizeheader_out[1]/36.))*2880

; open outfile for assoc
get_lun, unit_out
openw,unit_out,outfile,swap_endian=(endian ne endian_out)
if (bitpix_out eq -32) then outassoc=assoc(unit_out,fltarr(nx_out,ny_out),$
                                           header_outsize)
if (bitpix_out eq 16) then outassoc=assoc(unit_out,intarr(nx_out,ny_out),$
                                          header_outsize)
if (bitpix_out eq 8) then outassoc=assoc(unit_out,bytarr(nx_out,ny_out),$
                                         header_outsize)
if (header_outsize ne 0) then begin
  rec=assoc(unit_out, bytarr(header_outsize))
  rec[0]=byte(header_out)
endif

; for intfile get minmax input file
if (intfile eq 1) then begin
  mima=minmaxcubefile(infile,xrange=xrange,yrange=yrange,trange=trange)
  absmax=max([abs(mima[0]),abs(mima[1])])  ; maintain zero as zero
  fileintscale=32000./absmax
endif

; loop over images
; ----------------

for it=trange[0],trange[1] do begin 
  if (wavdim eq 1) then begin
    iz=it*nw_in*ns_in+istokes*nw_in+iwav[0]
    image=inassoc[iz]
    if (product(finite(image)) eq 0) then begin
      print,' ##### sst_mwfits2fitscube abort: image has NaN'
      print,'       try sst_reform5d.pro on input file'
      return
    endif
    if (intfile eq 1) then image=fix(image*fileintscale)
  endif
  if (wavdim eq 2) then begin
    ; sum range of wavs divided by number of wavs  
    iz=it*nw_in*ns_in+istokes*nw_in+iwav[0]
    image=inassoc[iz]
    if (finite(image) eq 0) then begin
      print,' ##### sst_mwfits2fitscube abort: NaN detected'
      print,'       try sst_reform5d.pro on input file'
      return
    endif
    for iw=iwav[0]+1,iwav[1] do begin
      iz=it*nw_in*ns_in+istokes*nw_in+iw
      nextimage=inassoc[iz]
      image=image+nextimage
    endfor
    image=image/(iwav[1]-iwav[0]+1)
    if (intfile eq 1) then image=fix(image*fileintscale)
  endif
  outassoc[it-trange[0]]=image[xrange[0]:xrange[1],yrange[0]:yrange[1]]
endfor

; rescale output cube when intcube=1
if (intcube eq 1) then begin
  reformcubefile,outfile,'/tmp/rescalecubefile.fits',/intrescale
  spawn,'mv /tmp/rescalecubefile.fits '+outfile
endif

; done
free_lun,unit_in,unit_out

end

; =============== test per IDLWAVE S-c ==============================

;; cd,'/home/rutten/data/SST/2019-08-03_kuridze'
;; ; single-wav wide-band file
;; infile='sst/wb_4846_2019-08-03T_scans_0-60_corrected_im.fits'
;; outfile='sst/wb_4846_cube.fits'
;; sst_mwfits2fitscube,infile,outfile,0,0
;; showex,'sst/sst_wbcube.fits',outfile ; OK = identical (first = reform)
;; ; selected wav narrow-band file
;; infile='nb_4896_rot_ip.fits' ; nw_in=25 ns_in=1  nt_in=61
;; outfile='/tmp/hbeta.fits'

; RR 3D file
cd,'/media/rutten/RRHOME/alldata/SST/2020-08-13-AR-5D/sst/crisp'
;; infile='nb_6173_reformed.fits'
;; infile='nb_6173_2020-08-13T07:58:01_scans=0-82_stokes_corrected_im.fits'
;; outfile='feiwav5.fits'

;; cd,'/media/rutten/RRHOME/alldata/SST/2019-08-03_kuridze'
;; infile='nb_4896_rot_ip.fits' ; nt_mw=61  
;; outfile='trialmwfits2fitscube.fits'
;; iwav=12

cd,'/media/rutten/RRHOME/alldata/SST/2020-08-13-AR-5D/sst/crisp'
;; infile='nb_8542_2020-08-13T07:58:01_scans=0-82_stokes_corrected_im.fits'
infile='nb_8542_sstrr.fits'
outfile='redwing_18_8542.fits'
iwav=18
istokes=0

; run the program and show result
sst_mwfits2fitscube,infile,outfile,iwav=iwav,istokes=istokes,/verbose
showex,outfile

end
