; file: sst_getturret.pro = get numbers from SST turret log
; init: Apr 11 2017  Rob Rutten  Deil
; last: Jan  4 2019  Rob Rutten  Deil

;+
pro sst_getturret,datetime,sst_x,sst_y,sst_angle,summertime=summertime,$
  oldfile=oldfile,tablecor=tablecor,quiet=quiet

 ; NAME: 
 ;   sdo_getturret
 ;   
 ; PURPOSE: 
 ;   distill SST pointing coordinates out of SST turret log on web
 ;   
 ; CALL:
 ;   see above and example at the end of this file
 ;   
 ; INPUTS:
 ;   datetime = string, e.g., '2013.05.05_09:00', UT, this format mandatory   
 ;   
 ; OPTIONAL KEYWORD INPUTS:
 ;  summertime=1/0: set to 1 for La Palma summertime (default 0)
 ;  oldfile=1/0: use existing turret.dat file (default 0 = get new)
 ;  tablecor=tablecor: some fixed quantity (I don't know) (default 48.)
 ;  quiet=1/0: do not print results if 1
 ; 
 ; OUTPUTS:
 ;   sst_x: solar X center of field of view 
 ;   sst_y: solar Y center of field of view
 ;   sst_angle: field of view rotation, degrees, clockwise from solar North
 ;   
 ; RESTRICTIONS:
 ;   more precise in more recent years (thanks to SST PIG guider)
 ;   older log files giving Stonyhursts seem to have La Palma time

 ; HISTORY:
 ;   Apr 12 2017 RR: start   
;-

; answer no-parameter query 
if (n_params(0) lt 4) then begin
  print,'sdo_getturret,datetime,sst_x,sst_y,sst_angle,'
  print,'  [summertime=summertime,oldfile=oldfile,tablecor=tablecor,'
  print,'   quiet=quiet]'
  return 
endif

; defaults for keywords
if (n_elements(summertime) eq 0) then summertime=0
if (n_elements(oldfile) eq 0) then oldfile=0
if (n_elements(tablecor) eq 0) then tablecor=48.
if (n_elements(quiet) eq 0) then quiet=0

; check format of datetime
if (strmid(datetime,4,1) ne '.' or $
    strmid(datetime,7,1) ne '.' or $
    strmid(datetime,10,1) ne '_' or $
    strmid(datetime,13,1) ne ':') then begin
  print, ' ##### sst_getturret abort: datetime not YYYY.MM.DD_HH:MM'
  retall
endif

; convert datestring to SST turret log notation
aa=strsplit(datetime,'_',/extract)
bb=strsplit(aa[0],'.',/extract)
yyyy=bb[0]
momo=bb[1]
dd=bb[2]
cc=strsplit(aa[1],':',/extract)
hh=cc[0]
mimi=cc[1]
; check
;; print,yyyy+momo+dd+hh+mimi

; subtract summertime hour for La Palma time > UT
; NB:  not clear whether logs were ever in La Palma time rather than UT 
if (summertime ne 0) then begin
  hhval=fix(hh)
  hhval=hhval-1
  hh=string(hhval,format='(I2)')
  if (hh lt 10) then hh='0'+trim(hh)  
  ;; print,' ------ summertime hh = ',hh
endif

; get a full page from the log (inspect: ew /tmp/turret.dat)
if (oldfile ne 1) then begin
;   http://www.royac.iac.es/Logfiles/turret/2013/positionLog_2013.05.05
  urlturret='http://www.royac.iac.es/Logfiles/turret/'+yyyy+$
    '/positionLog_'+yyyy+'.'+momo+'.'+dd
  spawn,'curl -s '+urlturret+' -o /tmp/turret.dat'+$ 
    ' && echo "1" > /tmp/curlsuccess || echo "0"  > /tmp/curlsuccess'
  readcol,'/tmp/curlsuccess',success,format='I',/silent 
  if (success ne 1) then begin
    print,' ##### sst_getturret abort: no log file found'
    return
  endif
endif

; ==== read file turret.dat
; example '2013.05.05_09:20
; starts with blocks with RA and Decl instead of Stonyhurst
; format changed from Stonyhurst to (x,y) between 2013 and 2016
; seems a block with separate comment header is a specific run
; readcol skips all lines with other format: comment lines and the RA lines
readcol,'/tmp/turret.dat',$
  yr,mon,day,hr,min,sec,az,el,name1,val1,name2,val2,parang,tilt,azth,elth,$
  delimiter='/: ',$
  format='I,I,I, I,I,I, F,F, A,F,A,F, F,F,F,F'

; get the line for the desired time
nlines=n_elements(yr)
for iline=0,nlines-1 do begin
  if (yr[iline] eq fix(yyyy) and $
      mon[iline] eq fix(momo) and $
      day[iline] eq fix(dd) and $
      hr[iline] eq fix(hh) and $
      min[iline] eq fix(mimi)) then break else i0=iline+1
endfor

if (i0 ge nlines-1) then begin
  print,' ##### sst_getturret abort: no log entry at this datetime'
  return
endif

; check
;; print,' --------- iline = ',iline
;; print,yr[i0],mon[i0],day[i0],hr[i0],min[i0],sec[i0],az[i0],el[i0],' ',name1[i0],val1[i0],' ',name2[i0],val2[i0],parang[i0],tilt[i0],azth[i0],elth[i0]

; get sst_angle = az-el-tc-tilt (recipe from Luc)
sst_angle=azth[i0]-elth[i0]-tablecor-tilt[i0]

; ===== get X and Y

if (name1[i0] eq 'X' or name1[i0] eq 'x') then begin
; newer logs have (X,Y), from the Schupmann PIG I gather
  sst_x=val1[i0]
  sst_y=val2[i0]
endif else begin
; older logs have Stonyhurst entries
  if (name1[i0] ne 'S' and name1[i0] ne 'N') then begin
    print,' ##### sst_getturret abort: log has no/wrong Stonyhursts?'
    return
  endif
  if (quiet eq 0) then print,' ----- logfile has Stonyhursts'
  b=val1[i0]
  if (name1[i0] eq 'S') then b=-b
  l=val2[i0]
  if (name2[i0] eq 'E') then l=-l

; below inspired by Markus Aschwanden's coord_cart_helio.pro for STEREO
; follows Thomson WCS paper eqs (11), see eg
;   http://www.mssl.ucl.ac.uk/grid/iau/extra/local_copy/coordinates.pdf
; get ephemeris values
  timedate=hh+':'+mimi+' '+yyyy+'/'+momo+'/'+dd
  ;; print,timedate
  eph=get_sun(timedate)
  r0=eph(1)   ; semidiameter of disk (arcsec)
  l0=eph(10)  ; Carrington longitude at center of disk (deg)
  b0=eph(11)  ; Carrington latitude at center of disk (deg)
  b0=-b0      ; Stonyhurst latitude of the observer = center of disk 
  l0=0        ; Stonyhurst longitude of the observer = center of disk
  ;; print,r0,b0
  ;; print,b,l
  raddeg=!pi/180.
  brad=b*raddeg
  lrad=l*raddeg
  b0rad=b0*raddeg
  l0rad=l0*raddeg
  sst_x=r0*cos(brad)*sin(lrad-l0rad)
  sst_y=r0*(sin(brad)*cos(b0rad)+cos(lrad)*cos(lrad-l0rad)*sin(b0rad))
endelse

if (quiet eq 0) then print,' ----- '+datetime+$
  ':  X ='+trimd(sst_x,1)+'  Y ='+trimd(sst_y,1)+$
  '  angle_sst ='+trimd(sst_angle,1)

end

; =============== main per IDLWAVE H-c ======================

;; Koza first set; he gave X,Y=-510,+510 from sdo_featurelocator, worked well
; datetime='2016.05.13_12:00'  & summertime=0
;;  ----- 2016.05.13_12:00:  X = -499.78  Y = 507.64  angle_sst = 31.5937
;; fairly close

;; 2015-09-27-filament; X,Y= 766,-218; 
;; angle_sst= 125.369 -10.6+0.035-0.016 -0.03 -0.03
; datetime='2015.09.27_07:47' & summertime=0 ; you have to take real SST start
;;  ----- 2015.09.27_07:47:  X = 752.04  Y = -196.84  angle_sst = 125.428
;; fairly close

; 2014-06-21-quiet
;; sdo_getdata_rr,'2014.06.21_07:00',150,-215,206  ;; angle_sst= 62.4410
;; SST start 08:02:31
; datetime='2014.06.21_08:03' & summertime=0
;;  ----- 2014.06.21_08:03:  X = -197.89934  Y = 208.50539  angle_sst = 64.699;;  OK, X difference is hour rotation; no summertime then?  Still Stonyhursts

;; 2013-05-05-krishna  that I had so much difficulty finding
;; sdo_getdata_rr,'2013.05.05_09:00',40,0.0,31.5 ; from Vasco; angle=46
;; SST start 09:06:15
;; datetime='2013.05.05_09:07' 
;; result:   N 12.39  E 52.74 
;; result: -739.60426       226.32196      50.1320
;; so Vasco was fully wrong? the Stonyhurst already say so
; datetime='2013.05.05_09:20'  
;;   -739.41928       226.81181      48.8550
; datetime='2013.05.05_09:20'  & summertime=1
;;   3.9758109      -3.9841680      55.4860  ; closer to Vasco but indeed off

;; 2014-06-19-limb 09:56:38 Luc set that I had so much difficulty finding
;; sdo_getdata_rr,'2014.06.19_09:55',50,15,925 
; datetime='2014.06.19_09:57'
;; file:  N  1.47  E  4.91   ; OOPS??
;; results -80.807069     -0.56950488      55.1390  ; OOPS?? fits Stonyhurst
;; but not at all at limb....  Just before the SST WAS at North limb
;; datetime='2014.06.19_09:50'
;; results  1.9516616       905.91097      55.7820 ; that's better
;?? was SST time not UT but local La Palma time?  straight from computer?
; datetime='2014.06.19_09:57'  & summertime=1
; ----- 2014.06.19_09:57:  X = 14.56272  Y = 889.55688  angle_sst = 60.136
;; that looks a lot better! indeed Luc's Y was far off giving trouble

datetime='2008.06.15_09:15'  ; from RBE paper

sst_getturret,datetime,sst_x,sst_y,sst_angle,summertime=summertime

end
