; file: sta_stb_align.pro = align SDO target cubes and STX image sequences
; init: Jan 13 2020  Rob Rutten  Deil
; last: Jan 13 2020  Rob Rutten  Deil
; note: @?? yet todo's 

;+

pro sta_stb_align,$
; ----- input (values must be specified)
  px_sa,angle_sta,px_stb,angle_stb$
; ----- mandatory parameters for passing (values need not be specified)
cadence_stx,px2asym,driftshifts,wobbleshifts,shiftfor,shiftrev,$
  nxfor,nyfor,nxrev,nyrev,euvcor,$
; ----- keywords for findalignimages.pro
  obsdate=obsdate,staalignfile=staalignfile,$
  stbalignfile=stbalignfile,stbdriftfile=stbdriftfile,$
  statimesfile=statimesfile,stbtimesfile=stbtimesfile,$
  itsample_sta=itsample_sta,$
  skipmetcalf=skipmetcalf,applypx2asym=applypx2asym,$
  inshiftab=inshiftab,trimboxstb=trimboxstb,$
  smearsta=smearsta,smearstb=smearstb,$
  histopsta=histopsta,histopstb=histopstb,$
  flatten=flatten,$
  muckdarksta=muckdarksta,muckdarkstb=muckdarkstb,$
  muckbrightsta=muckbrightsta,muckbrightstb=muckbrightstb,$
; ----- keywords for sta_stb_findalignfiles.pro
  driftspline=driftspline,wobblespline=wobblespline,aligndir=aligndir,$
; ----- keywords setting output choices
  splinip=splinip,$ 
  nxscale=nxscale,nyscale=nyscale,$
  do_findalign=do_findalign,$
  sta_sellist=sta_sellist,stb_sellist=stb_sellist,$
; ----- keywords defining dirnames
  stxname=stxname,stb2stadir=stb2stadir,driftsdir=driftsdir,$
; ----- keywords to choose response
  blink=blink,show=show,verbose=verbose

 ; NAME:
 ;   sta_stb_align with STX generic for "Solar Telescope X",
 ;   e.g., different SST/CRISP wavelengths, SST/CHROMIS, etc.
 ;
 ; PURPOSE:
 ;   align STB image cubefile to STA image cubefile
 ;
 ; DESCRIPTION:
 ;   wrapper with main tasks:
 ;     findalignimages: find shifts plus angle and px corrections
 ;       from a sample pair for the moment with best STX seeing 
 ;     sta_stb_findalignfiles: determine STX pointing drift versus SDO
 ;     timings: @?? 
 ;
 ; TUNING SUGGESTIONS:
 ;  - apply appropriate image mucking for sta_stb_findalignimages
 ;  - choose to set applypx2asym 
 ;
 ; CALL:
 ;   see above 
 ;
 ; EXAMPLE:
 ;   at the end of this file
 ;
 ; INPUT PARAMETERS: @??
 ;   px_stx = pixel size STX data in arcsec
 ;   angle_stx = STX data rotation from solar (X,Y) in degrees NESW (CCW)
 ;     determine these first with stx_findlocation.pro
 ;     px_stx and angle_stx get improved if skipmetcalf=0 (but slow)
 ; ----- keywords for parameter passing, values need not be sepecified   
 ;   cadence_stx = mean interval in seconds between STX images 
 ;   px2asym = px_stx asymmetry (x,y) (may be input for skipmetcalf=1)
 ;   driftshifts,wobbleshifts,shiftfor,shiftrev,
 ;     nxfor,nyfor,nxrev,nyrev,euvcor = alignment outputs
 ; ----- keywords for sta_stb_findalignimages.pro
 ;   obsdate = observing date, mandatory format '2016.09.05'
 ;   sdoalignfile = string name SDO fitscube file for alignment
 ;     (default 'sdo/target/cubes/hmicont.fits')
 ;   stxalignfile = string name STX file for Metcalf alignment
 ;     SST/CRISP: default stxname+'/ha_wb.fits' but may also be La Palma
 ;       cubefile as 'wb.6563.hh:mm:ss.corrected.aligned.icube'
 ;     DST/IBIS: single-wav fitscube file with reddest Halpha wavelength
 ;   stxdriftfile: string name STX file for guiding alignment
 ;     SST/CRISP: default = stxalignfile since MOMFBD makes wb-CRISP coherent
 ;     DST:CRISP: despeckled white-band file (may be angled to IBIS) 
 ;   stxtimesfile = string name STX timings file (save file for SST)
 ;   itsample_stx = best it (sharpest STX moment; default -1 = auto-find)
 ;   skipmetcalf = 1/0: determine shift only, no scale + rotate correction
 ;   applypx2asym = 1/0: apply Metcalf-found px2 asymmetry within iteration
 ;   inshift12: start-off shift of SDO im in SDO pixels from SDO center to 
 ;     where SDO shows the feature that is located at the center of STX; 
 ;     measure with: showex,im1,/markcenter,/noblock & showex,im2,/markcenter
 ;     (such specification is needed when this offset gets above 10-20 arcsec)
 ;     The program returns an improved value and prints this when verbose=1.
 ;   trimboxstx = 4-elem int vector [xmin,ymin,xmax,ymax] to select subfield
 ;     will be used only in the alignment determination, for example a
 ;     better look-alike part.  However, since December 2017 these are
 ;     not needed to strip monochrome borders or derotation triangles in 
 ;     SST images: these are stripped within the image mucking for alignment.
 ;   smearstx: muck by smearing STX image (default -1 = 1.5*pxratio)
 ;   histopstx: hist_opt STX muckimage at given value (< 1, top only)
 ;   flatten: subtract boxcar average from each, value = width in coarsest px
 ;     default flatten=5 (good for spots or limb in the field)
 ;   muckdark: set intensities below threshold (units mean=1) to mean
 ;   muckbright: set intensities above threshold (units mean=1) to mean
 ; ----- keywords for sta_stb_findalignfiles.pro
 ;   driftspline = smooth value (default 1.E4; -1 = no spline, 1.E8 = linear)
 ;   wobblespline = idem, lower, to follow SDO wobble (default 0.1)
 ;   aligndir = string dir to put results in (default 'sta_stbname_align')
 ;   check304ha = get alignment correction for aia304 versus STX Ha_lc
 ;   splineha = smooth value for Ha_lc (default 1.E3*driftspline)
 ;   halcfile = string STX Ha-linecenter fitscube (default 'stxname/ha_lc')
 ;   aia304file =  string with path/name of AIA 304 fitscube file
 ;     (default 'sdo/target/cubes/aia304.fits')
 ; ----- keywords: output choices
 ;   splinip = 0: nearest-neighbour = retain pixels (e.g., coarse SDO) 
 ;           = 1: cubic spline interpolation 
 ;   rotate_up = 1/0: rotate reverse and full to "up=up" (default 0)
 ;   nxscale,nyscale: increase/decrease fields stx2sdo, stx2rotsdo (def 1.0)
 ;   undo304_1600 = 1/0: undo the EUV anchoring to measured 304 > 1600 shifts
 ;   makecrispexsdocube = 1/0: add multiwav SDO La Palma file for CRISPEX
 ;   do_findalign = 1/0: do the alignment
 ;   do_fulldisk = add full-disk SDO images showing STX field
 ;   do_sdo2stx = 1/0: put STX-cropped STX-rotated STX-timing SDO on STX 
 ;   do_stx2sdo = 1/0: put back-rotated STX on STX-cropped SDO-timing SDO 
 ;   do_stx2rotsdo = 1/0: put STX on STX-cropped STX-rotated STX-regtimed SDO
 ;   do_stx2fullsdo = 1/0: put STX within full SDO-timing SDO
 ;   stx_sel_xx = ['file1',...,'fileXX'] STX files to write/add in output xx
 ;      (format: fits and/or La Palma, [nx,ny,nt] and/or [nx,ny,nt+nwav]
 ;       puts '_for_xx' files in sdo2stxdir if applypx2asym is set
 ;       puts '_rev_xx' files in stx2sdodir
 ;       puts '_rot_xx' files in stx2rotsdodir
 ;       puts '_full_xx' files in stx2fullsdodir
 ;         with _xx = _ip for interpolated or _px for pixelated files
 ;   sdo_sel_xx = SDO files to write/add in output xx
 ;      default = 'all', or select, e.g., ['cont','mag','1600','304','171']
 ; ----- keywords defining dirnames
 ;   stxname = string telescope or instrument name for dirs, default 'sst/'
 ;   sdocubesdir = string SDO cubefiles dir (default sdo/target/cubes)
 ;   fulldiskdir = string SDO full-disk images dir (default fulldisk)
 ;   sdo2stxdir = string forward output dir (default 'sdo2'+stxname)
 ;   stx2sdodir = string reverse output dir (default stxname+'2sdo[up]')
 ;   stx2rotsdodir = string reverse rot-SDO output dir (...+'2rotsdo') 
 ;   stx2fullsdodir = string reverse full-SDO output dir (...+'2fullsdo[up]') 
 ;   driftsdir = dir with SDO alignment files (default 'sdo/driftscenter/')
 ; ----- keywords to choose amount of response
 ;   blink > 0: flickrr intermediate results during blink seconds (default 0)
 ;   show = 1/0: showex final results for stxalignfile and sdoalignfile
 ;   verbose = 1/0: more/less printout in terminal
 ; 
 ; OUTPUTS:
 ;   alignment results in sta_stb_align (data file and driftcuvre graph)
 ;   do_fuldisk: full-disk SDO images at it sharpest STX in fulldiskdir
 ;   do_sdo2stx: aligned fitscube files in sdo2stxdir
 ;   do_stx2sdo: aligned fitscube files in stx2sdodir
 ;   do_stx2rotsdo: aligned fitscube files in stx2rotsdodir
 ;   do_stx2fullsdo: aligned fitscube files in stx2fullsdodir
 ; 
 ; HISTORY:
 ;   Jun  2 2017 RR: previous sdo_sst_align.pro from sdo2sst_v1.pro
 ;   Dec 31 2017 RR: start as generalization of sdo_sst_align.pro
 ;   Jan  4 2018 RR: option applypx2asym 
 ;   Jan  9 2018 RR: option cadence_reverse (out Dec 19 2018)
 ;   Dec 20 2018 RR: changed sdo2stx, added stx2rotsdo, stx2fullsdo
 ;   Dec 28 2018 RR: options rotate_up, do_fulldisk
;-

; answer no-parameter query 
if (n_params() lt 13) then begin
  print,' sta_stb_align,px_stx,angle_stx,cadence_stx,px2asym,'
  print,'   driftshifts,wobbleshifts,shiftfor,shiftrev,'
  print,'   nxfor,nyfor,nxrev,nyrev,euvcor,'
  print,'      plus many keywords, see header'
  sp,sta_stb_align
  return
endif

; set wall-clock timer (seconds)
timestart=systime(1) 

; print pacifier
print,' ===== sta_stb_align starts'

; keyword defaults
; ----- parameters (fake angle and cadence values)
if (n_elements(px_stx) eq 0) then px_stx=0.058
if (n_elements(angle_stx) eq 0) then angle_stx=90.
if (n_elements(cadence_stx) eq 0) then cadence_stx=10.
if (n_elements(px2asym) eq 0) then px2asym=[1.0,1.0]
; ----- main keywords
if (n_elements(stxname) eq 0) then stxname='sst'
if (n_elements(do_fulldisk) eq 0) then do_fulldisk=0
if (n_elements(do_findalign) eq 0) then do_findalign=0
if (n_elements(do_sdo2stx) eq 0) then do_sdo2stx=0
if (n_elements(do_stx2sdo) eq 0) then do_stx2sdo=0
if (n_elements(do_stx2rotsdo) eq 0) then do_stx2rotsdo=0
if (n_elements(do_stx2fullsdo) eq 0) then do_stx2fullsdo=0
if (n_elements(stx_sel_sdo2stx) eq 0) then stx_sel_sdo2stx=''
if (n_elements(stx_sel_stx2sdo) eq 0) then stx_sel_stx2sdo=''
if (n_elements(stx_sel_stx2rotsdo) eq 0) then stx_sel_stx2rotsdo=''
if (n_elements(stx_sel_stx2fullsdo) eq 0) then stx_sel_stx2fullsdo=''
if (n_elements(sdo_sel_fulldisk) eq 0) then sdo_sel_fulldisk='all'
if (n_elements(sdo_sel_sdo2stx) eq 0) then sdo_sel_sdo2stx='all'
if (n_elements(sdo_sel_stx2sdo) eq 0) then sdo_sel_stx2sdo='all'
if (n_elements(sdo_sel_stx2rotsdo) eq 0) then sdo_sel_stx2rotsdo='all'
if (n_elements(sdo_sel_stx2fullsdo) eq 0) then sdo_sel_stx2fullsdo='all'
if (n_elements(rotate_up) eq 0) then rotate_up=0
if (n_elements(fulldiskdir) eq 0) then fulldiskdir='fulldisk'
if (n_elements(sdo2stxdir) eq 0) then sdo2stxdir='sdo2'+stxname
if (n_elements(stx2sdodir) eq 0) then stx2sdodir=stxname+'2sdo'
if (n_elements(stx2rotsdodir) eq 0) then stx2rotsdodir=stxname+'2rotsdo'
if (n_elements(stx2fullsdodir) eq 0) then stx2fullsdodir=stxname+'2fullsdo'
if (n_elements(undo304_1600) eq 0) then undo304_1600=0
if (n_elements(driftsdir) eq 0) then driftsdir='sdo/driftscenter'
if (n_elements(splinip) eq 0) then splinip=0
if (n_elements(makecrispexsdocube) eq 0) then makecrispexsdocube=0
if (n_elements(nxscale) eq 0) then nxscale=1.0
if (n_elements(nyscale) eq 0) then nyscale=1.0 
if (n_elements(blink) eq 0) then blink=10
if (n_elements(show) eq 0) then show=0
if (n_elements(verbose) eq 0) then verbose=0
 ; ----- keywords for sta_stb_findalignimages.pro
if (n_elements(obsdate) eq 0) then obsdate=''
if (n_elements(sdoalignfile) eq 0) then $
  sdoalignfile='sdo/target/cubes/hmicont.fits'
if (n_elements(sdocubesdir) eq 0) then sdocubesdir=getdirpath(sdoalignfile)
if (n_elements(stxalignfile) eq 0) then stxalignfile=stxname+'/ha_wb.fits'
if (n_elements(stxdriftfile) eq 0) then stxdriftfile=stxalignfile
if (n_elements(stxtimesfile) eq 0) then stxtimesfile=''
if (n_elements(itsample_stx) eq 0) then itsample_stx=-1 ; NB default
if (n_elements(skipmetcalf) eq 0) then skipmetcalf=0
if (n_elements(applypx2asym) eq 0) then applypx2asym=0
if (n_elements(inshift12) eq 0) then inshift12=[0.0,0.0]
if (n_elements(trimboxstx) eq 0) then trimboxstx=-1 ; NB default
if (n_elements(smearstx) eq 0) then smearstx=-1.5   ; NB default
if (n_elements(histopstx) eq 0) then histopstx=0
if (n_elements(flatten) eq 0) then flatten=5 ; NB default
if (n_elements(muckdarksdo) eq 0) then muckdarksdo=0 
if (n_elements(muckdarkstx) eq 0) then muckdarkstx=0 
if (n_elements(muckbrightsdo) eq 0) then muckbrightsdo=0 
if (n_elements(muckbrightstx) eq 0) then muckbrightstx=0 
; ----- extra keywords for sta_stb_findalignfiles.pro
if (n_elements(driftspline) eq 0) then driftspline=1.E3
if (n_elements(wobblespline) eq 0) then wobblespline=0.1
if (n_elements(aligndir) eq 0) then aligndir='sdo_'+stxname+'_align/'
if (n_elements(check304ha) eq 0) then check304ha=0
if (n_elements(halcfile) eq 0) then halcfile='stx/ha_lc.fits'
if (n_elements(aia304file) eq 0) then $
  aia304file='sdo/target/cubes/aia304.fits'
if (n_elements(splineha) eq 0) then splineha=driftspline*1.E3

; check tasks
if (do_findalign+do_fulldisk+do_sdo2stx+do_stx2sdo+$
    do_stx2rotsdo+do_stx2fullsdo eq 0) then begin
  print,' ##### sta_stb_align abort: all do_ options zero = nothing to do'
  return
endif

; check undo304_1600 versus check304ha
if (check304ha ne 0 and undo304_1600 ne 0) then begin
  print,' ----- warning: check304ha is set, therefore undo304_1600 unset'
  print,'       but undo304_1600 returns as optional alternative for ha304'
  undo304_1600=0
endif

; add '_up' to corresponding dirnames when rotate_up
if (rotate_up) then begin
  if (not(strmatch(fulldiskdir,'*_up*'))) then fulldiskdir=fulldiskdir+'_up'
  if (not(strmatch(stx2sdodir,'*_up*'))) then stx2sdodir=stx2sdodir+'_up'
  if (not(strmatch(stx2fullsdodir,'*_up*'))) then $
    stx2fullsdodir=stx2fullsdodir+'_up'
endif 

; make dirs if not yet existing
if (do_fulldisk) then spawn,'mkdir -p '+fulldiskdir
if (do_sdo2stx) then spawn,'mkdir -p '+sdo2stxdir
if (do_stx2sdo) then spawn,'mkdir -p '+stx2sdodir
if (do_stx2rotsdo) then spawn,'mkdir -p '+stx2rotsdodir
if (do_stx2fullsdo) then spawn,'mkdir -p '+stx2fullsdodir

; do_findalign is set: find alignment parameters
if (do_findalign ne 0) then begin

  ; delete older drift figures
  spawn,'rm -f '+aligndir+'/*muck*--*muck*.ps'
  
  sta_stb_findalignfiles,$ 
; ----- parameters sta_stb_findalignimages
    obsdate,sdoalignfile,stxalignfile,stxtimesfile,$
    px_stx,angle_stx,$
    cadence_stx,px2asym,driftshifts,wobbleshifts,shiftfor,shiftrev,$
    nxfor,nyfor,nxrev,nyrev,euvcor,$
    itsample_stx=itsample_stx,stxdriftfile=stxdriftfile,$
    skipmetcalf=skipmetcalf,applypx2asym=applypx2asym,$
    inshift12=inshift12,trimboxstx=trimboxstx,$
    smearstx=smearstx,histopstx=histopstx,flatten=flatten,$
    muckdarksdo=muckdarksdo,muckdarkstx=muckdarkstx,$
    muckbrightsdo=muckbrightsdo,muckbrightstx=muckbrightstx,$
; ----- extra keyword inputs of this program
    stxname=stxname,$
    driftspline=driftspline,wobblespline=wobblespline,$
    aligndir=aligndir,splinip=splinip,$
    check304ha=check304ha,splineha=splineha,$
    halcfile=halcfile,aia304file=aia304file,$ 
    undo304_1600=undo304_1600,$
    blink=blink,show=show,verbose=verbose

; stop here if do_findalign was all to do 
  if (do_fulldisk+do_sdo2stx+do_stx2sdo+do_stx2rotsdo+do_stx2fullsdo eq 0) $
  then return

; get timearr_stx
  xeuvcor=reform(euvcor[*,0])
  yeuvcor=reform(euvcor[*,1])
  xshiftfor=reform(shiftfor[*,0])
  yshiftfor=reform(shiftfor[*,1])
  dummy=cgrootname(stxtimesfile,extension=timefileext)  ;RR needs coyotelib
; SST formats for timing file
  if (timefileext eq 'sav' or timefileext eq 'idlsave') then begin
    restore,stxtimesfile
    if (n_elements(time) ne 0) then datetime_stx=obsdate+'_'+time else $
      if (n_elements(csday) ne 0) then begin
      tai_stx=anytim2tai(obsdate)+double(csday)
      datetime_stx=anytim2utc(tai_stx,/ccsds)
    endif else begin
      print,' ##### sta_stb_findalignfiles abort: no known timing format'
      retall
    endelse
; non SST, eg DST/IBIS format timing file
  endif else readcol,stxtimesfile,datetime_stx,format='(A)'
  timearr_stx=anytim2tai(datetime_stx)
  nt_stx=n_elements(timearr_stx)
  cadence_stx=(timearr_stx[nt_stx-1]-timearr_stx[0])/(nt_stx-1)
endif

; do_findalign not set: read earlier alignment determinations from file
alignfile=aligndir+'sdo-'+stxname+'-align.dat'
if (do_findalign eq 0) then begin
  if (file_test(alignfile) eq 0) then begin
    print,' ##### sta_stb_align abort: do_findalign=0 but no alignfile found'
    return
  endif
  print,' ----- no new SD0-STX alignment but use '+alignfile
  openr,42,alignfile  ;RR 42 is answer to everything
  readf,42,angle_stx,px_stx,xasym,yasym
  readf,42,nt_stx,cadence_stx,nxfor,nyfor,nxrev,nyrev
  close,42
  px2asym[0]=xasym
  px2asym[1]=yasym
  readcol,alignfile[0],timearr_stx,xdrift,ydrift,xwobble,ywobble,$
    xshiftfor,yshiftfor,xeuvcor,yeuvcor,$
    skipline=5,format='D',/silent   ;RR double precision needed for timearr
  driftshifts=[[xdrift],[ydrift]]
  wobbleshifts=[[xwobble],[ywobble]]
endif  

; set shifts
forshifts=[[xshiftfor],[yshiftfor]]        ; apply to rotated SDO > STX 
rotshifts=-forshifts                       ; apply to STX > rotated SDO
revshifts=-rotatevec(forshifts,-angle_stx) ; apply to reverse STX > SDO

; read pointing parameters file sdo-stx-align.dat (written here or before)
openr,42,alignfile
readf,42,angle_stx,px_stx,xasym,yasym
readf,42,nt_stx,cadence_stx,nxfor,nyfor,nxrev,nyrev
readf,42,solx_stx,soly_stx,angle_limb,muobs
close,42

; modify revshifts when rotate_up
angle_rotate=0
if (rotate_up) then begin
  angle_rotate=angle_limb
  revshifts=-rotatevec(forshifts,-angle_stx+angle_rotate) 
endif 

; set congrid factors 
px_sdo=0.6  ; SDO (also HMI)
if (applypx2asym) then px_stx2d=px_stx*px2asym else px_stx2d=[px_stx,px_stx]
scale_sdo2stx=px_sdo/[px_stx,px_stx] ; plm 10, no asymmetry into SDO pixels
scale_stx2sdo=px_stx2d/px_sdo        ; plm 0.1, in STX>SDO apply asymmetry

; optional correction to undo the EUV slaving to measured 304 to 1600 shifts
; (not when check304ha set)
undoeuvrev=fltarr(nt_stx,2)
undoeuvfor=fltarr(nt_stx,2)
if (undo304_1600 ne 0) then begin
  wavpair=68   ; 304 > 1600
  for it=0,nt_stx-1 do begin
    undoeuvfor[it,*]=$
      sdo_getonesplineshift(timearr_stx[it],wavpair,driftsdir)
    undoeuvfor[it,*]=undoeuvfor[it,*]*scale_sdo2stx
    undoeuvrev[it,*]=-rotatevec(undoeuvfor[it,*],-angle_stx)
;RR ?? is the above okay with applypx2asym?
  endfor
endif

; define switch for combined EUV correction
euvcor=(max(abs(float([[xeuvcor],[yeuvcor]])-undoeuvfor)) gt 0.01) 

; if euvcor then correct and rewrite SDO target cubes 
if (euvcor) then begin
  alreadydone=findfile(sdocubesdir+'/*EUVcorrected*')
  if strmatch(alreadydone,'*EUVcorrected*') then begin
    print,' !!!!! euvcor requested but EUV files in '+sdocubesdir
    print,' !!!!! are already corrected; further correction skipped'
    print,' !!!!! for renewal uncorrected rerun sdo_maketargetcubes.pro'
  endif else begin
    print,' ----- correct EUV SDO cubefiles in '+sdocubesdir
  ; copy uncorrected target cubes to tmp as backup for now
    print,' ----- uncorrected originals temporarily in /tmp/sdotargetcubes'
    spawn,'mkdir -p /tmp/sdotargetcubes; cp -r '+sdocubesdir+'/*'+$
      ' /tmp/sdotargetcubes/.'
    spawn,'touch /tmp/sdotargetcubes/00-EUVnotcorrected'

    ; set EUV corshifts (still on STX pixel scale)
    euvforcors=float([[xeuvcor],[yeuvcor]])+undoeuvfor
    euvrevcors=-rotatevec(euvforcors,-angle_stx)

  ; apply EUV corshifts to sdo/target/cubes
    targetfiles=findfile(sdocubesdir+'/*.fits')
    nfiles=n_elements(targetfiles)
    spawn,'touch '+sdocubesdir+'/00-EUVcorrected'
    for ifile=0,nfiles-1 do begin
      infile=targetfiles[ifile]
      outfile='/tmp/sdoeuvcorrected.fits'
      if (strmatch(infile,'*aia131*.fits') or $
          strmatch(infile,'*aia171*.fits') or $
          strmatch(infile,'*aia193*.fits') or $
          strmatch(infile,'*aia211*.fits') or $
          strmatch(infile,'*aia304*.fits') or $
          strmatch(infile,'*aia335*.fits') or $
          strmatch(infile,'*aia94*.fits')) then begin
        reformcubefile,infile,outfile,shift=euvrevcors*scale_stx2sdo
        ;RR ?? sign correct? needs checking
        spawn,'mv '+outfile+' '+infile
      endif
    endfor
  endelse
endif

; define permitted SDO name strings for SDO request checks 
sdonames=['94','131','171','193','211','304','335',+$
          '1600','1700','mag','cont'] 

; ===== do_fulldisk = get SDO full-disk images at it_best with STX field box 

if (do_fulldisk ne 0) then begin
  print,' ----- do_fulldisk: get SDO full-disk images'
  sdowavs=['94','131','171','193','211','304','335','1600','1700',$  
           'magnetogram','continuum'] 
  if (sdo_sel_fulldisk[0] eq 'all') then sdo_sel_fulldisk=sdowavs
  nwavs=n_elements(sdo_sel_fulldisk)
  for iwav=0,nwavs-1 do begin
    match=0
    for iname=0,10 do $
      if (strmatch(sdo_sel_fulldisk[iwav],'*'+sdowavs[iname]+'*')) $
      then match=1
    if (match eq 1) then sta_stb_fulldisk,sdo_sel_fulldisk[iwav],stxname,$
      rotate_up=rotate_up,fulldiskdir=fulldiskdir,/markcenter,/marknorth $
    else print," !!!!! sta_stb_align warning:"+ $
      " SDO selection '"+sdo_sel_fulldisk[iwav]+"' empty or invalid"
  endfor
endif

; ========== sdo2stx production ("forward")

if (do_sdo2stx ne 0) then begin

  print,' ----- do_sdo2stx: write SDO _for files in '+sdo2stxdir

; set outtail acording to splinip setting
  if (splinip eq 0) then outtail='_for_px.fits' else outtail='_for_ip.fits'

; transform all or selected SDO fitscubes
  if (sdo_sel_sdo2stx[0] eq 'all') then $
    doallfiles,sdocubesdir,sdo2stxdir,'.fits',outtail,$
    'reformcubefile',$
    rotate=angle_stx,congridfactor=scale_sdo2stx,$
    shift=forshifts,$
    cutcentralx=nxfor,cutcentraly=nyfor,$
    timearr_out=timearr_stx,missingvalue=-1,splinip=splinip $
  else begin
    nselect=n_elements(sdo_sel_sdo2stx)
    for ifile=0,nselect-1 do begin
      match=0
      for iname=0,10 do $
        if (strmatch(sdo_sel_sdo2stx[ifile],'*'+sdonames[iname]+'*')) $
        then match=1
      if (match eq 1) then begin
        infile=findfile(sdocubesdir+'/*'+sdo_sel_sdo2stx[ifile]+'*.fits')
        outfile=sdo2stxdir+'/'+file_basename(infile,'.fits')+outtail
        reformcubefile,infile,outfile,$
          rotate=angle_stx,congridfactor=scale_sdo2stx,$
          shift=forshifts,$
          cutcentralx=nxfor,cutcentraly=nyfor,$
          timearr_out=timearr_stx,missingvalue=-1,splinip=splinip
      endif else print," !!!!! do_sdo2stx warning:"+ $
        " SDO selection '"+sdo_sel_sdo2stx[ifile]+"' empty or invalid"
    endfor
  endelse
  
; optionally produce SDO multi-channel crispex file in La Palma format
  if (makecrispexsdocube ne 0) then begin
    outfilestart='/sdo_crispex_for_px'
    if (splinip ne 0) then outfilestart='/sdo_crispex_for_ip'
    sdo_fitscubes2crispex,sdo2stxdir,outtail,sdo2stxdir+outfilestart,$
      /diagname,bytscale=2,/verbose
  endif

; when applypx2asym put dewarped STX files into sdo2stxdir
  if (applypx2asym ne 0) then begin
    nstxfiles=n_elements(stx_sel_sdo2stx)
    for ifile=0,nstxfiles-1 do begin
      infile=stx_sel_sdo2stx[ifile]
      if (verbose ne 0) then print,' ----- start dewarping file = '$
        +infile+' into '+sdo2stxdir+'  ## may take long'
      dummy=cgrootname(infile,extension=ext)  ;RR needs coyotelib
      stxforfile=file_basename(infile,'.'+ext)
      stxforfile=sdo2stxdir+'/'+stxforfile+'_for_ip.fits'
      reformcubefile,infile,stxforfile,scale=px2asym,splinip=1
      ;RR ?? or 1./px2asym?? Needs checking
    endfor
  endif

endif ; ----- end of forward sdo2stx SDO production

; skip the rest including non-STX timing if no more productions requested
if (do_stx2fullsdo+do_stx2sdo+do_stx2rotsdo eq 0) then goto, ENDPRODUCTION

; get SDO target size and timing
header_sdo=headfits_rr(sdoalignfile)
nx_sdo=fxpar(header_sdo,'naxis1')
ny_sdo=fxpar(header_sdo,'naxis2')
nt_sdo=fxpar(header_sdo,'naxis3') 
starttim_sdo=fxpar(header_sdo,'starttim')
cadence_sdo=fxpar(header_sdo,'cadence')
if (starttim_sdo eq 0 or cadence_sdo eq 0) then begin
  print, ' ##### sta_stb_align abort: SDO input file has no STARTTIM'+$
    'or CADENCE in header'
  return
endif
timearr_sdo=indgen(nt_sdo)*cadence_sdo+anytim2tai(starttim_sdo)


; ========== stx2fullsdo(up) production ("full")

; NB: ?? this currently does not undo the SDO wobble but it should

if (do_stx2fullsdo ne 0) then begin 

  print,' ----- do_stx2fullsdo starts'
  
; set outtail
  if (rotate_up) then outtail='_full_up' else outtail='_full'
  if (splinip eq 0) then outtail=outtail+'_px.fits' $
  else outtail=outtail+'_ip.fits'

; reverse-transform stx files, maintain extensions, dewarp when applypx2asym
  if (verbose ne 0) then print,' ----- convert stxfiles > '+stx2fullsdodir
  nstxfiles=n_elements(stx_sel_stx2fullsdo)
  for istxfp=0,nstxfiles-1 do begin
    infile=stx_sel_stx2fullsdo[istxfp]
    dummy=cgrootname(infile,extension=ext) ;RR needs coyotelib
    if (rotate_up) then $
      outfile=$
      stx2fullsdodir+'/'+file_basename(infile,'.'+ext)+'_full_up_ip.'+ext $
    else outfile=$
      stx2fullsdodir+'/'+file_basename(infile,'.'+ext)+'_full_ip.'+ext
    if (verbose ne 0) then print,' ----- start on '+outfile+$
      '  #### may take very long'
    reformcubefile,infile,outfile,$
      shift=revshifts*avg(scale_stx2sdo),$
      rotate=-angle_stx+angle_rotate,congridfactor=scale_stx2sdo,$
      nxlarge=nx_sdo,nylarge=ny_sdo,$
      timearr_in=timearr_stx,timearr_out=timearr_sdo,$
      greyborders=0,missingvalue=-1,splinip=1
;RR /greyborders gives images=0 in dark Halpha center at large embedding
  endfor
  
; rotate sdo files if rotate_up; otherwise use target originals 
  if (rotate_up) then begin
; transform all or selected SDO fitscubes
    if (sdo_sel_stx2fullsdo[0] eq 'all') then $
      doallfiles,sdocubesdir,stx2fullsdodir,'.fits',outtail,$
      'reformcubefile',$
      rotate=angle_rotate,$
      greyborders=0,missingvalue=-1,splinip=splinip $
    else begin
      nselect=n_elements(sdo_sel_stx2fullsdo)
      for ifile=0,nselect-1 do begin
        match=0
        for iname=0,10 do $
          if (strmatch(sdo_sel_stx2fullsdo[ifile],'*'+sdonames[iname]+'*')) $
          then match=1
        if (match eq 1) then begin
          infile=findfile(sdocubesdir+'*'+sdo_sel_stx2fullsdo[ifile]+'*.fits')
          outfile=stx2fullsdodir+'/'+file_basename(infile,'.fits')+outtail
          reformcubefile,infile,outfile,$
            rotate=angle_rotate,$        
            greyborders=0,missingvalue=-1,splinip=splinip
        endif else print," !!!!! do_stx2fullsdo warning:"+ $
          " SDO selection '"+sdo_sel_stx2fullsdo[ifile]+"' empty or invalid"
      endfor
    endelse
  endif
  
; optionally produce multi-channel SDO cubefile in La Palma crispex format
  if (makecrispexsdocube ne 0) then begin
    if (rotate_up) eq 0 then begin 
      filestart='/sdo_crispex_full_px'
      if (splinip ne 0) then filestart='/sdo_crispex_full_ip'
      sdo_fitscubes2crispex,sdocubesdir,outtail,stx2fullsdodir+filestart,$
        /diagname,bytscale=2,/verbose
    endif else begin
      filestart='/sdo_crispex_full_up_px'
      if (splinip ne 0) then filestart='/sdo_crispex_full_up_ip'
      sdo_fitscubes2crispex,stx2fullsdodir,outtail,stx2fullsdodir+filestart,$
        /diagname,bytscale=2,/verbose
    endelse
  endif

endif ; end of stx2fullsdo production

; skip the rest if no more productions requested
if (do_stx2sdo+do_stx2rotsdo eq 0) then goto, ENDPRODUCTION

; ========== stx2sdo production ("reverse")

; NB: ?? this currently does not undo the SDO wobble but it should

if (do_stx2sdo ne 0) then begin 

  print,' ----- stx2sdo: write SDO _rev files in '+stx2sdodir

; set outtail
  if (rotate_up) then outtail='_rev_up' else outtail='_rev'
  if (splinip eq 0) then outtail=outtail+'_px.fits' $
  else outtail=outtail+'_ip.fits'

; complication: split reverse shifts between SDO and STX in order to
; maintain STX field center in the rotated reverse display: 
;   - backshift SDO over temporal mean (integer for splinip=0 usage) 
;   - shift STX only over the remaining modulation
  meanrevshifts=[round(mean(revshifts[*,0])),$
                 round(mean(revshifts[*,1]))]
  modrevshifts=[[revshifts[*,0]-meanrevshifts[0]],$
                [revshifts[*,1]-meanrevshifts[1]]]

; optional field rescale
  nxrevscaled=fix(nxrev*nxscale+0.5)
  nyrevscaled=fix(nyrev*nyscale+0.5)
  
; transform all or selected SDO fitscubes
  if (sdo_sel_stx2sdo[0] eq 'all') then $
    doallfiles,sdocubesdir,stx2sdodir,'.fits',outtail,$
    'reformcubefile',$
    rotate=angle_rotate,$
    shift=-meanrevshifts,congridfactor=scale_sdo2stx,$
    cutcentralx=nxrevscaled,cutcentraly=nyrevscaled,$
    missingvalue=-1,splinip=splinip $
  else begin
    nselect=n_elements(sdo_sel_stx2sdo)
    for ifile=0,nselect-1 do begin
      match=0
      for iname=0,10 do $
        if (strmatch(sdo_sel_stx2sdo[ifile],'*'+sdonames[iname]+'*')) $
        then match=1
      if (match eq 1) then begin
        infile=findfile(sdocubesdir+'/*'+sdo_sel_stx2sdo[ifile]+'*.fits')
        outfile=stx2sdodir+'/'+file_basename(infile,'.fits')+outtail
        reformcubefile,infile,outfile,$
          rotate=angle_rotate,$
          shift=-meanrevshifts,congridfactor=scale_sdo2stx,$
          cutcentralx=nxrevscaled,cutcentraly=nyrevscaled,$
          missingvalue=-1,splinip=splinip
      endif else print," !!!!! do_stx2sdo warning:"+ $
        " SDO selection '"+sdo_sel_stx2sdo[ifile]+"' empty or invalid"
    endfor
  endelse
  
; optionally produce multi-channel SDO cubefile in La Palma crispex format
  if (makecrispexsdocube ne 0) then begin
    if (rotate_up eq 0) then filestart='/sdo_crispex_rev' else $
      filestart='/sdo_crispex_rev_up'
    if (splinip ne 0) then filestart=filestart+'_ip' else $
      filestart=filestart+'_px'
    sdo_fitscubes2crispex,stx2sdodir,outtail,stx2sdodir+filestart,$
      /diagname,bytscale=2,/verbose
  endif

; reverse-transform stx files, maintain extensions, dewarp when applypx2asym
  if (verbose ne 0) then print,' ----- convert stxfiles > stx2sdo'
  nstxfiles=n_elements(stx_sel_stx2sdo)
  for istxfp=0,nstxfiles-1 do begin
    infile=stx_sel_stx2sdo[istxfp]
    dummy=cgrootname(infile,extension=ext) ;RR needs coyotelib
    outfile=stx2sdodir+'/'+file_basename(infile,'.'+ext)+'_rev_ip.'+ext
    if (verbose ne 0) then print,' ----- start on '+outfile+' may take long'
    reformcubefile,infile,outfile,$
      shift=modrevshifts,rotate=-angle_stx+angle_rotate,scale=pxscale2,$
      nxlarge=nxrevscaled,nylarge=nyrevscaled,$
      cutcentralx=nxrevscaled,cutcentraly=nyrevscaled,$
      timearr_in=timearr_stx,timearr_out=timearr_sdo,$
      greyborders=1,missingvalue=-1,splinip=1
  endfor
;RR I found that setting greytriangles can give erratic heavy croppings
  
endif ; end of stx2sdo production

; ========== stx2rotsdo production ("rotated")

; NB: includes undoing wobble attributing that to SDO

if (do_stx2rotsdo ne 0) then begin

  print,' ----- do_stx2rotsdo: write SDO _rot files in '+stx2rotsdodir

; set outtail acording to splinip setting
  outtail='_rot_px.fits'
  if (splinip ne 0) then outtail='_rot_ip.fits'

; select SDO sampling time nearest SXT start time 
  starttai_stx=timearr_stx[0]
  dummy=min(abs(timearr_sdo-starttai_stx[0]),it_startsdo) ;f##k IDL: [0] 

; set timearr_rot with cadence = SXT cadence but precise timing regularized, 
; starting at it_startsdo closest to start STX moment
  timearr_rot=indgen(nt_stx-1)*cadence_stx+timearr_sdo[it_startsdo]

; complications: split forward shifts between SDO and STX in order to
;   maintain STX field center in the rotated reverse display: 
;   - shift SDO over temporal mean (integer for splinip=0 usage) 
;   - backshift STX only over the remaining modulation
;   correct SDO for the wobble part and STX for the drift part
  
; get temporal mean forward shift and drift seperation from mean
; (mean = integer pixel values for splinip=0 applications)
  meandriftshifts=[round(mean(driftshifts[*,0])),$
                   round(mean(driftshifts[*,1]))]
  moddriftshifts=[[driftshifts[*,0]-meandriftshifts[0]],$
                  [driftshifts[*,1]-meandriftshifts[1]]]

; set rot image size (just contain the shifting STX field for scale=1)
  nxrot=fix(nxscale*(nxfor+abs(min(driftshifts[*,0]-meandriftshifts[0]))+$
                     abs(max(driftshifts[*,0]-meandriftshifts[0])))+0.5)
  nyrot=fix(nyscale*(nyfor+abs(min(driftshifts[*,1]-meandriftshifts[1]))+$
                     abs(max(driftshifts[*,1]-meandriftshifts[1])))+0.5)

; transform all or selected SDO fitscubes
  if (sdo_sel_stx2rotsdo[0] eq 'all') then $
    doallfiles,sdocubesdir,stx2rotsdodir,'.fits',outtail,$
    'reformcubefile',$
    rotate=angle_stx,shift=meandriftshifts-wobbleshifts,$
    congridfactor=scale_sdo2stx,$
    cutcentralx=nxrot,cutcentraly=nyrot,$
    timearr_in=timearr_sdo,timearr_out=timearr_rot,$
    missingvalue=-1,splinip=splinip $
  else begin
    nselect=n_elements(sdo_sel_stx2rotsdo)
    for ifile=0,nselect-1 do begin
      match=0
      for iname=0,10 do $
        if (strmatch(sdo_sel_stx2rotsdo[ifile],'*'+sdonames[iname]+'*')) $
        then match=1
      if (match eq 1) then begin
        infile=findfile(sdocubesdir+'/*'+sdo_sel_stx2rotsdo[ifile]+'*.fits')
        outfile=stx2rotsdodir+'/'+file_basename(infile,'.fits')+outtail
        reformcubefile,infile,outfile,$
          rotate=angle_stx,shift=meandriftshifts-wobbleshifts,$
          congridfactor=scale_sdo2stx,$
          cutcentralx=nxrot,cutcentraly=nyrot,$
          timearr_in=timearr_sdo,timearr_out=timearr_rot,$
          missingvalue=-1,splinip=splinip 
      endif else print," !!!!! do_stx2rotsdo warning:"+ $
        " SDO selection '"+sdo_sel_stx2rotsdo[ifile]+"' empty or invalid"
    endfor
  endelse

; optionally produce multi-channel SDO cubefile in La Palma crispex format
  if (makecrispexsdocube ne 0) then begin
    filestart='/sdo_crispex_rot_px'
    if (splinip ne 0) then filestart='/sdo_crispex_rot_ip'
    sdo_fitscubes2crispex,stx2rotsdodir,outtail,stx2rotsdodir+filestart,$
      /diagname,bytscale=2,/verbose
  endif

; reverse-transform stx files, maintain extensions, dewarp when applypx2asym
  if (verbose ne 0) then print,' ----- convert stxfiles > stx2rotsdo'
  nstxfiles=n_elements(stx_sel_stx2rotsdo)
  for istxfp=0,nstxfiles-1 do begin
    infile=stx_sel_stx2rotsdo[istxfp]
    dummy=cgrootname(infile,extension=ext) ;RR needs coyotelib
    outfile=stx2rotsdodir+'/'+file_basename(infile,'.'+ext)+'_rot_ip.'+ext
    if (verbose ne 0) then print,' ----- start on '+outfile+' may take long'
    reformcubefile,infile,outfile,$
      shift=-moddriftshifts,scale=pxscale2,$
      nxlarge=nxrot,nylarge=nyrot,cutcentralx=nxrot,cutcentraly=nyrot,$
      timearr_in=timearr_stx,timearr_out=timearr_rot,$
      greyborders=1,missingvalue=-1,splinip=1
  endfor
;RR I found that setting greytriangles can give erratic heavy croppings

endif ; ----- end of stx2rotsdo production

ENDPRODUCTION:  

; print nt's and cadences
if (do_sdo2stx) then print,$
  ' ----- forward nt ='+trimd(n_elements(timearr_stx))+$
  '  cadence ='+trimd(cadence_stx)
if (do_stx2fullsdo) then print,$
  ' ----- full nt ='+trimd(n_elements(timearr_full))+$
  '  cadence ='+trimd(cadence_sdo)
if (do_stx2sdo) then print,$
  ' ----- reverse nt ='+trimd(n_elements(timearr_sdo))+$
  '  cadence ='+trimd(cadence_sdo)
if (do_stx2rotsdo) then print,$
  ' ----- rotsdo nt ='+trimd(n_elements(timearr_rot))+$
  '  cadence ='+trimd(cadence_stx)

; print elapsed time
timedone=systime(1)
timelaps=ntostr((timedone-timestart)/60.,format='(F11.1)')
print,' ===== sta_stb_align took '+timelaps+' minutes'

end


; ================= main for testing per IDLWAVE hyper ==================


cd,'/home/rutten/data/SST/2016-09-05-demo/'  ; standard test 

;; produce ha_lc fitscube for possible ha_lc versus AIA 304 alignment
;; ; inspect CRISP Halpha to find wavelength nrs (or with showex)
;; crispex,'crispex/crispex.6563.09:48:31.time_corrected.aligned.icube',$
;;   'crispex.6563.09:48:31.time_corrected.aligned_sp.icube'
;; crispex2fits,'crispex/crispex.6563.09:48:31.time_corrected.aligned.icube',$
;;   'sst/ha_lc.fits',7,15  ; line center iw=7, nwavs=15

; input parameters
px_stx=0.0567    ; improved from earlier run with skipmetcalf=0
angle_stx=98.16  ; improved from earlier run with skipmetcalf=0

; keywords for findalignimages.pro
obsdate='2016.09.05'
stxalignfile='crispex/wb.6563.09:48:31.corrected.aligned.icube' 
stxtimesfile='crispex/tseries.8542.09:48:31.calib.sav' ; holds also for 6563??
itsample_stx=29  ; specify when derived in initial run
histopstx=0.1    ; muck: cut Halpha network and moat bright points
;; inshift12=[] 
;; trimboxstx=[]

; keywords for sta_stb_findalignfiles.pro
driftspline=1.E3   ; needs lower value for shorter duration?
wobblespline=0.1

; keyword output choices
splinip=1            ; for comparing morphologies
rotate_up=1          ; bit silly for this data set close to disk center
makecrispexsdocube=0 ; much work & not needed for showex (only for crispex)

; keywords defining dirnames
stxname='sst'

; keywords to select amount of response; use all three in initial run(s)
blink=10 
show=0
verbose=1  

; choose alignment mode: afresh or earlier results
do_findalign=0   ; set 0 after running =1 to improve angle, px, shifts
skipmetcalf=1    ; set 1 after entering improved angle and px results 
applypx2asym=0   ; SST doesn't need/want that
check304ha=0     ; plot halc_muck--aia304_muck.ps was terrible
halcfile='sst/ha_lc.fits'  ; needed when check304ha=1
undo304_1600=0   ; needed? sdo/driftscenter/full_shifts_304-1600.ps is bad

; ===== output mode and file selections

do_fulldisk=0    ; get full-disk SDO images at sharpest moment STX
sdo_sel_fulldisk='171'

do_sdo2stx=0       ; forward: SDO on STX (rewrite STX only if px2asym)
stx_sel_sdo2stx=['crispex/crispex.6563.09:48:31.time_corrected.aligned.icube','crispex/crispex.8542.09:48:31.stokesI.icube']
sdo_sel_sdo2stx=['cont','mag','1600','1700','304','171']

do_stx2sdo=0      ; reverse: STX on SDO, cropped to contain rotated STX
stx_sel_stx2sdo=['crispex/crispex.6563.09:48:31.time_corrected.aligned.icube','crispex/crispex.8542.09:48:31.stokesI.icube']
sdo_sel_stx2sdo=['cont','mag','1600','1700','304','171']

do_stx2fullsdo=0   ; reverse with STX inserted into full = non-cropped SDO
stx_sel_stx2fullsdo=['crispex/crispex.6563.09:48:31.time_corrected.aligned.icube','crispex/crispex.8542.09:48:31.stokesI.icube']
sdo_sel_stx2fullsdo=['mag','1600','1700','304','171']

do_stx2rotsdo=1    ; reverse with SDO first rotated to STX orientation
stx_sel_stx2rotsdo=['crispex/crispex.6563.09:48:31.time_corrected.aligned.icube','crispex/crispex.8542.09:48:31.stokesI.icube']
sdo_sel_stx2rotsdo=['cont','mag','1600','1700','304','171']


; full call copied from the top of sta_stb_align.pro

sta_stb_align,$
; ----- input (values must be specified)
  px_stx,angle_stx,$
; ----- mandatory parameters for passing (values need not be specified)
  cadence_stx,px2asym,driftshifts,wobbleshifts,shiftfor,shiftrev,$
  nxfor,nyfor,nxrev,nyrev,euvcor,$
; ----- keywords for findalignimages.pro
  obsdate=obsdate,sdoalignfile=sdoalignfile,$
  stxalignfile=stxalignfile,stxdriftfile=stxdriftfile,$
  stxtimesfile=stxtimesfile,itsample_stx=itsample_stx,$
  skipmetcalf=skipmetcalf,applypx2asym=applypx2asym,$
  inshift12=inshift12,trimboxstx=trimboxstx,$
  smearstx=smearstx,histopstx=histopstx,flatten=flatten,$
  muckdarksdo=muckdarksdo,muckdarkstx=muckdarkstx,$
  muckbrightsdo=muckbrightsdo,muckbrightstx=muckbrightstx,$
; ----- keywords for sta_stb_findalignfiles.pro
  driftspline=driftspline,wobblespline=wobblespline,aligndir=aligndir,$
  check304ha=check304ha,splineha=splineha,$
  halcfile=halcfile,aia304file=aia304file,$
; ----- keywords: output choices
  splinip=splinip,rotate_up=rotate_up,$ 
  nxscale=nxscale,nyscale=nyscale,$
  undo304_1600=undo304_1600,$
  makecrispexsdocube=makecrispexsdocube,$
  do_findalign=do_findalign,do_fulldisk=do_fulldisk,$
  do_sdo2stx=do_sdo2stx,do_stx2sdo=do_stx2sdo,$
  do_stx2rotsdo=do_stx2rotsdo,do_stx2fullsdo=do_stx2fullsdo,$
  stx_sel_sdo2stx=stx_sel_sdo2stx,stx_sel_stx2sdo=stx_sel_stx2sdo,$
  stx_sel_stx2rotsdo=stx_sel_stx2rotsdo,$
  stx_sel_stx2fullsdo=stx_sel_stx2fullsdo,$
  sdo_sel_fulldisk=sdo_sel_fulldisk,$
  sdo_sel_sdo2stx=sdo_sel_sdo2stx,sdo_sel_stx2sdo=sdo_sel_stx2sdo,$
  sdo_sel_stx2rotsdo=sdo_sel_stx2rotsdo,$
  sdo_sel_stx2fullsdo=sdo_sel_stx2fullsdo,$
; ----- keywords defining dirnames
  stxname=stxname,$
  sdocubesdir=sdocubesdir,fulldiskdir=fulldiskdir,$
  sdo2stxdir=sdo2stxdir,stx2sdodir=stx2sdodir,$
  stx2rotsdodir=stx2rotsdodir,stx2fullsdodir=stx2fullsdodir,$
  driftsdir=driftsdir,$
; ----- keywords to choose amount of response
  blink=blink,show=show,verbose=verbose

; ============ examples output inspection 

; inspect full-disk SDO images
if (do_fulldisk) then showex,/allsdo,sdodir=fulldiskdir+'/annot'

; inspect forward results
if (do_sdo2stx) then $
  showex,$
  /allsdo,sdodir='sdo2sst',$
  /allmwf,mwfdirs='crispex',$      ; if no applypx2asym
  ;;  /allmwf,mwfdirs='sdo2sst',$ ; if applypx2asym
  /allfits,fitsdirs=['sdo2sst','sst'],$
  /addlabels,$
  mwfwavsfiles=['crispex/spectfile.6563.idlsave',$
                'crispex/spectfile.8542.idlsave']

; inspect full results
if (do_stx2fullsdo) then begin
  if (rotate_up eq 0) then $
    showex,/allsdo,sdodir=sdocubesdir,$
    /allmwf,mwfdirs=stx2fullsdodir,$
    /addlabels,$
    mwfwavsfiles=['crispex/spectfile.6563.idlsave',$
                  'crispex/spectfile.8542.idlsave']
  if (rotate_up eq 1) then $
    showex,/allsdo,sdodir=stx2fullsdodir,$
    /allmwf,mwfdirs=stx2fullsdodir,$
    /addlabels,$
    mwfwavsfiles=['crispex/spectfile.6563.idlsave',$
                  'crispex/spectfile.8542.idlsave']
endif

; inspect reverse results
if (do_stx2sdo) then $
  showex,/allsdo,sdodir=stx2sdodir,$
  /allmwf,mwfdirs=stx2sdodir,$
  /allfits,fitsdirs=stx2sdodir,/addlabels,$
  mwfwavsfiles=['crispex/spectfile.6563.idlsave',$
                'crispex/spectfile.8542.idlsave']

; inspect rotated results
if (do_stx2rotsdo) then $
  showex,/allsdo,sdodir='sst2rotsdo',$
  /allmwf,mwfdirs='sst2rotsdo',$ 
  /allfits,fitsdirs='sst2rotsdo',/addlabels,$
  mwfwavsfiles=['crispex/spectfile.6563.idlsave',$
                'crispex/spectfile.8542.idlsave']

end

