; file: sst_align.pro = align two SST image sequences a > b 
; init: Jan 14 2020  Rob Rutten  Deil most from sdo_stx_findalignfiles.pro
; last: Jan 27 2020  Rob Rutten  Deil
; note: as sdo_stx_findalignfile but "for" > a2b only (b2a less good) 

;+

pro sst_align,obsdate,name_a,name_b,$
  px_a,px_b,angle_ab,$
  alignfile_a,alignfile_b,timefile_a,timefile_b,$
  sellist_a,sellist_b,$
  time_out=time_out,px_out=px_out,nx_out=nx_out,ny_out=ny_out,$
  itsample_slow=itsample_slow,skipmetcalf=skipmetcalf,$
  inshift_ab=inshift_ab,trimbox_a=trimbox_a,trimbox_b=trimbox_b,$
  smear_a=smear_a,smear_b=smear_b,$
  histop_a=histop_a,histop_b=histop_b,flatten=flatten,$
  muckdark_a=muckdark_a,muckdark_b=muckdark_b,$
  muckbright_a=muckbright_a,muckbright_b=muckbright_b,$
  driftspline=driftspline,aligndir=aligndir,splinip=splinip,$
  a2bdir=a2bdir,do_findalign=do_findalign,$
; ----- keywords to choose response
  blink=blink,show=show,verbose=verbose

 ; align SST sequences "a" to SST sequences "b"
 ;   
 ; MANDATORY INPUT PARAMETERS:
 ;   obsdate = observing date, mandatory format '2016.09.05'
 ;   name_a = string diagnostic a, eg 'cah', 'hb', 'ha', 'cair', 'fei'
 ;   name_b = string diagnostic b
 ;   px_a = pixel size a in arcsec
 ;   px_b = pixel size b in arcsec, may get improved per Metcalf
 ;   angle_ab = rotation angle a > b in degrees, may get improved per Metcalf
 ;   alignfile_a = string name file a for Metcalf and/or drift alignment
 ;   alignfile_b = idem for b
 ;   timefile_a = IDL save file with timing a in CRISP style
 ;   timefile_b = idem for b
 ;   sellist_a = string array filenames a to handle
 ;   sellist_b = idem for b
 ;
 ; OPTIONAL KEYWORD INPUTS:
 ;   time_out: 'a' or 'b' using the overlap timing of that sequence
 ;     (suggestion: dequence with the fastes cadence; default 'a')
 ;   px_out: 'a' or 'b' using that value (suggestion: smallest; default 'a')
 ;   nx_out,ny_out: dimensions aligned products (default 0 = auto containment)
 ;     suggestion: large enough to contain both fields after alignment
 ;     must be the same for different alignments for intercomparison
 ;   itsample_slow = best (sharpest) it in lowest-cadence sequence (auto)
 ;   skipmetcalf = 1/0: determine shift only, no scale + rotate correction
 ;   inshift_ab: start-off shift a > b in finest pixels, 
 ;     ahe program returns an improved value and prints this when verbose=1.
 ;   trimbox_a: 4-elem int vector [xmin,ymin,xmax,ymax] to select subfield
 ;     used in the alignment determination, for example a
 ;     better look-alike part.  Not needed to strip monochrome borders
 ;     or derotation triangles.
 ;   trimbiox_b: idem for b 
 ;   smear_a: muck a by smearing (nr boxcar pixels)
 ;   smear_a: idem b
 ;   histop-a: hist_opt muckimage a at given value (< 1, top only)
 ;   histop_b: idem b
 ;   flatten: subtract boxcar average from each, value = width in coarsest px
 ;     use for spots or limb in the field
 ;   muckdark_a: set a intensities below threshold (units mean=1) to mean
 ;   muckdark_b: idem for b
 ;   muckbright_a: set intensities above threshold (units mean=1) to mean
 ;   muckbright_b: idem for b
 ; ----- keywords for findalignfiles.pro
 ;   driftspline = smooth value (default 1.E3; -1 = no spline, 1.E8 = linear)
 ;   aligndir = string dir to put aligndata in (default aligndat_namea_nameb)
 ; ----- keywords for output choices
 ;   a2bdir = string dir for outpit files (default aligned_namea_nameb)
 ;   splinip = 0: nearest-neighbour = retain pixels (e.g., coarse SDO) 
 ;           = 1: cubic spline interpolation 
 ;   do_findalign = 1/0: do the alignment (default 0)
 ; ----- keywords to choose amount of response
 ;   blink > 0: flickrr intermediate results during blink seconds (default 0)
 ;   show = 1/0: showex final results for stxalignfile and sdoalignfile
 ;   verbose = 1/0: more/less printout in terminal
 ;   
 ; HISTORY:
 ;   Jan 27 2020 RR: start  
;-

; answer no-parameter query 
if (n_params(0) lt 12) then begin
  sp,sst_align
  return       
endif

; set wall-clock timer (seconds)
timestart=systime(1) 

; defaults for keywords
if (n_elements(nx_out) eq 0) then nx_out=0
if (n_elements(ny_out) eq 0) then ny_out=0
if (n_elements(itsample_slow) eq 0) then itsample_slow=-1
if (n_elements(trimbox_a) eq 0) then trimbox_a=-1
if (n_elements(trimbox_b) eq 0) then trimbox_b=-1
if (n_elements(inshift_ab) eq 0) then inshift_ab=[0.,0.]
if (n_elements(smear_a) eq 0) then smear_a=0  
if (n_elements(smear_b) eq 0) then smear_b=0  
if (n_elements(histop_a) eq 0) then histop_a=0
if (n_elements(histop_b) eq 0) then histop_b=0
if (n_elements(flatten) eq 0) then flatten=0  
if (n_elements(muckdark_a) eq 0) then muckdark_a=0 
if (n_elements(muckdark_b) eq 0) then muckdark_b=0 
if (n_elements(muckbright_a) eq 0) then muckbright_a=0 
if (n_elements(muckbright_b) eq 0) then muckbright_b=0 
if (n_elements(driftspline) eq 0) then driftspline=0
if (n_elements(splinip) eq 0) then splinip=0
if (n_elements(aligndir) eq 0) then aligndir='aligndat_'+name_a+'2'+name_b
if (n_elements(a2bdir) eq 0) then a2bdir='aligned_'+name_a+'2'+name_b
if (n_elements(do_findalign) eq 0) then do_findalign=0
if (n_elements(blink) eq 0) then blink=10
if (n_elements(show) eq 0) then show=0
if (n_elements(verbose) eq 0) then verbose=0

; make dirs if not yet existing
if (do_findalign) then spawn,'mkdir -p '+aligndir
spawn,'mkdir -p '+a2bdir

; get time arrays and parameters from SST-style "tseries" timefiles
dummy=cgrootname(timefile_a,extension=timefileext)  ;RR needs coyotelib
if (timefileext eq 'sav' or timefileext eq 'idlsave') then begin
  restore,timefile_a 
  if (n_elements(time) ne 0) then datetime_a=obsdate+'_'+time else $
    if (n_elements(csday) ne 0) then begin
    tai_a=anytim2tai(obsdate)+double(csday) ; seconds on day 
    datetime_a=anytim2utc(tai_a,/ccsds)
  endif else begin
    print,' ##### sst_align.pro abort: timefile_a not known SST format'
    retall
  endelse
  timearr_a=anytim2tai(datetime_a)
  nt_a=n_elements(timearr_a)
  cadence_a=(timearr_a[nt_a-1]-timearr_a[0])/(nt_a-1)
endif
dummy=cgrootname(timefile_b,extension=timefileext)  ;RR needs coyotelib
if (timefileext eq 'sav' or timefileext eq 'idlsave') then begin
  restore,timefile_b 
  if (n_elements(time) ne 0) then datetime_b=obsdate+'_'+time else $
    if (n_elements(csday) ne 0) then begin
    tai_b=anytim2tai(obsdate)+double(csday)  ; seconds on day 
    datetime_a=anytim2utc(tai_b,/ccsds)
  endif else begin
    print,' ##### sst_align.pro abort: timefile_a not known SST format'
    retall
  endelse
  timearr_b=anytim2tai(datetime_b)
  nt_b=n_elements(timearr_b)
  cadence_b=(timearr_b[nt_b-1]-timearr_b[0])/(nt_b-1)
endif

; set timearr_out
if (cadence_b gt cadence_a) then begin
  slow='b'
  name_slow=name_b
  infile=alignfile_b
  trimbox=trimbox_b
  timearr_out=timearr_a
  cadence_out=cadence_a
endif else begin
  slow='a'
  name_slow=name_a
  infile=alignfile_a
  trimbox=trimbox_a
  timearr_out=timearr_b
  cadence_out=cadence_b
endelse  

; restrict timeearr_out to temporal overlap sequences a and b
start_out=max([min(timearr_a),min(timearr_b)])
end_out=min([max(timearr_a),max(timearr_b)])
timearr_out=timearr_out[where(timearr_out ge start_out)]
timearr_out=timearr_out[where(timearr_out le end_out)]
nt_out=n_elements(timearr_out)

; ======== do_findalign set: find alignment parameters (slow business)

if (do_findalign) then begin

    ; delete older drift figures
  spawn,'rm -f '+aligndir+'/*muck*--*muck*.ps'

; find itsample_slow for sequence with lowest cadence
  if (itsample_slow eq -1) then begin
    if (verbose ne 0 and nt_b gt 100) then print,$
      ' ----- find itsample_slow takes long; better note result'
    findrmscubefile,infile,maxrms,itsample_slow,$
      trimbox=trimbox,croptriangles=1,rmsplot=show,show=show
    if (verbose ne 0) then print,' ----- itsample_slow '+name_slow+' = '+$
      trimd(itsample_slow)
  endif

; find closest-in-time itsample_fast for the faster sequence
  if (slow eq 'a') then timebest=timearr_a[itsample_slow] else $
    timebest=timearr_b[itsample_slow]
  dummy=min(abs(timearr_out-timebest[0]),itsample_fast) 

; get the two sample images
  if (slow eq 'a') then begin
    itsample_a=itsample_slow
    itsample_b=itsample_fast
  endif else begin
    itsample_b=itsample_slow
    itsample_a=itsample_fast
  endelse
  dummy=cgrootname(alignfile_a,extension=ext_a)  ;RR needs coyotelib
  if (ext_a eq 'fits') then $
    im_a=readfitscube(alignfile_a,trange=[itsample_a,itsample_a])
  if (ext_a eq 'icube' or ext_a eq 'fcube' or ext_a eq 'bcube') then $
    im_a=readsstfile(alignfile_a,trange=[itsample_a,itsample_a])
  dummy=cgrootname(alignfile_b,extension=ext_b)  ;RR needs coyotelib
  if (ext_b eq 'fits') then $
    im_b=readfitscube(alignfile_b,trange=[itsample_b,itsample_b])
  if (ext_b eq 'icube' or ext_b eq 'fcube' or ext_b eq 'bcube') then $
    im_b=readsstfile(alignfile_b,trange=[itsample_b,itsample_b])

; extend im_a to larger field of im_b in arcsec
  size_a=size(im_a)
  size_b=size(im_b)
  extra=[500,500]
  if (inshift_ab[0] ne 0) then extra=abs(fix(3*inshift_ab))
  nx_a_large=fix(max([size_a[1],size_b[1]*(px_b/px_a)])+extra[0])
  ny_a_large=fix(max([size_a[2],size_b[2]*(px_b/px_a)])+extra[1])
  reformimage,im_a,im_a_large,nxlarge=nx_a_large,nylarge=ny_a_large,$
    cutcentralx=nx_a_large,cutcentraly=ny_a_large,missingvalue=-1

;; ; write temporary files for standalone findalignimages
;; writefits,'im_a_large.fits',im_a_large
;; writefits,'im_b.fits',im_b

; do full alignment for this "best" image pair (full call)
  findalignimages,im_a_large,im_b,px_a,px_b,angle_ab,$,
    px2asym,shifta2b,shiftb2a,nxa2b,nya2b,nxb2a,nyb2a,$    
    nxmuckfor=nxmucka2b,nymuckfor=nymucka2b,$
    nxmuckrev=nxmuckb2a,nymuckrev=nymuckb2a,$
    skipmetcalf=skipmetcalf,applypx2asym=0,$
    trimboxim2=trimbox,inshift12=inshift_ab,$
    smearim1=smear_a,smearim2=smear_b,$
    histopim1=histop_a,histopim2=histop_b,$
    muckdarkim1=muckdark_a,muckdarkim2=muckdark_b,$
    muckbrightim1=muckbright_a,muckbrightim2=muckbright_b,$
    flatten=flatten,blink=blink,show=show,verbose=verbose

; set congridfactors as under @@@@@ demo in findalignimages.pro
  pxratio=float(px_a)/px_b
  pxrat2d=[pxratio,pxratio]
  pxscale_a=[1.0,1.0]
  pxscale_b=[1.0,1.0]
  if (pxratio lt 0.999) then pxscale_b=1./pxrat2d
  if (pxratio gt 1.001) then pxscale_a=pxrat2d

; ------reform both files forward to temp files with splinip=1 for 
;       subpixel cross-alignment in drift determination
;       (as sdo_stx_findalignfiles.pro)

; get two unique muck filenames
  rannr=fix(abs(randomn(seed))*10000)
  muckfile_a='/tmp/'+name_a+'_muck'+$
    strmid(string(rannr+1E5,format='(i6)'),2)+'.fits'
  muckfile_b='/tmp/'+name_b+'_muck'+$
    strmid(string(rannr+1E5,format='(i6)'),2)+'.fits'

  if (verbose) then print,' ----- write '+muckfile_a+' and '+muckfile_b+$
    ' may take long'
  reformcubefile,alignfile_a,muckfile_a,$
    congridfactor=pxscale_a,rotate=angle_ab,shift=shifta2b,$
    nxlarge=nx_a_large,nylarge=ny_a_large,$
    cutcentralx=nxmucka2b,cutcentraly=nymucka2b,missingvalue=-1,$
    histopttop=histop_a,smear=smear_a,$
    muckdark=muckdark_a,muckbright=muckbright_a,flatten=flatten*pxratio,$
    timearr_in=timearr_a,timearr_out=timearr_out,$
    splinip=1

  reformcubefile,alignfile_b,muckfile_b,$
    congridfactor=pxscale_b,$
    cutcentralx=nxmucka2b,cutcentraly=nymucka2b,$
    histopttop=histop_b,smear=smear_b,$
    muckdark=muckdark_b,muckbright=muckbright_b,flatten=flatten*pxratio,$
    timearr_in=timearr_b,timearr_out=timearr_out,splinip=1

;;;  inspect
;; showex,muckfile_a,muckfile_b
;; STOP

; cross-align these temporary cubefiles to find time-dependent drifts b>a
  wobblespline=0 ; no SDO wobble here
  shiftalignfitscubes,muckfile_b,muckfile_a,$
    driftshifts,wobbleshifts,$
;;  outfile1='/tmp/'+name_b+'_xal.fits',outfile2='/tmp/'+name_a+'_xal.fits',$
    niter=3,cutiter=0.05,$
    driftspline=driftspline,wobblespline=wobblespline,nspliter=2,$
    plotdir=aligndir

;; ; check mucked-pair alignment (ONLY if optional output files written)
;; showex,'/tmp/'+name_a+'_xal.fits','/tmp/'+name_b+'_xal.fits'
;; STOP

; get total shift arrays on the rescaled scale
; no split into drift and wobble (as for SDO-SST)
  a2bshifts=fltarr(nt_out,2)
  for it=0,nt_out-1 do begin
    a2bshifts[it,0]=shifta2b[0]+driftshifts[it,0]+wobbleshifts[it,0]
    a2bshifts[it,1]=shifta2b[1]+driftshifts[it,1]+wobbleshifts[it,1]
  endfor

; write results in data file for future usage (use unit nr to append)
; add explanation text after variables per line for direct reading
  aligndatfile=aligndir+'/'+name_a+'2'+name_b+'-align.dat'
  openw,42,aligndatfile   ;RR answer to everything
  printf,42,trimd(angle_ab)+trimd(px_a,4)+trimd(px_b,4)+$
    ' = angle_ab  px_a  px_b'
  printf,42,trimd(nt_out)+trimd(cadence_out)+trimd(nxa2b)+trimd(nya2b)+$
    ' = nt_out  cadence_out  nxa2b nya2b'
  printf,42,trimd(itsample_a)+trimd(itsample_b)+' = itsample_a  itsample_a'
  printf,42,' below: timearr_out  [xy]drift  [xy]wobble  [xy]a2b'
  xdrift=driftshifts[*,0]
  ydrift=driftshifts[*,1]
  xwobble=wobbleshifts[*,0]
  ywobble=wobbleshifts[*,1]
  xshifta2b=a2bshifts[*,0] ; this is total "forward" shift
  yshifta2b=a2bshifts[*,1]
  writecol,aligndatfile,timearr_out,xdrift,ydrift,xwobble,ywobble,$
    xshifta2b,yshifta2b,fmt='(1E20.10,6F10.4)',filnum=42
  close,42

; showblink forward result
  if (show ne 0) then begin

; write splinipped alignfile_a in /tmp for show
    aligna2bfile_a=file_basename(alignfile_a,'.fits')
    aligna2bfile_a='/tmp/'+aligna2bfile_a+'_a2b_ip.fits'
    if (verbose) then print,' ----- writing show aligna2bfile_a = '+$
      aligna2bfile_a
    reformcubefile,alignfile_a,aligna2bfile_a,$
      rotate=angle_ab,congridfactor=pxscale_a,$
      shift=a2bshift,cutcentralx=nxa2b,cutcentraly=nya2b,$
      timearr_out=timearr_out,missingvalue=-1,splinip=1

    print,' ----- showblink: a>b result align files'
    showex,aligna2bfile_a,alignfile_b,trimbox=trimbox
  endif

; ====== end of do_findalign alignment

endif else begin

; do_findalign=0: instead read earlier alignment data
  aligndatfile=aligndir+'/'+name_a+'2'+name_b+'-align.dat'
  if (file_test(aligndatfile) eq 0) then begin
    print,' ##### sst_align abort: '+$
      'do_findalign=0 but no aligndatafile found'
    return
  endif
  print,' ----- no new alignment but use '+aligndatfile
  openr,42,aligndatfile  ;RR 42 is answer to everything
  readf,42,angle_ab,px_a,px_b
  readf,42,nt_out,cadence_out,nxa2b,nya2b
  close,42
  readcol,aligndatfile[0],timearr_out,xdrift,ydrift,xwobble,ywobble,$
    xshifta2b,yshifta2b,$
    skipline=3,format='D',/silent   ;RR double precision needed for timearr
  driftshifts=[[xdrift],[ydrift]]
  wobbleshifts=[[xwobble],[ywobble]]
  a2bshifts=[[xshifta2b],[yshifta2b]]
  if (n_elements(timearr_out) ne nt_out) then $
    print,' ##### sst_align error: nt_out does not match timearr_out '+$
    'in file '+aligndatfile
endelse

; ======= end of getting alignment = determination or reading older

; set congridfactors as under @@@@@ demo in findalignimages.pro
; repeat because px_b may have been adapted per Metcalf or just read
pxratio=float(px_a)/px_b
pxrat2d=[pxratio,pxratio]
pxscale_a=[1.0,1.0]
pxscale_b=[1.0,1.0]
if (pxratio lt 0.999) then pxscale_b=1./pxrat2d
if (pxratio gt 1.001) then pxscale_a=pxrat2d

; ====== find nx and ny containing both fields after maximum a2b shift

if (nx_out eq 0 or ny_out eq 0) then begin
  
; get largest forward shift in finest px
  mima_x=minmax(a2bshifts[*,0])
  mima_y=minmax(a2bshifts[*,1])
  if (mima_x[0] lt 0) then xoff=-max(abs(mima_x)) else xoff=max(mima_x)
  if (mima_y[1] lt 0) then yoff=-max(abs(mima_y)) else yoff=max(mima_y)
; rotate 
  rotoff=rotatevec([xoff,yoff],angle_ab)

; read first images for their sizes
  dummy=cgrootname(alignfile_a,extension=ext_a)  ;RR needs coyotelib
  if (ext_a eq 'fits') then $
    im_a=readfitscube(alignfile_a,trange=[0,0])
  if (ext_a eq 'icube' or ext_a eq 'fcube' or ext_a eq 'bcube') then $
    im_a=readsstfile(alignfile_a,trange=[0,0])
  dummy=cgrootname(alignfile_b,extension=ext_b)  ;RR needs coyotelib
  if (ext_b eq 'fits') then $
    im_b=readfitscube(alignfile_b,trange=[0,0])
  if (ext_b eq 'icube' or ext_b eq 'fcube' or ext_b eq 'bcube') then $
    im_b=readsstfile(alignfile_b,trange=[0,0])

; get sizes in finest px
  size_a=size(im_a)
  nx_a_fine=size_a[1]*pxscale_a[0]
  ny_a_fine=size_a[2]*pxscale_a[1]
  size_b=size(im_b)
  nx_b_fine=size_b[1]*pxscale_b[0]
  ny_b_fine=size_b[2]*pxscale_b[1]

; find size of rotated a including rotated shift
  rotdim_a=rotatevec([nx_a_fine,nx_b_fine],angle_ab)
  nx_a_large=rotdim_a[0]+2*abs(rotoff[0])
  ny_a_large=rotdim_a[1]+2*abs(rotoff[1])

; find full forward = a2b contain size
  nx_out=fix(max([nx_a_large,nx_b_fine]))
  ny_out=fix(max([ny_a_large,ny_b_fine]))

  print,' ===== nx_out,ny_out = '+trimd(nx_out)+',  '+trimd(ny_out)
  
endif

; ========== a2b production ("forward")

print,' ----- write a2b files in '+a2bdir

; set outtail acording to splinip setting
if (splinip eq 0) then outtail='_'+name_a+'2'+name_b+'_px.fits' else $
  outtail='_'+name_b+'2'+name_a+'ip.fits'

; do the a files
nselect=n_elements(sellist_a)
for ifile=0,nselect-1 do begin
  infile=sellist_a[ifile]
  dummy=cgrootname(infile,extension=ext)  ;RR needs coyotelib ;
  ext='.'+ext
  outfile=a2bdir+'/'+file_basename(infile,ext)+outtail
  reformcubefile,infile,outfile,$
    rotate=angle_ab,congridfactor=pxscale_a,$
    shift=a2bshifts,$
    nxlarge=nx_out,nylarge=ny_out,$
    cutcentralx=nx_out,cutcentraly=ny_out,$
    timearr_in=timearr_a,timearr_out=timearr_out,$
    missingvalue=-1,splinip=splinip
endfor

; do the b files
nselect=n_elements(sellist_b)
for ifile=0,nselect-1 do begin
  infile=sellist_b[ifile]
  dummy=cgrootname(infile,extension=ext)  ;RR needs coyotelib ;
  ext='.'+ext
  outfile=a2bdir+'/'+file_basename(infile,ext)+outtail
  reformcubefile,infile,outfile,$
    rotate=0,congridfactor=pxscale_b,$
    shift=[0,0],$
    nxlarge=nx_out,nylarge=ny_out,$
    cutcentralx=nx_out,cutcentraly=ny_out,$
    timearr_in=timearr_b,timearr_out=timearr_out,$
    missingvalue=-1,splinip=splinip
endfor

; =======================================================================

; print nx_out, ny_out again for visibility and noting
print,' ===== NOTE: nx_out,ny_out = '+trimd(nx_out)+',  '+trimd(ny_out)

; print elapsed time
timedone=systime(1)
timelaps=ntostr((timedone-timestart)/60.,format='(F11.1)')
if (verbose ne 0) then print,' ===== sst_align.pro took '+timelaps+' minutes'

end ; end of program            ;

; =============== main for  testing per IDLWAVE H-c ======================

cd,'/media/rutten/SSTNEW/alldata/SST/2018-06-10-vasco'

obsdate='2018.06.10'
name_a='cah'
name_b='ha'
px_a=0.0377  
px_b=0.0587     ; after Metcalf
angle_ab=-1.05  ; after Metcalf
alignfile_a='chromis/cont_cah.fits'
alignfile_b='crisp/wb.6563.12:21:09.corrected.icube'
timefile_a='chromis/tseries_chromis.sav'
timefile_b='crisp/tseries.6563.12:21:09.calib.sav'
trimbox_b=[200,200,800,800]
inshift_ab=[200,81]
itsample_slow=125

; output files
; bloody lot
sellist_a=['chromis/nb_3950_2018-06-10T12:21:19_scans=0-286_corrected_int_im.icube',$
           'chromis/cont_cah.fits'] 
sellist_b=['crisp/crispex.6563.12:21:09.time_corrected.icube',$
           'crisp/crispex.8542.12:21:09.time_corrected.icube',$
           'crisp/crispex.stokes.6302.12:21:09.time_corrected.icube',$
           'crisp/wb.6302.12:21:09.corrected.icube',$
           'crisp/wb.6563.12:21:09.corrected.icube',$
           'crisp/wb.8542.12:21:09.corrected.icube']

;; ; overwrite sellist: continuum only
;; sellist_a='chromis/cont_cah.fits'
;; sellist_b='crisp/wb.6563.12:21:09.corrected.icube'

smear_a=4       ; CHROMIS px 2x finer
smear_b=2 
histop_a=1E-3   ; cut MBPs 
histop_b=1E-3
driftspline=0   ; spline bad at jump and early end; better none
splinip=0

do_findalign=1
skipmetcalf=1

blink=5
show=0
verbose=1

; copy of complete call at the start of this file
sst_align,obsdate,name_a,name_b,$
  px_a,px_b,angle_ab,$
  alignfile_a,alignfile_b,timefile_a,timefile_b,$
  sellist_a,sellist_b,$
  nx_out=nx_out,ny_out=ny_out,$
  itsample_slow=itsample_slow,skipmetcalf=skipmetcalf,$
  inshift_ab=inshift_ab,trimbox_a=trimbox_a,trimbox_b=trimbox_b,$
  smear_a=smear_a,smear_b=smear_b,$
  histop_a=histop_a,histop_b=histop_b,flatten=flatten,$
  muckdark_a=muckdark_a,muckdark_b=muckdark_b,$
  muckbright_a=muckbright_a,muckbright_b=muckbright_b,$
  driftspline=driftspline,aligndir=aligndir,splinip=splinip,$
  a2bdir=a2bdir,do_findalign=do_findalign,$
; ----- keywords to choose response
  blink=blink,show=show,verbose=verbose

end
