; file: readsstfile.pro = load (part of) a (classic) SST LP cube file 
; init: Oct 28 2012 Rob Rutten
; last: Aug 12 2014  Rob Rutten  Oslo

function readsstfile,cubefile,$
  xrange=xrange,yrange=yrange,trange=trange,$
  byt=byt,nx=nx,ny=ny,nt=nt

;+
 ; NAME:
 ;   readsstfile
 ; PURPOSE:
 ;   read [partial] SST non-CRISP cubefile into cube in memory
 ; CALL:
 ;   cube=readsstfile(filename,$
 ;     xrange=xrange,yrange=yrange,trange=trange,$
 ;     byt=byt,nx=nx,ny=ny,nt=nt
 ; INPUTS:
 ;   SST [x,y,t] data filename in string quotes
 ; KEYWORDS:
 ;   xrange=xrange,yrange=yrange,trange=trange,   [input]: limit ranges  
 ;   byt=byt         [input]: return bytescaled cube to conserve memory 
 ;   nx, ny, nt     [output]: return output cube dimensions for further use
 ; OUTPUTS:
 ;   cube(nx,ny,nt), bytescaled if requested
 ; OPTIONAL OUTPUTS:
 ;   nx, ny, nt     [output]: return output cube dimensions for further use
 ; RESTRICTIONS:
 ;   requires lp_readheader.pro (RJR: in sstlib)
 ; HISTORY:
 ;   Oct 28 2012 RJR started
 ;   Oct 12 2013 RJR optional size cutters: xrange, yrange, trange
;-

; no-parameter answer to x=readsstfile() query
if n_params() eq 0 then begin
  print,' cube=readsstfile(cubefile [, options])'
  print,' options: xrange=xrange,yrange=yrange,trange=trange,'
  print,'          byt=byt,nx=nx,ny=ny,nt=nt'
  return,-1           ; function must return something
endif

; default keywords
if (n_elements(xrange) eq 0) then xrange=[0,-1]
if (n_elements(yrange) eq 0) then yrange=[0,-1]
if (n_elements(trange) eq 0) then trange=[0,-1]
if (n_elements(byt) eq 0) then byt=0

; read the file header to get data type and cube dimensions
lp_readheader,cubefile,header=header,datatype=datatype, $
          dims=dims,nx=nxdat,ny=nydat,nt=ntdat,endian=endian_file

; define assoc input size and type
if (datatype eq 1) then imarr=bytarr(nxdat,nydat)
if (datatype eq 2) then imarr=intarr(nxdat,nydat)
if (datatype eq 4) then imarr=fltarr(nxdat,nydat)

; define output cube size and type
if (xrange[1] eq -1) then xrange[1]=nxdat-1
if (yrange[1] eq -1) then yrange[1]=nydat-1
if (trange[1] eq -1) then trange[1]=ntdat-1
nx=xrange[1]-xrange[0]+1
ny=yrange[1]-yrange[0]+1
nt=trange[1]-trange[0]+1
if (datatype eq 1) then cube=bytarr(nx,ny,nt)
if (datatype eq 2) then cube=intarr(nx,ny,nt)
if (datatype eq 4) then cube=fltarr(nx,ny,nt)

; get the images in the [segment of the] cube
; assoc the images from [only the reqested segment of] the cubefile
; (cut immediately to reduce array size for too large data cubes)
get_lun, unit
openr,unit,cubefile
p=assoc(unit,imarr,512)  
for it=trange[0],trange[1] do begin
  image=p[it]
  cube[*,*,it-trange[0]]=image[xrange[0]:xrange[1],yrange[0]:yrange[1]]
endfor
free_lun,unit

; bytescale the whole cube (not the individual images) if requested
if (byt eq 1) then cube=bytscl(cube)

; done
return,cube
end 

; ==================== check per IDLWAVE S-c ========================

path='/home/rutten/data/SST/2011-05-07-test/crispex/'
cubefile=path+'halpha_1.2011.05.07.strt.algn.icube'
cube=readsstfile(cubefile)
showex,cube,/bytsclind
end
