; file: readcrispfile.pro = read one wav of SST/CRISP data file into memory 
; init: Jul 17 2013  Rob Rutten
; last: Mar 16 2018  Rob Rutten  Deil
; note: requires knowledge of spectral samplings: total number
;       (parameter nwav) and selected wavelength (parameter iwav). Get
;       them from lp_readheader,filename or from running crispex

function readcrispfile,infile,iwav,nwav,$
  istokes=istokes,$
  xrange=xrange,yrange=yrange,trange=trange,$
  byt=byt,nx=nx,ny=ny,nt=nt

;+
 ; NAME:
 ;   readcrispfile
 ; PURPOSE:
 ;   read [partial] SST/CRISP cubefile into cube in memory
 ; CALL:
 ;   cube=readcrispfile(infile,iwav,nwav,$
 ;     istokes=istokes,xrange=xrange,yrange=yrange,trange=trange,$
 ;     byt=byt,nx=nx,ny=ny,nt=nt) 
 ; INPUTS:
 ;   infile = string 'path/file' 
 ;   iwav = index selected wavelength
 ;   nwav = total nr of sampling wavelengths
 ; OPTIONAL INPUTS:
 ;   istokes=istokes: [I,Q,U,V], so istokes=0 selects intensity
 ;   xrange=xrange,yrange=yrange,trange=trange: limit ranges  
 ;    byt=byt: return bytscaled values to conserve memory 
 ; OUTPUTS:
 ;   cube in memory   
 ; OPTIONAL OUTPUTS:
 ;   nx, ny, nt: output cube dimensions for further use
 ; HISTORY:
 ;   Jul 17 2013 RR: start   
 ;   Jul 25 2014 RR: this info block
;-

; default keywords
if (n_elements(byt) eq 0) then byt=0
if (n_elements(istokes) eq 0) then istokes=0
if (n_elements(xrange) eq 0) then xrange=[0,-1]
if (n_elements(yrange) eq 0) then yrange=[0,-1]
if (n_elements(trange) eq 0) then trange=[0,-1]

; read the file header to get data type and cube dimensions
;RR OOPS there is no specification of the number of wavelength samples
crispex_read_header,infile,header=header,datatype=datatype, $
  dims=dims,nx=nxdat,ny=nydat,nt=ntdat,endian=endian_file,$
  stokes=stokes, ns=ns, diagnostics=diagnostics
ntdat=fix(0.1+ntdat/nwav) ;RR get the real nt in case of small mistake
ntdat=ntdat/ns  ;RR 4 Stokes samplings if Stokes

; check on Stokes clash
if (istokes ne 0 and ns eq 1) then istokes=0

; define assoc input size and type
if (datatype eq 1) then imarr=bytarr(nxdat,nydat)
if (datatype eq 2) then imarr=intarr(nxdat,nydat)
if (datatype eq 4) then imarr=fltarr(nxdat,nydat)

; define output cube size and type
if (xrange[1] eq -1) then xrange[1]=nxdat-1
if (yrange[1] eq -1) then yrange[1]=nydat-1
if (trange[1] eq -1) then trange[1]=ntdat-1
nx=xrange[1]-xrange[0]+1
ny=yrange[1]-yrange[0]+1
nt=trange[1]-trange[0]+1
if (datatype eq 1) then cube=bytarr(nx,ny,nt)
if (datatype eq 2) then cube=intarr(nx,ny,nt)
if (datatype eq 4) then cube=fltarr(nx,ny,nt)

; assoc the images from [only the requested segment of] the infile
; (cut immediately to reduce array size for too large data cubes)
get_lun, unit
openr,unit,infile
p=assoc(unit,imarr,512)  
for it=trange[0],trange[1] do begin
;RR crispex cubes are 3D but actually have 4 or 5 dimensions
;RR crispex FAQ answer: image = image_cube[*,*,t*nlp*ns+s*nlp+lp]
  image=p[it*nwav*ns+istokes*nwav+iwav]
  cube[*,*,it-trange[0]]=image[xrange[0]:xrange[1],yrange[0]:yrange[1]]
endfor
free_lun,unit

; bytscale if requested
if (byt eq 1) then cube=bytscl(cube)

; done
return,cube

end

; =============== test per IDLWAVE S-c ==============================

path='/home/rutten/data/SST/2011-05-07-test/crispex/'
infile='halpha_1.2011.05.07.strt.algn.icube'
cube=readcrispfile(path+infile,1,3)
showex,cube
end
