; file prof2special.pro
; init: Oct 12 2017  Rob Rutten  Deil = Luc's lp_halpha_core_width.pro
; last: Apr  7 2018  Rob Rutten  Deil
; site: rridl/sstlib

;+

pro prof2special,delwav,prof,chordval,mode,meanprof,$
  minint,minshift,chordshift,chordwidth,endmin,endmax,enddif,enddop,$
  show=show

; PURPOSE
;   RR special: measure parameters from a single spectral line profile
;   Used in sstlib/mwseq2special.pro
; 
; METHOD:
;   rewrite of Luc Rouppe van der Voort's "lp_halpha_core_width.pro" which was
;   inspired by "core" measures in Cauzzi et al. 2009A&A...503..577C where:
;     minint = minimum of 2nd-order fit through 5 points around lowest pixel
;     shift = wavelength shift of the minimum from zero delwav mean profile
;     width = FWHM of partial profile extending to the mean of the intensities
;             at chordval from this minimum.
;     This is the 'kevin' mode below, used in 2009A&A...503..577C ("caha")
;     to evaluate properties of "chromospheric cores".
;     The only differences is the five-point wavelength spacing set in
;     delwav (it was 0.1 A in Halpha and 0.08 A in CaII 8542 in
;     2009A&A...503..577C) and its zero value (set by mean profile
;     here, by atlas profile in 2009A&A...503..577C).
;     The chordval "end" values defining the "core" extent as delta wavelength
;     from the fit minimum wavelength was 0.9 AA for Halpha, 0.6 AA for 8542.
;   'luc' mode puts the profile cutoff points at fixed chordval AA from the 
;     nominal line center, not from the fitted minimum per pixel.  The FWHM 
;     chord is similarly the one halfway up between the actual pixel profile 
;     minimum and the mean of these two core-cutoff "end" intensities.   
;     NB: Luc mode with chordval=0.9 AA Halpha shiftmaps closely resemble SST
;         Dopplergrams at 0.6 AA, much simpler to make (e.g., within movex).
;   'chord' mode does not use a partial-profile FWHM to define the measurement
;     chord but uses the chord at the highest of the profile intensities 
;     at delwav=-chordval and delwav=+chordval.  
;   'frac' mode defines the chord ends where the profile per pixel equals 
;     the intensity reached at inverse linedepth = chordval of the 
;     mean profile averaged over the data sequence (1.0 = at mean maximum
;     intensity, 0.0 = at mean minimum intensity).  
;
; MANDATORY INPUT PARAMETRS:
;   delwav = array with wavelength offsets from nominal line center in AA 
;   prof   = array with corresponding intensities at these wavelength points
;   chordval = value that defines the core cutoff ends, see above
;     (does not need to correspond to actual delwav samples)
;   mode = 'kevin', 'luc', 'chord', 'frac'
;   meanprof = mean profile in 2D array [delwav,intensity] 
; 
; MANDATORY OUTPUT PARAMETERS:
;   minint     = intensity at fitted profile minimm 
;   minshift   = delwav of fitted minimum in AA from nominal line center
;   chordshift = delwav of the midpoint of the FWHM measurement chord in AA
;   chordwidth = extent in AA of the FWHM measurement chord
;   endmin = smallest of the intensities at "core" ends (may be both in core)
;   endmax = largest  of the intensities at "core" ends (may be both in core)
;   enddif = red-blue difference of these intensities (in core = shock?)
;   enddop = Dopplergram (red-blue)/(red+blue) at "core" ends
;
; OPTIONAL INPUT PARAMETERS:
;   show = 1/0: show profile with results (default 0)
;
; ISSUES:
;   Luc used IDL interpol but that requires monotonous slope and misfires
;   at profile humps (as in core 8542 in Kevin Fig 2 paper caha).  
;   Replaced here by lambdameter-like chord dropping.
;
; HISTORY:
;   xxx xx 2016 Luc Rouppe van der Voort: lp_halpha_core_width.pro 
;   Oct 22 2017 RR: ruttenization, lambdameter, more modes
;   Mar 23 2018 RR: > prof2special, endmin, endmax, enddif, enddop
;-

; answer no-parameter query 
if (n_params(0) lt 13) then begin
  sp,prof2special
  return
endif

; defaults for keywords
if (n_elements(show) eq 0) then show=0  

; check on mode
if (mode ne 'kevin' and mode ne 'luc' and mode ne 'chord' $
    and mode ne 'frac') then begin
  print," ##### prof2special abort: "$
    +"not mode 'kevin', 'luc', 'chord', 'frac'"
  return
endif

; check that full extent includes partial extent
if (min(delwav) gt -chordval or max(delwav) lt +chordval) then begin
  print,' ##### prof2special.pro abort: spectral range too short for'
  print,' chordval ='+trimd(chordval)
  return
endif 

; profile  quantities
nwav=n_elements(delwav)
wavrange=max(delwav)-min(delwav)
profmax=max(prof,iwmax)  ; iwmax = wav index profile max
profmin=min(prof,iwmin)  ; iwmin = wav index profile min 

; start plot if show=1
if (show ne 0) then begin
  window,1
  ymax=max([profmax,max(meanprof[*,1])])*1.05
  plot,delwav,prof,xstyle=1,yrange=[0,ymax],ystyle=1
  oplot,delwav,meanprof[*,1],linestyle=2
endif 

; skip bad profile (eg derotation edge) and flag 
if (iwmin lt nwav/4. or iwmin gt nwav*3./4 $
    or profmin gt profmax or profmax lt 0 $
    or profmax lt 0.5*min(meanprof[*,1]) ) then begin
;;;  print,' ----- prof2special: bad profile'
  minshift=-1000
  chordshift=-1000
  chordwidth=-1000
  return
endif

; fit 5 points betwixt minimum with 2nd degree polynomial (after Kevin)
minfit=poly_fit(delwav[iwmin-2:iwmin+2],prof[iwmin-2:iwmin+2],2,$
                yfit=yfit)
minshift=-minfit[1]/(2*minfit[2])  ; derivative polynomial zero 
minint=minfit[0]+minfit[1]*minshift+minfit[2]*minshift^2
if (mode eq 'kevin') then addshift=minshift else addshift=0

; show core fit in plot
if (show) then begin
  oplot,delwav[iwmin-2:iwmin+2],yfit,linestyle=3,psym=5  ; triangles
  plots,minshift,minint,psym=2,symsize=2                 ; asterisk
endif

; find full (upper) chord end intensity values
; get separations of chordval values from closest delwav sammples
m0=min(abs(delwav+chordval+addshift),l0)   ;RR l0 = subscript for minimum
m1=min(abs(delwav-chordval+addshift),l1)
if (m0 lt 0.001 and m1 lt 0.001) then begin
; chordval values coincide with delwav points
  prof0=prof[l0]
  prof1=prof[l1]
endif else begin
; chordval values not sampled in delwav 
;   (using IDL interpol is OK since delwav is monotonous)
  prof0=interpol(prof[0:iwmin-1],delwav[0:iwmin-1],-chordval+addshift)
  prof1=interpol(prof[iwmin+1:*],delwav[iwmin+1:*],chordval+addshift)
endelse 

; outputs for intensity values at ends of the full (upper) chord
endmin=min([prof0,prof1])
endmax=max([prof0,prof1])
enddif=prof1-prof0
enddop=(prof0-prof1)/(prof0+prof1)  ; black = blue coreshift 

; show plus sign at these ends
if show then begin
  plots,-chordval+addshift,prof0,psym=1,symsize=3  ; psym 1 plus
  plots,+chordval+addshift,prof1,psym=1,symsize=3
endif 

; set chordint = intensity of the (lower) width measurement chord 
if (mode eq 'kevin' or mode eq 'luc') then $
  chordint=(mean([prof0,prof1])-profmin)/2.+profmin    ; FWHM
if (mode eq 'frac') then chordint= $
  chordval*(max(meanprof[*,1])-min(meanprof[*,1]))+min(meanprof[*,1]) 
if (mode eq 'chord') then chordint=max([prof0,prof1])

; check limits  ;?? needed?
if (chordint gt profmax) then chordint=profmax
if (chordint lt profmin) then chordint=profmin

; measure FWHM chord end wavelengths
;; chord0=interpol(delwav[0:iwmin-1],prof[0:iwmin-1],chordint)
;; chord1=interpol(delwav[iwmin+1:*],prof[iwmin+1:*],chordint) 
;RR OOPS above was used by Luc but interpol extrapolates and misfires in hump
;RR instead use "lambdameter drop"
for iw=0,iwmin-2 do begin
  iwabove=iw
  if (prof[iw+1] lt chordint) then break
endfor
chord0=interpol(delwav[iwabove:iwabove+1],prof[iwabove:iwabove+1],chordint)
for iw=nwav-1,iwmin+2,-1 do begin
  iwabove=iw
  if (prof[iw-1] lt chordint) then break
endfor
chord1=interpol(delwav[iwabove-1:iwabove],prof[iwabove-1:iwabove],chordint)

; assign results
chordwidth=chord1-chord0
chordshift=chord0+(chord1-chord0)/2.

; flag funny excesses (needed??)
if (finite(chordwidth) eq 0 or chordwidth gt wavrange) then begin
;; print,' ##### chordint,profmin,profmax ='+$
;;    trimd(chordint)+trimd(profmin)+trimd(profmax)
  minshift=-1000
  chordwidth=-1000
  chordshift=-1000
  return
endif 

; show = add to plot  
if (show ne 0) then begin
  title='chordwidth = '+trimd(chordwidth)+$
    '     midpoint chordshift = '+trimd(chordshift)+$
    '     mode = '+mode
  xyouts,(!x.window[1]-!x.window[0])/2.+!x.window[0],$
    !y.window[1]+.01,title,alignment=0.5,/normal,charsize=1.5
  plots,delwav[iwmin],profmin,psym=6,symsize=2  ; psym 6 square
  plots,delwav[iwmax],profmax,psym=6,symsize=2
  plots,chord0,chordint,psym=2,symsize=2  ; psym 2 asterisk
  plots,chord1,chordint,psym=2,symsize=2
  plots,[chord0,chord1],[1,1]+chordint,li=1
  plots,[1,1]*chordshift,[!y.crange[0],chordint],li=1
endif

end

; ========================== try per Hyper-C ==============================

cd,'/media/rutten/SSTDATA/alldata/SST/2014-06-21-quiet/crispex'

crispfile='crispex.6563.08:02:31.time_corrected.aligned.icube'
spectfile='spectfile.6563.idlsave'
restore,spectfile
delwav=spect_pos-float(6563) ; NB: Halpha_lc ad-hoc defined at 6563.000

;; crispfile='crispex.8542.08:02:31.time_corrected.icube'
;; spectfile='spectfile.8542.idlsave'
;; restore,spectfile
;; delwav=spect_pos-float(8542) ; NB: Halpha_lc ad-hoc defined at 6563.000

nwav=n_elements(delwav)
prof=intarr(nwav)
trange=[88,88] ; contrail precursor
for iwav=0,nwav-1 do begin
  im=readcrispfile(crispfile,iwav,nwav,trange=trange)
;;  prof[iwav]=im[582,512]    ;; Ha demo = very shifted+wide precursor pixel
;;  prof[iwav]=im[370,375]    ;; Ca  hump in core
  prof[iwav]=im[443,572] ;; Ha bottom contrail
;;  prof[iwav]=im[406,496]    ;; funny black-in-white pixel outer wing
;; prof[iwav]=im[410,560]    ;; weird 8542 profile 
;; prof[iwav]=im[16,33]    ;; edge
endfor

; get meanprof for 'frac'
meanprof=crisp2meanprof(crispfile,spectfile,istokes=0)

chordval=0.9
;; mode='luc'
mode='kevin'
;; mode='chord'
;; mode='frac'
prof2special,delwav,prof,chordval,mode,meanprof,$
  minint,minshift,chordshift,chordwidth,endmin,endmax,enddif,enddop,show=1

print,' ----- minint ='+trimd(minint)+$
  '  minshift ='+trimd(minshift)+$
  '  chordshift ='+trimd(chordshift)+$
  '  chordwidth ='+trimd(chordwidth)+$
  '  endmin ='+trimd(endmin)+$
  '  endmax ='+trimd(endmax)+$
  '  enddif ='+trimd(enddif)+$
  '  enddop ='+trimd(enddop)

end
