; file: mwseq2special.pro
; init: Oct  7 2017  Rob Rutten  Deil from Luc's lp_halpha_widths_cube.pro
; last: Apr  7 2018  Rob Rutten  Deil
; site: rridl/sstlib
; note: very slow, typically 2-5 hours
; todo: speed up by loop unrolling?  
;       more parameters?

;+

pro mwseq2special,infile,wavsfile,chordval,mode,$
  getminint=getminint,getshift=getshift,getwidth=getwidth,$
  getendmin=getendmin,getendmax=getendmax,getenddif=getenddif,$
  getenddop=getenddop,$
  trimbox=trimbox,trange=trange,outdir=outdir

; assocwrite profile parameter images for multi-wav sequence
;
; DESCRIPTION:
;   Adaptation of Luc Rouppe van der Voort's "lp_halpha_widths_cube.pro".
;   For explanation of the different measurement modes see the 
;     introductory comment in "prof2special.pro" in sstlib directory,
;     e.g., by entering "sp,prof2special" in IDL.
;   Results in output files: 
;     minint = intensity of Kevin-style-fitted line core minimum
;     shift = offset from nominal line center of the midpoint of the 
;             FWHM measurement chord (in 'kevin' mode offset of central dip)
;     width = length of FHM measurement chord 
;     endmin = minimum of measured intensities at core-cut ends 
;     endmax = maximum of measured intensities at core-cut ends 
;     enddif = red-blue difference of these intensities
;     enddop = Dopplergram (red-blue)/(red+blue) of these intensities
;   Combine into sum/differences etc with sstlib/combinespecial.pro
;
; MANDATORY INPUTS:
;   infile = string, input crispex file in LP format or IBIS fits file
;   wavsfile = string, file with wavelengths (crispex save or IBIS fits)
;   chordval = value defining the measurement chord (may differ from samples)
;   mode = 'kevin', 'luc', 'chord', 'frac': measurement mode 
;
; OUTPUT CHOICES (set 1 or 0)
;   getminint = getminint
;   getshift = getshift
;   getwidth = getwidth
;   getendmin = getendmin
;   getendmax = getendmax
;   getenddif = getenddif 
;   getenddop = getenddop
;
; OPTIONAL KEYWORD INPUTS
;   trimbox = [xmin,ymin,ymin,ymax] slection box
;   trange = [itstart,itend] option to limit time range for quicker tests 
;   outdir = subdir/ to put output file in (default 'sst/'), e.g. 'trials/'
;
; OUTPUTS:
;   outdir/files in (x,y,t*3) containing time sequences
;     of the output maps.  First two in units 0.1 mA. 
;     filenames: 
;         'minint_mode_lineident.filetype',
;         'shift_mode_lineident.filetype',
;          etcetera 
;       with mode the specified one, lineident taken from the infile,
;       filetype = fits
;       For example file "sst/shift_chord_6563.fits".
;
; RESTRICTIONS:
;   exceedingly slow due to explicit triple loop over ix,iy,it calling 
;   prof2special.pro; therefore intermediate progress printout.
;   Beter run in off-line server (UU) (?? how to unloop this?).
;
; HISTORY:
;   xxx xx 2016 Luc Rouppe van der Voort: "lp_halpha_widths_cube.pro"
;   Oct 15 2017 RR: ruttenization, product, more modes
;   Mar 20 2018 RR: also for DST/IBIS 
;   Mar 23 2018 RR: shxwi out, into combinespecial.pro
;   Apr  2 2018 RR: emoved loop multiple chords, only fits output
;-

; answer no-parameter query 
if (n_params(0) lt 4) then begin
  sp,mwseq2special
  return
endif

; set wall-clock timer (seconds)
timestartspecial=systime(1) 

; keyword defaults
if (n_elements(getminint) eq 0) then getminint=0
if (n_elements(getshift) eq 0) then getshift=0
if (n_elements(getwidth) eq 0) then getwidth=0
if (n_elements(getendmin) eq 0) then getendmin=0
if (n_elements(getendmax) eq 0) then getendmax=0
if (n_elements(getenddif) eq 0) then getenddif=0
if (n_elements(getenddop) eq 0) then getenddop=0
if (n_elements(trimbox) eq 0) then trimbox=-1
if (n_elements(trimbox) eq 0) then trimbox=-1
if (n_elements(trange) eq 0) then trange=[0,-1]
if (n_elements(outdir) eq 0) then outdir='./'

; set endian for this computer 
;RR l = little, b = big; fits files are big; linux machines are little
if ((byte(1L, 0, 1))[0] eq 1) then endian = 'l' else endian='b'

; check on output
if (getminint+getshift+getwidth+getendmin+getendmax+$
    getenddif+getenddop eq 0) then begin
  print,' ##### mwseq2special abort: no output requested'
  return
endif

; check infile
if (not(strmatch(infile,'*crispex*') or strmatch(infile,'*mwseq*'))) $
then begin
  print,' ##### mwseq2special abort: no crispex or mwseq in filename'
  return
endif

; check on mode
if (mode ne 'kevin' and mode ne 'luc' $
    and mode ne 'chord' and mode ne 'frac') then begin
  print,' ##### prof_shift_width abort: unknown mode'
  sp,mwseq2special
  return
endif

; check not kevin and Doppler
if (mode eq 'kevin' and getenddop) then $
  print,' ===== mwseq2special WARNING: Dopplergram bad for mode kevin'

; get file extension
dummy=cgrootname(infile,extension=ext_in)  ;RR needs coyotelib

; get line (= wavelength) from file name
line=-1
if (strmatch(infile,'*6563*')) then line=6563
if (strmatch(infile,'*8542*')) then line=8542
if (line eq -1) then begin
  print, ' ##### mwseq2special abort: no known line in infile name'
  return
endif

; get profile sampling wavelength offsets and mean profile
if (ext_in eq 'icube') then begin
  restore,wavsfile   
  nw=n_elements(spect_pos)     ; number of wavelengths
  delwav=spect_pos-float(line) ; NB: eg Halpha_lc ad-hoc defined at 6563.000
  delwav=float(delwav)
  meanprof=crisp2meanprof(infile,wavsfile,istokes=0)
endif
if (ext_in eq 'fits') then begin
  meanprof=ibis2meanprof(ibisfile,wavsfile,ibisdir='./')
  delwav=meanprof[*,0]
endif

; read infile header to get data type and cube dimensions
if (ext_in eq 'icube') then begin
  crispex_read_header,infile,header=header,datatype=datatype, $
    dims=dims,nx=nx,ny=ny,nt=ntdat,endian=endian_file,$
    stokes=stokes, ns=ns, diagnostics=diagnostics
  endian_in='l'  ; La Palma 
  inheadersize=512
  nt_in=fix(0.1+ntdat/nw/ns)    ;RR get the real nt in case of small mistake
endif
if (ext_in eq 'fits') then begin
  inheader=headfits_rr(ibisdir+ibisfile)
  inheadersize=(1+fix(n_elements(header)/36.))*2880
  bitpix=fxpar(inheader,'bitpix')
  nx=fxpar(inheader,'naxis1') 
  ny=fxpar(inheader,'naxis2') 
  nz_file=fxpar(inheader,'naxis3') 
  nt_in=nz_file
  nw=fxpar(inheader,'nrwavs')  ; becomes 0 if not present
  if (nw eq 0) then nw=1 else nt_in=nz_file/nw
  endian_in='b'  ; fits files are always big endian
; check
  if (nx eq 0 or ny eq 0 or nz_file eq 0) then begin
    print,' ##### reformcubefile abort: no nx, ny, nz in fits header '+infile
    return
  endif
endif

; default trange 
if (trange[1] eq -1) then trange=[0,nt_in-1]
nt_out=trange[1]-trange[0]+1

; default trimbox = full image 
if (trimbox[0] eq -1) then trimbox=[0,0,nx-1,ny-1]

; define output files
ext_out='.fits'
strchord='_'+trim(chordval)
minintfile='minint_'+mode+'_'+trim(line)+strchord+ext_out
shiftfile='shift_'+mode+'_'+trim(line)+strchord+ext_out
widthfile='width_'+mode+'_'+trim(line)+strchord+ext_out
endminfile='endmin_'+mode+'_'+trim(line)+strchord+ext_out
endmaxfile='endmax_'+mode+'_'+trim(line)+strchord+ext_out
enddiffile='enddif_'+mode+'_'+trim(line)+strchord+ext_out
enddopfile='enddop_'+mode+'_'+trim(line)+strchord+ext_out

; define assoc input 
openr,lu_in,infile,/get_lun,swap_endian=(endian ne endian_in)
assoc_in=assoc(lu_in,intarr(nx,ny,nw),inheadersize)

; outheader and endian
mkhdr,outheader,2,[nx,ny,nt_out] 
sizeheader_out=size(outheader) 
sizeoutheader=(1+fix(sizeheader_out[1]/36.))*2880
  ;RR fits header = Nx36 "card images" = Nx2880 bytes
endian_out='b'

; open assocs for the various outputs 
; file minintmaps
if (getminint) then begin 
  openw,lu_minint,outdir+minintfile,/get_lun,/swap_if_little_endian 
  assoc_minint=assoc(lu_minint,bytarr(sizeoutheader))
  assoc_minint[0]=byte(outheader)
  assoc_minint=assoc(lu_minint,intarr(nx,ny),sizeoutheader)
endif
; file shiftmaps
if (getshift) then begin
  openw,lu_shift,outdir+shiftfile,/get_lun,/swap_if_little_endian
  assoc_shift=assoc(lu_shift,bytarr(sizeoutheader))
  assoc_shift[0]=byte(outheader)
  assoc_shift=assoc(lu_shift,intarr(nx,ny),sizeoutheader)
endif
; file widthmaps
if (getwidth) then begin
  openw,lu_width,outdir+widthfile,/get_lun,/swap_if_little_endian
  assoc_width=assoc(lu_width,bytarr(sizeoutheader))
  assoc_width[0]=byte(outheader)
  assoc_width=assoc(lu_width,intarr(nx,ny),sizeoutheader)
endif
; file endminmaps
if (getendmin) then begin 
  openw,lu_endmin,outdir+endminfile,/get_lun,/swap_if_little_endian
  assoc_endmin=assoc(lu_endmin,bytarr(sizeoutheader))
  assoc_endmin[0]=byte(outheader)
  assoc_endmin=assoc(lu_endmin,intarr(nx,ny),sizeoutheader)
endif
; file endmaxmaps
if (getendmax) then begin 
  openw,lu_endmax,outdir+endmaxfile,/get_lun,/swap_if_little_endian
  assoc_endmax=assoc(lu_endmax,bytarr(sizeoutheader))
  assoc_endmax[0]=byte(outheader)
  assoc_endmax=assoc(lu_endmax,intarr(nx,ny),sizeoutheader)
endif
; file enddifmaps
if (getenddif) then begin 
  openw,lu_enddif,outdir+enddiffile,/get_lun,/swap_if_little_endian
  assoc_enddif=assoc(lu_enddif,bytarr(sizeoutheader))
  assoc_enddif[0]=byte(outheader)
  assoc_enddif=assoc(lu_enddif,intarr(nx,ny),sizeoutheader)
endif
; file enddopmaps
if (getenddop) then begin 
  openw,lu_enddop,outdir+enddopfile,/get_lun,/swap_if_little_endian
  assoc_enddop=assoc(lu_enddop,bytarr(sizeoutheader))
  assoc_enddop[0]=byte(outheader)
  assoc_enddop=assoc(lu_enddop,intarr(nx,ny),sizeoutheader)
endif

; print pacifier
print,' ----- mwseq2special starts with chordval '+trim(chordval)

; do the measurement (explicit triple loop over [x,y,t] so bloody slow...)
; how to unroll the loop over x,y and use IDL array processing?
minintmap=fltarr(nx,ny)
widthmap=fltarr(nx,ny)
shiftmap=fltarr(nx,ny)
endminmap=fltarr(nx,ny)
endmaxmap=fltarr(nx,ny)
enddifmap=fltarr(nx,ny)
enddopmap=fltarr(nx,ny)
for it=trange[0],trange[1] do begin
  cube=assoc_in[it]    ; (x,y,wav) wavelength scan
  for ix=0, nx-1 do begin
    for iy=0, ny-1 do begin
      prof=float(reform(cube[ix,iy,*]))       ; profile at (ix,iy,it) 
      prof2special,delwav,prof,chordval,mode,meanprof,$
        minint,minshift,chordshift,chordwidth,endmin,endmax,enddif,enddop
      minintmap[ix,iy]=minint 
      if (mode eq 'kevin') then shiftmap[ix,iy]=minshift else $
        shiftmap[ix,iy]=chordshift 
      widthmap[ix,iy]=chordwidth
      endminmap[ix,iy]=endmin
      endmaxmap[ix,iy]=endmax
      enddifmap[ix,iy]=enddif
      enddopmap[ix,iy]=enddop
    endfor
  endfor

; still per it: replace shift and width bad pixels flagged by prof2special.pro
  zero=where(widthmap eq -1000 and shiftmap eq -1000)
  nrbadpx=0
  if (zero[0] ne -1) then begin  ;RR bloody IDL
    nrbadpx=n_elements(zero)
    shiftmap[zero]=0             ; mean value presumably zero 
    widthmap[zero]=avg(widthmap>0)  
  endif
; histo_opt to take off freak pixels upsetting movex etc greyscaling
  shiftmap=histo_opt_rr(shiftmap,1E-4)   ; NB!
  widthmap=histo_opt_rr(widthmap,1E-4)   ; NB!

; write all for this time step as integers into the output file
  if (getminint) then $
    assoc_minint[(it-trange[0])]=fix(round(minintmap))
  if (getshift) then $
    assoc_shift[(it-trange[0])]=fix(round(shiftmap*10000))
  if (getwidth) then $
    assoc_width[(it-trange[0])]=fix(round(widthmap*10000))
  if (getendmin) then $
    assoc_endmin[(it-trange[0])]=fix(round(endminmap))
  if (getendmax) then $
    assoc_endmax[(it-trange[0])]=fix(round(endmaxmap))
  if (getenddif) then $
    assoc_enddif[(it-trange[0])]=fix(round(enddifmap))
  if (getenddop) then $
    assoc_enddop[(it-trange[0])]=fix(round(enddopmap*10000))

; this time step done
  print,' ----- done (it/range):'$
    +trimd(it)+'/['+trim(trange[0])+','+trim(trange[1])+']'
  if (nrbadpx gt 0) then print,'       nr bad px = '+trim(nrbadpx)+' = '+$
    trim(100.*nrbadpx/(nx*ny),1)+' percent' 

endfor  ; end of loop over time steps

; all done 
close,/all
if (getshift) then print,' ===== mwseq2special wrote: '+outdir+shiftfile
if (getwidth) then print,' ===== mwseq2special wrote: '+outdir+widthfile
if (getminint) then print,' ===== mwseq2special wrote: '+outdir+minintfile
if (getendmin) then print,' ===== mwseq2special wrote: '+outdir+endminfile
if (getendmax) then print,' ===== mwseq2special wrote: '+outdir+endmaxfile
if (getenddif) then print,' ===== mwseq2special wrote: '+outdir+enddiffile
if (getenddop) then print,' ===== mwseq2special wrote: '+outdir+enddopfile

; print elapsed wall-clock time because this is so slow
timedone=systime(1)
timelaps=ntostr((timedone-timestartspecial)/60.,format='(F11.1)') 
print,' ===== mwseq2special took '+timelaps+' minutes'

end

; ============== trial per Hyper-C ================================

cd,'/media/rutten/SSTDATA/alldata/SST/2014-06-21-quiet'
outdir='constructs/'
line='ca'
line='ha'
if (line eq 'ha') then begin
  infile='sst_iris_sdo/crispex.6563.08:02:31.time_corrected.aligned.icube'
  wavsfile='crispex/spectfile.6563.idlsave' 
  chordval=0.9
endif
if (line eq 'ca') then begin
  infile='sst_iris_sdo/crispex.8542.08:02:31.time_corrected.icube'
  wavsfile='crispex/spectfile.8542.idlsave'
  chordval=0.6
endif
trimbox=[50,50,860,860]

; trange?
;; trange=[88,90]  ; ONLY contrail precursor
;; trange=[3,7] ; bad pixels in it=3 
;; trange=[0,2]
;; trange=[80,110]

; chordval?
;; chordval=[0.8,0.7,0.6,0.5]
;; chordval=1.0
;; chordval=0.05
;; chordval=0.2

; mode?
mode='luc'
;; mode='chord' 
;; mode='frac'
;; mode='kevin'

; outputs?
getminint=0
getshift=0
getwidth=0
getendmin=0
getendmax=0
getenddif=0
getenddop=1

mwseq2special,infile,wavsfile,chordval,mode,$
   getminint=getminint,getshift=getshift,getwidth=getwidth,$
   getendmin=getendmin,getendmax=getendmax,getenddif=getenddif,$
   getenddop=getenddop,$
   outdir=outdir,trimbox=trimbox,trange=trange

; inspect 
showex,outdir+'minint_kevin_8542_0.6.fits'


end



