; file: mapsav2crisp.pro
; init: Oct 24 2017  Rob Rutten  Deil
; last: Oct 24 2017  Rob Rutten  Deil

;+

pro mapsav2crisp,savdir,crispfile,trange=trange,verbose=verbose

 ; PURPOSE:
 ;   assoc-write many singe-time-step .sav files with SST/CRISP
 ;   spectral-scan cubes back into a single LP-format crisp file   
 ;   
 ; CALL:
 ;   see above
 ;   
 ; INPUTS:
 ;   savdir = string path/ to dir with the .sav files
 ;   crispfile = string with output file, must have 'icube' extension
 ;
 ; OPTIONAL KEYWORDS INPUTS:
 ;   trange=[t0,t1]: for shorter tests
 ;   verbose=1/0: print progress
 ; 
 ; OUTPUTS:
 ;   crispfile 
 ;   
 ; RESTRICTIONS:
 ;   supposes that the intensity values are fit to be fixed back to integer
 ;   (probably they are since data counts to begin with)
 ; 
 ; HISTORY:
 ;   Oct 24 2017 RR: for Huang & Madjarska SST2IRIS data  
;-

; answer no-parameter query 
if (n_params(0) lt 2) then begin
  print,'mapsav2crisp,savdir,crispfile,trange=trange,verbose=verbose'
  return
endif

; set wall-clock timer (seconds)
timestart=systime(1) 

; defaults for keywords
if (n_elements(trange) eq 0) then trange=[-1,-1]
if (n_elements(verbose) eq 0) then verbose=0

; check that crispfile has .icube  extension 
dummy=cgrootname(crispfile,extension=ext)  ;RR needs coyotelib
if not(ext eq 'icube') then begin
  print,' ##### mapsav2crisp abort: crispfile does  not end with .icube'
  return
endif

; find all .sav files
infiles=findfile(savdir+'/*.sav')

; set trange
nt=n_elements(infiles)
if (trange[0] eq -1) then trange=[0,nt-1]

; get dimensions
restore,infiles[0]
im0=spmap.data
sizeim=size(im0)
nx=sizeim[1]
ny=sizeim[2]
nw=sizeim[3]

; check whether map intensity range fits conversion to integer
minim0=min(im0)
maxim0=max(im0)
if (minim0 lt -2000 or maxim0 gt 20000 or maxim0 lt 1000) then begin
  print,' ##### mapsav2crisp abort: image 0 range not fit to be made integer'
  print,' @@@@@ now you have to do something!'
  return
endif

; set header_out
nt_out=trange[1]-trange[0]+1
header_out=make_lp_header(fix(im0),nt=nt_out*nw)
header_out=str_replace(header_out,' datatype',$
                       'stokes=[I], ns=1 :  datatype')
header_outsize=512 ; LP standard

; open output file for assoc (little now; swap for others than my intel) 
get_lun, unit_out
openw,unit_out,crispfile,/swap_if_big_endian 
outassoc=assoc(unit_out,intarr(nx,ny),header_outsize)

; write header_out
if (header_outsize ne 0) then begin
  rec=assoc(unit_out, bytarr(header_outsize))
  rec[0]=byte(header_out)
endif

; write images per sav file iw-distributedly into the output file
for it=trange[0],trange[1] do begin
  restore,infiles[it]
  imcube=spmap.data
  for iw=0,nw-1 do begin
    iwit=it*nw+iw
    outassoc[iwit]=imcube[*,*,iw]
  endfor
  if (verbose) then print,' ----- wrote nw images for it ='+trimd(it) 
endfor

; free the output file
free_lun,unit_out

; print elapsed time
timedone=systime(1)
timelaps=ntostr((timedone-timestart)/60.,format='(F11.1)')
if (verbose ne 0) then print,' ===== mapsav2crisp took '+timelaps+' minutes'

end


; =============== main for testing per IDLWAVE H-c ======================

savdir='/media/rutten/SSTDATA/alldata/SST/2014-06-07-maria/sstB2iris_zh/'
crispfile=savdir+'crispex_sstB_6563_2iris_zh.icube'

;;; trange=[0,10]  ; quickie check
mapsav2crisp,savdir,crispfile,trange=trange,/verbose

movex,crispfile,ntfile=trange[1]-trange[0]+1

end


