; file: lp2fits.pro = rewrite classical SST "lp" cube as fitscube
; init: Aug 31 2014  Rob Rutten  Deil
; last: Aug 31 2014  Rob Rutten  Deil
; note: also works for crispex files (except for setting trange: wav-folded)

pro lp2fits,infile,outfile,$
  xrange=xrange,yrange=yrange,trange=trange,$
  byt=byt,nxout=nxout,nyout=nyout,ntout=ntout

;+
 ; NAME:
 ;   lp2fits
 ; PURPOSE:
 ;   write [partial] SST "LP" cube file as fits file
 ; CALL:
 ;   lp2fits,infile,outfile,$
 ;     xrange=xrange,yrange=yrange,trange=trange,$
 ;     byt=byt,nxout=nxout,nyout=nyout,ntout=ntout
 ; INPUTS:
 ;   infile = string 'path/crispfile'
 ;   outfile = string 'path/outfile.fits' 
 ; OPTIONAL INPUTS:
 ;   xrange=xrange,yrange=yrange,trange=trange: limit ranges  
 ;   byt=byt: bytscale each image
 ; OUTPUTS:
 ;   fits file [x,y,t] 
 ; OPTIONAL OUTPUTS:
 ;   nxout, nyout, ntout: output cube dimensions for further use
 ; METHOD:
 ;   uses assoc to permit large files
 ; HISTORY:
 ;   Aug 31 2014 RR: start, copied fom crisp2fits.pro   
;-

; answer no-parameter query
if (n_params() lt 2) then begin
  print,' lp2fits,infile,$'
  print,'  [xrange=xrange,yrange=yrange,trange=trange,$'
  print,'   byt=byt,nxout=nxout,nyout=nyout,ntout=ntout]'
  return
endif

; default keywords
if (n_elements(byt) eq 0) then byt=0
if (n_elements(xrange) eq 0) then xrange=[0,-1]
if (n_elements(yrange) eq 0) then yrange=[0,-1]
if (n_elements(trange) eq 0) then trange=[0,-1]

; set endian
bigendian=1

; read the file header to get data type and cube dimensions
lp_readheader,infile,header=inheader,datatype=datatype, $
  dims=dims,nx=nxin,ny=nyin,nt=ntin,endian=endian_file

; define input assoc size and type
if (datatype eq 1) then inarr=bytarr(nxin,nyin)
if (datatype eq 2) then inarr=intarr(nxin,nyin)
if (datatype eq 4) then inarr=fltarr(nxin,nyin)

; define output assoc size and type
if (xrange[1] eq -1) then xrange[1]=nxin-1
if (yrange[1] eq -1) then yrange[1]=nyin-1
if (trange[1] eq -1) then trange[1]=ntin-1
nxout=xrange[1]-xrange[0]+1
nyout=yrange[1]-yrange[0]+1
ntout=trange[1]-trange[0]+1

; open infile for assoc
get_lun, unit_in
openr,unit_in,infile
inassoc=assoc(unit_in,inarr,512)

; output header
if (datatype eq 1) then bitpix=8
if (datatype eq 2) then bitpix=16
if (datatype eq 4) then bitpix=-32
mkhdr,outheader,abs(bitpix)/8,[nxout,nyout,ntout]
sizeoutheader=size(outheader)  
; fits header = Nx36 "card images" = Nx2880 bytes
outheadersize=(1+fix(sizeoutheader[1]/36.))*2880

; open outfile for assoc
get_lun, unit_out
if (bigendian) then openw,unit_out,outfile,/swap_if_little_endian $
else openw,unit_out,outfile
if (bitpix eq -32) then outassoc=assoc(unit_out,fltarr(nxout,nyout),$
  outheadersize)
if (bitpix eq 16) then outassoc=assoc(unit_out,intarr(nxout,nyout),$
  outheadersize)
if (bitpix eq 8) then outassoc=assoc(unit_out,bytarr(nxout,nyout),$
  outheadersize)
if (outheadersize ne 0) then begin
  rec=assoc(unit_out, bytarr(outheadersize))
  rec[0]=byte(outheader)
endif

; loop over images
  
for it=trange[0],trange[1] do begin
  inimage=inassoc[it]
  if (byt eq 1) then inimage=bytscl(inimage)
  outassoc[it-trange[0]]=inimage[xrange[0]:xrange[1],yrange[0]:yrange[1]]
endfor

free_lun,unit_in,unit_out

; completed
print, ' ===== wrote '+outfile

end

; =============== test per IDLWAVE S-c ==============================

path='/home/rutten/data/SST/2016-09-05-demo/sst/'
infile=path+'crispex.6563.09:48:31.time_corrected.aligned.icube'
outfile=path+'trial.fits'
lp2fits,infile,outfile
showex,outfile 

end
