; file: crisp2meanprof
; init: Oct 14 2017  Rob Rutten  Deil
; last: Mar 16 2018  Rob Rutten  Deil
; site: rridl/sstlib

;+

function crisp2meanprof,crispfile,spectsavfile,istokes=istokes,$ 
  itstep=itstep

 ; PURPOSE:
 ;   return mean profile of a CRISP LP file using assoc
 ; CALL:
 ;   see above
 ; INPUTS:
 ;   crispfile = string with CRISP dir/filename
 ;   spectsavfile = corresponding IDL save file with wvelengths
 ; OPTIONAL INPUTS:
 ;   istokes = istokes: [I,Q,U,V] (istokes=0 selects intensity)
 ;   itstep = step time steps, > 1 speeds up this program (default 10)
 ; OUTPUTS:
 ;   meanprof = double array delwav, intensity
 ; HISTORY:
 ;   Oct 14 2017 RR: restart (couldn't find earlier version I had?)
;-

; default keywords
if (n_elements(istokes) eq 0) then istokes=0
if (n_elements(itstep) eq 0) then itstep=10

; get line (= wavelength) from file name
line=-1
if (strmatch(crispfile,'*crispex.6563*')) then line=6563
if (strmatch(crispfile,'*halpha*')) then line=6563
if (strmatch(crispfile,'*crispex.8542*')) then line=8542
if (line eq -1) then begin
  print, ' ##### crisp2meanprof abort: no known line in crispfile'
  return,0
endif

; get profile sampling wavelength offsets
restore,spectsavfile   
nw=n_elements(spect_pos)     ; number of wavelengths
delwav=spect_pos-float(line) ; NB: Halpha_lc ad-hoc defined at 6563.000
delwav=float(delwav)

; read the file header to get data type and cube dimensions
crispex_read_header,crispfile,header=header,datatype=datatype, $
  dims=dims,nx=nx,ny=ny,nt=ntdat,endian=endian_file,$
  stokes=stokes, ns=ns, diagnostics=diagnostics
ntdat=fix(0.1+ntdat/nw) ;RR get the real nt in case of small mistake
nt=ntdat/ns  ;RR 4 Stokes samplings if Stokes

; check on Stokes clash
if (istokes ne 0 and ns eq 1) then istokes=0

; define assoc input size and type
if (datatype eq 1) then imscan=bytarr(nx,ny,nw)
if (datatype eq 2) then imscan=intarr(nx,ny,nw)
if (datatype eq 4) then imscan=fltarr(nx,ny,nw)

; open assoc to the input images 
get_lun, unit
openr,unit,crispfile
p=assoc(unit,imscan,512) 

; do the work

sumprof=fltarr(nw)
for it=0,nt-1,itstep do begin
;RR crispex FAQ answer: image=image_cube[*,*,t*nlp*ns+s*nlp+lp]
  scan=p[it*ns+istokes*nw]
  prof=total(total(scan,1),1)/(float(nx)*ny)
  sumprof=sumprof+prof
endfor
meanprof=sumprof/fix(nt/itstep+1)

; ready with reading
free_lun,unit

; done
return,[[delwav],[meanprof]]

end

; =============== test per IDLWAVE S-c ==============================

cd,'/media/rutten/SSTDATA/alldata/SST/2014-06-21-quiet/'
crispfile='crispex/crispex.6563.08:02:31.time_corrected.aligned.icube'
spectsavfile='crispex/spectfile.6563.idlsave'

mp=crisp2meanprof(crispfile,spectsavfile,istokes=0)
plot,mp[*,0],mp[*,1]

end
