; file: crisp2fits.pro for <old> LP files from SST/CRISP
; init: Aug 12 2014  Rob Rutten  Oslo
; last: Feb  2 2021  Rob Rutten  Deil
; note: requires knowledge of spectral samplings: total number
;       (parameter nwav) and selected wavelength (parameter iwav). Get
;       them from lp_readheader,filename or from running showex or crispex


;+
pro crisp2fits,infile,outfile,iwav,nwav,$
  istokes=istokes,$
  filterwidth=filterwidth,limbshift=limbshift,spectfile=spectfile,$
  xrange=xrange,yrange=yrange,trange=trange,$
  byt=byt,nxout=nxout,nyout=nyout,ntout=ntout

 ;   write [partial] SST crisp cube file as fits file
 ;
 ; INPUTS:
 ;   infile = string 'path/crispfile'
 ;   outfile = string 'path/outfile.fits' 
 ;   iwav = index selected wavelength
 ;     may also be 2-elem vector defining summation range
 ;     with /byt each then gets bytescaled first
 ;     not used for filterwidth summing
 ;   nwav = total nr of sampling wavelengths (necessary = lack header info)
 ;
 ; OPTIONAL KEYWORD INPUTS:
 ;   istokes=istokes: [I,Q,U,V], so istokes=0 selects intensity
 ;   filterwidth: sum over Gaussian filter with filterwidth FWHM in Angstrom
 ;   limbshift: 0.044 AA at limb; GONG filter doesn't follow (SST does)
 ;   spectfile: SST-style save file with dlamda's  for filterwidth
 ;   xrange=xrange,yrange=yrange,trange=trange: limit cube ranges  
 ;   byt=byt: bytscale each image, also before summation
 ;
 ; OUTPUTS:
 ;   fits file [x,y,t] for selected wavelength(s)
 ;
 ; OPTIONAL OUTPUTS:
 ;   nxout, nyout, ntout: output cube dimensions for further use
 ; 
 ; HISTORY:
 ;   Aug 12 2014 RR: start   
 ;   Sep  6 2014 RR: range of iwav's to sum over
 ;   Feb  2 2021 RR: filterwidth, limbshift (to mimic GONG)
;-

; answer no-parameter query
if (n_params() lt 4) then begin
  sp,crisp2fits
  return
endif

; default keywords
if (n_elements(byt) eq 0) then byt=0
if (n_elements(istokes) eq 0) then istokes=0
if (n_elements(filterwidth) eq 0) then filterwidth=0
if (n_elements(limbshift) eq 0) then limbshift=0
if (n_elements(spectfile) eq 0) then spectfile=''
if (n_elements(xrange) eq 0) then xrange=[0,-1]
if (n_elements(yrange) eq 0) then yrange=[0,-1]
if (n_elements(trange) eq 0) then trange=[0,-1]

; set endian
bigendian=1

; read the file header to get data type and cube dimensions
;RR OOPS there is no specification of the number of wavelength samples
crispex_read_header,infile,header=inheader,datatype=datatype, $
  dims=dims,nx=nxin,ny=nyin,nt=ntin,endian=endian_file,$
  stokes=stokes, ns=ns, diagnostics=diagnostics
ntin=fix(0.1+ntin/nwav) ;RR get the real nt in case of small mistake
ntin=ntin/ns            ;RR 4 Stokes samplings if Stokes

; check on Stokes clash
if (istokes ne 0 and ns eq 1) then istokes=0

; set iwav to all for filterwidth and get dlambda's
nwavrange=n_elements(iwav)
if (filterwidth ne 0) then begin
  nwavrange=2  
  iwav=[0,nwav-1]   ; multiply whole measured profile for summing
  if (spectfile eq '') then begin
    print,' ##### crisp2fits abort: filterwidth needs spectfile'
    return
  endif
  restore,spectfile
  if (n_elements(spect_pos) ne nwav) then begin
    print,' ##### crisp2fits abort: nr spect_pos not equal nwav'
    return
  endif
  species='null'
  if strmatch(spectfile,'*6563*') then species='Ha'
  if strmatch(spectfile,'*8542*') then species='CaIR'
  if strmatch(spectfile,'*3968*') then species='CaH'
  if strmatch(spectfile,'*6302*') then species='Fe'
  if strmatch(spectfile,'*10830*') then species='He'
  if (species eq 'null') then begin
    print,' ##### crisp2fits abort: need spectral species'
    return
  endif
  if (species eq 'Ha') then $
    iwlc=where(abs(spect_pos-6563.0) lt 0.005)
  if (species eq 'CaIR') then $
    iwlc=where(abs(spect_pos-8542.0) lt 0.005)
  if (species eq 'Fe') then $
    iwlc=where(abs(spect_pos-6302.0) lt 0.005)
  if (species eq 'CaK') then $
    iwlc=where(abs(spect_pos-3933.7) lt 0.005)
  if (species eq 'CaH') then $
    iwlc=where(abs(spect_pos-3968.5) lt 0.005)
endif

; check iwav: single or multiple or all?
if (nwavrange gt 2) then begin
  print,' ##### crisp2fits abort: iwav not 1 or 2 elements'
  return
endif

; define input assoc size and type
if (datatype eq 1) then inarr=bytarr(nxin,nyin)
if (datatype eq 2) then inarr=intarr(nxin,nyin)
if (datatype eq 4) then inarr=fltarr(nxin,nyin)

; define output assoc size and type
if (xrange[1] eq -1) then xrange[1]=nxin-1
if (yrange[1] eq -1) then yrange[1]=nyin-1
if (trange[1] eq -1) then trange[1]=ntin-1
nxout=xrange[1]-xrange[0]+1
nyout=yrange[1]-yrange[0]+1
ntout=trange[1]-trange[0]+1

; open infile for assoc
; (cut immediately to reduce array size for too large data cubes)
get_lun, unit_in
openr,unit_in,infile
inassoc=assoc(unit_in,inarr,512)

; output header
if (datatype eq 1) then bitpix=8
if (datatype eq 2) then bitpix=16
if (datatype eq 4) then bitpix=-32
mkhdr,outheader,abs(bitpix)/8,[nxout,nyout,ntout]
sizeoutheader=size(outheader)  
; fits header = Nx36 "card images" = Nx2880 bytes
outheadersize=(1+fix(sizeoutheader[1]/36.))*2880

; @@@ add the keywords here needed for crispex etc

; open outfile for assoc
get_lun, unit_out
if (bigendian) then openw,unit_out,outfile,/swap_if_little_endian $
else openw,unit_out,outfile
if (bitpix eq -32) then outassoc=assoc(unit_out,fltarr(nxout,nyout),$
                                       outheadersize)
if (bitpix eq 16) then outassoc=assoc(unit_out,intarr(nxout,nyout),$
                                      outheadersize)
if (bitpix eq 8) then outassoc=assoc(unit_out,bytarr(nxxout,nyout),$
                                     outheadersize)
if (outheadersize ne 0) then begin
  rec=assoc(unit_out, bytarr(outheadersize))
  rec[0]=byte(outheader)
endif

; loop over images
;RR crispex cubes are 3D but actually have 4 or 5 dimensions
;RR crispex FAQ answer: image = image_cube[*,*,t*nlp*ns+s*nlp+lp]
for it=trange[0],trange[1] do begin

; single wavelength
  if (nwavrange eq 1) then begin
    image=inassoc[it*nwav*ns+istokes*nwav+iwav]
    if (byt eq 1) then image=bytscl(image)
  endif

; wavelength range
  if (nwavrange eq 2) then begin
    image=inassoc[it*nwav*ns+istokes*nwav]
    sumimage=image*0. ; must be float, set to zero
    for iw=iwav[0],iwav[1] do begin
      image=inassoc[it*nwav*ns+istokes*nwav+iw]
      if (filterwidth ne 0) then begin
        dx=spect_pos[iw]+limbshift-spect_pos[iwlc]  
        fracdx=normgauss(dx,filterwidth)
        fracdx=float(fracdx[0]) ; F**K IDL, gave array, why?
        image=image*fracdx
      endif
      sumimage=sumimage+image
    endfor

; summation done
    image=sumimage/(iwav[1]-iwav[0])
    if (datatype eq 1 or byt eq 1) then image=bytscl(image)
    if (datatype eq 2) then image=fix(image)
  endif

  outassoc[it-trange[0]]=image[xrange[0]:xrange[1],yrange[0]:yrange[1]]
endfor ; end loop over wav range

free_lun,unit_in,unit_out

end

; =============== test per IDLWAVE S-c ==============================

; standard test = small spot with dangler near center, nt=35 nwav=15
cd,'~/data/SST/2016-09-05-demo' 
infile='crispex/crispex.6563.09:48:31.time_corrected.aligned.icube'

; line center 
outfile1='/tmp/ha_lc.fits'
crisp2fits,infile,outfile1,7,15

; apply GONG filter
filterwidth=0.6 ; nominal value (believe it?)
limbshift=0.04 ; 2 km/s = 0.044

spectfile='spectfiles/spectfile.6563.idlsave'
outfile2='/tmp/ha_gong.fits'
crisp2fits,infile,outfile2,0,15,$
  filterwidth=filterwidth,limbshift=limbshift,spectfile=spectfile

; smear heavily
smearfile='/tmp/ha_gong_smear.fits'
reformcubefile,outfile2,smearfile,smear=100

; check result
showex,smearfile,outfile2,outfile1,/blink 

end
