; file: crisp2dopp.pro for <old> LP files from SST/CRISP
; init: Mar 11 2019  Rob Rutten  Deil from crisp2fits.pro
; last: Mar 12 2019  Rob Rutten  Deil
; note: requires desired Doppler iwavs and nwav, get e.g., per showex 

;+

pro crisp2dopp,infile,outfile,iwav1,iwav2,nwav,$
  xrange=xrange,yrange=yrange,trange=trange,$
  nxout=nxout,nyout=nyout,ntout=ntout

 ; NAME:
 ;   crisp2dopp
 ; PURPOSE:
 ;   write [partial] SST crisp Doppler cube file as fits file
 ; CALL:
 ;   see above
 ; INPUTS:
 ;   infile = string 'path/crispfile'
 ;   outfile = string 'path/outfile.fits'  
 ;   iwav1 = index selected Doppler wavelength
 ;   iwav2 = index selected Doppler wavelength
 ;   nwav = total nr of sampling wavelengths (necessary = lack header info)
 ; OPTIONAL INPUTS:
 ;   xrange=xrange,yrange=yrange,trange=trange: limit cube ranges  
 ; OUTPUTS:
 ;   fits file [x,y,t] Dopplergrams, integer, range +- 20000
 ; OPTIONAL OUTPUTS:
 ;   nxout, nyout, ntout: output cube dimensions for further use
 ; METHOD:
 ;   uses assoc to permit large files
 ; HISTORY:
 ;   Mar 11 2019 RR: start from copy of crisp2fits.pro
;-

; answer no-parameter query
if (n_params() lt 5) then begin
  print,' crisp2dopp,infile,outfile,iwav1,iwav2,nwav,$'
  print,'  [xrange=xrange,yrange=yrange,trange=trange,$'
  print,'   nxout=nxout,nyout=nyout,ntout=ntout]'
  return
endif

; default keywords
if (n_elements(xrange) eq 0) then xrange=[0,-1]
if (n_elements(yrange) eq 0) then yrange=[0,-1]
if (n_elements(trange) eq 0) then trange=[0,-1]

; set endian
bigendian=1

; read the file header to get data type and cube dimensions
;RR OOPS there is no specification of the number of wavelength samples
crispex_read_header,infile,header=inheader,datatype=datatype, $
  dims=dims,nx=nxin,ny=nyin,nt=ntin,endian=endian_file,$
  stokes=stokes, ns=ns, diagnostics=diagnostics
ntin=fix(0.1+ntin/nwav) ;RR get the real nt in case of small mistake
ntin=ntin/ns            ;RR 4 Stokes samplings if Stokes

; define input assoc size and type
if (datatype eq 1) then inarr=bytarr(nxin,nyin)
if (datatype eq 2) then inarr=intarr(nxin,nyin)
if (datatype eq 4) then inarr=fltarr(nxin,nyin)

; define output assoc size and type
if (xrange[1] eq -1) then xrange[1]=nxin-1
if (yrange[1] eq -1) then yrange[1]=nyin-1
if (trange[1] eq -1) then trange[1]=ntin-1
nxout=xrange[1]-xrange[0]+1
nyout=yrange[1]-yrange[0]+1
ntout=trange[1]-trange[0]+1

; open infile for assoc
get_lun, unit_in
openr,unit_in,infile
inassoc=assoc(unit_in,inarr,512)

; output header
;; if (datatype eq 1) then bitpix=8
;; if (datatype eq 2) then bitpix=16
;; if (datatype eq 4) then bitpix=-32
bitpix=16  ; output always integer
mkhdr,outheader,abs(bitpix)/8,[nxout,nyout,ntout]
sizeoutheader=size(outheader)  
; fits header = Nx36 "card images" = Nx2880 bytes
outheadersize=(1+fix(sizeoutheader[1]/36.))*2880

; ?? add keywords needed for crispex here

; open outfile for assoc
get_lun, unit_out
if (bigendian) then openw,unit_out,outfile,/swap_if_little_endian $
else openw,unit_out,outfile
if (bitpix eq -32) then outassoc=assoc(unit_out,fltarr(nxout,nyout),$
                                       outheadersize)
if (bitpix eq 16) then outassoc=assoc(unit_out,intarr(nxout,nyout),$
                                      outheadersize)
if (bitpix eq 8) then outassoc=assoc(unit_out,bytarr(nxxout,nyout),$
                                     outheadersize)
if (outheadersize ne 0) then begin
  rec=assoc(unit_out, bytarr(outheadersize))
  rec[0]=byte(outheader)
endif

; find doppmin and doppmax of whole sequence for time-constant scaling
for it=0,ntin-1 do begin
  image1=inassoc[it*nwav*ns+iwav1]
  image2=inassoc[it*nwav*ns+iwav2]
  doppimage=(float(image2)-image1)/(image2+image1)
  if (it eq 0) then begin
    doppmin=min(doppimage)
    doppmax=max(doppimage)
  endif else begin
    if (min(doppimage) lt doppmin) then doppmin=min(doppimage)
    if (max(doppimage) gt doppmax) then doppmax=max(doppimage)
  endelse
endfor
if (abs(doppmin) gt doppmax) then doppmax=abs(doppmin)

; make Dopplergram and scale within sequence extreme (zero = zero)
for it=trange[0],trange[1] do begin
  image1=inassoc[it*nwav*ns+iwav1]
  image2=inassoc[it*nwav*ns+iwav2]
  doppimage=(float(image2)-image1)/(image2+image1)
  doppimage=fix(20000*doppimage/doppmax)  ; retain zero = zero

; output this one
  outassoc[it-trange[0]]=doppimage[xrange[0]:xrange[1],yrange[0]:yrange[1]]
endfor

; finish
print,' ----- crisp2dopp wrote '+outfile
free_lun,unit_in,unit_out

end

; =============== test per IDLWAVE S-c ==============================

cd,'/home/rutten/data/SST/2016-09-05-demo/'
infile='crispex/crispex.6563.09:48:31.time_corrected.aligned.icube'
outfile='tests/ha_doppler.fits'
crisp2dopp,infile,outfile,10,4,15   ; black=blue Pit convention
showex,outfile,infile

end
