; file: combinespecial.pro
; init: Feb 21 2018  Rob Rutten  Deil
; last: Nov  6 2019  Rob Rutten  Deil
;+

pro combinespecial,file1,file2,fileout,manipulation,$
  trimbox=trimbox,trange=trange,verbose=verbose

 ; PURPOSE:
 ;   RR special: produce special cubefile as combination of 2 others
 ;   
 ; DESCRIPTION:
 ;   combination = special manipulation
 ;   sort of equal weighting by equalizing means and rms spread
 ;
 ; INPUTS:
 ;   file1: string for first inputfile, fits or La Palma icube
 ;   file2: string for second inputfile, fits or La Palma icube
 ;   fileout: string for output file, always fits
 ;   manipulation = string, choice under "define manipulation" below
 ;     eg "plus' 'minus'

 ; OPTIONAL INPUTS:
 ;   obvious keywords
 ;   
 ; OUTPUTS:
 ;   fileout   
 ;   
 ; HISTORY:
 ;   Feb 22 2018 RR: start
;-

; answer no-parameter query 
if (n_params(0) lt 4) then begin
  sp,combinespecial
  return
endif

; defaults for keywords
if (n_elements(trimbox) eq 0) then trimbox=-1
if (n_elements(trange) eq 0) then trange=[0,-1]
if (n_elements(verbose) eq 0) then verbose=0

; set wall-clock timer (seconds)
timestart=systime(1) 

; print pacifier
if (verbose ne 0) then print,' ===== combinespecial started at UT = '+$
  anytim2utc(timestart,/vms)

; set swap_endian for this computer 
;RR l = little, b = big; fits = big; linux machines are little
if ((byte(1L, 0, 1))[0] eq 1) then endian = 'l' else endian='b'
if (endian eq 'l') then swap_endian=1 else swap_endian=0

; get file extensions
dummy=cgrootname(file1,extension=ext1)  ;RR needs coyotelib
dummy=cgrootname(file2,extension=ext2)

; open assoc files 1 and 2 
if (ext1 eq 'fits') then $
  fits_openassoc,file1,1,nx1,ny1,nt1,bitpix1,headeroff1,swap1
if (ext1 eq 'icube') then $
  lp_openassoc,file1,1,nx1,ny1,nt1,bitpix1,headeroff1,swap1
if (ext2 eq 'fits') then $
  fits_openassoc,file2,2,nx2,ny2,nt2,bitpix2,headeroff2,swap2
if (ext2 eq 'icube') then $
  lp_openassoc,file2,2,nx2,ny2,nt2,bitpix2,headeroff2,swap2

; check input files
if (nx1 ne nx2 or ny1 ne ny2 or nt1 ne nt1) then begin
  print,' ##### combinespecial abort: file1, file2 unequal dimensions'
  return
endif
if (bitpix1 ne 16 or bitpix2 ne 16) then begin
  print,' ##### combinespecial abort: file1 or file2 not integers'
  return
endif

; default trimbox = full image
if (trimbox[0] eq -1) then trimbox=[0,0,nx1-1,ny1-1]

; set partial time sequence (e.g, for quickie tests) 
if (trange[1] ne -1) then begin
  itstart=trange[0]
  itend=trange[1]
endif else begin
  itstart=0
  itend=nt1-1
endelse

; define inassoc
inassoc1=assoc(1,intarr(nx1,ny1),headeroff1)
inassoc2=assoc(2,intarr(nx2,ny2),headeroff2)

; make minimal fits output header
outtype=2
ntout=itend-itstart+1
mkhdr,header_out,outtype,[nx1,ny1,ntout]
header_outsize=(1+fix(n_elements(header_out)/36.))*2880  

; open fits output file for assoc, write header_out
get_lun, unit_out
endian_out='b'
openw,unit_out,fileout,swap_endian=(endian ne endian_out)
outassoc=assoc(unit_out,intarr(nx1,ny1),header_outsize)
rec=assoc(unit_out, bytarr(header_outsize))
rec[0]=byte(header_out)

; frame by frame read, manipulate, write 
for it=itstart,itend do begin

  if (verbose ne 0 and it/20 eq it/20.) then $
    print,' ----- progress: it = '+trimd(it)
  
; get images for this time step 
  im1=float(inassoc1[it])
  im2=float(inassoc2[it])
  
;; ; ### SPECIAL FOR SHIFT
;;   im2=abs(im2)

; set normalization parameters from first image pair in sequence
  if (it eq itstart) then begin
    im1trim=im1[trimbox[0]:trimbox[2],trimbox[1]:trimbox[3]]
    im2trim=im2[trimbox[0]:trimbox[2],trimbox[1]:trimbox[3]]
    meanim1=avg(histo_opt_rr(im1trim))
    meanim2=avg(histo_opt_rr(im2trim))
    im1norm=im1trim/meanim1
    im2norm=im2trim/meanim2
    momim1=moment(histo_opt_rr(im1norm,1.E-5))
    momim2=moment(histo_opt_rr(im2norm,1.E-5))
    rms1=sqrt(momim1[1])
    rms2=sqrt(momim2[1])
  endif

; rescale images to mean and dynamic range defined from first image pair
; NB: wrong for signed quantities as shift, Doppler, magnetogram 
  im1scal=(im1-meanim1)/meanim1/rms1+10
  im2scal=(im2-meanim2)/meanim2/rms2+10

;; ; show distributions
;; cgwindow
;; cghistoplot,im1scal,nbins=50,/window,xrange=[0,20],xstyle=1
;; cgwindow
;; cghistoplot,im2scal,nbins=50,/window,xrange=[0,20],xstyle=1 
;; STOP

; define manipulation 
;;  im3=im1scal*(im1scal-im2scal) ; ionization detector from W_ha, W_ca
;; im3=(im1scal>11.)*(im2scal<11.)  ; 11-11 = GOOD mean correlation IRIS 1400
;; im3=(im1scal>11.)*(im2scal>11.)  ; 11-11 less correlation  
;; im1scal[where(im1scal le 11)]=0  ; 11 means 1 rms above mean
;; im2scal[where(im2scal ge 11)]=0
;; im1scal[where(im1scal le 11)]=11  ; 11 means 1 rms above mean
;; im2scal[where(im2scal ge 11)]=0
;; im3=im1scal*im2scal ; pure = mean IRS correspondence = where go together
;; im3=im1scal*im2scal ; old shxwi spicule-II finder 
;; im3=im1scal/im2scal*100.

 ; set as input paramter
  if (manipulation eq 'plus') then im3=im1scal+im2scal ; network heating
  if (manipulation eq 'minus') then im3=im1scal-im2scal ; spicular heating

   im4=histo_opt_rr(im3,1.E-4)
;;  im4=im4>0

; convert to nice integer range
  intim=fix(im4*1.E2) 

;; ; inspect first image
;; print,' ----- minmax intim = '+trimd(minmax(intim))
;; sv,intim
;; STOP

; done: write output image
  outassoc[it-itstart]=intim

endfor

; get minmax output file for check on integer excess
minmaxout=minmaxcubefile(fileout,trimbox=trimbox)
if (abs(minmaxout[0]) ge 32000 or abs(minmaxout[1]) ge 32000) then begin
  print,' ##### combinespecial warning: output exceeds integer range'
  print,' ----- minmaxout ='+trimd(minmaxout)
endif
if (verbose ne 0) then begin
  print,' ----- minmaxout ='+trimd(minmaxout) 
  print,' ----- combinespecial wrote file '+fileout
endif

; free the input and output files
free_lun,1,2,unit_out

;; reformcubefile,fileout,'/tmp/newfileout.fits',cubehistopttop=1.E-5

; print elapsed time
timedone=systime(1)
timelaps=ntostr((timedone-timestart)/60.,format='(F11.1)')
if (verbose ne 0) then print,' ===== combinespecial took '+timelaps+' minutes'

end



; =============== main for testing per IDLWAVE H-c ======================

; SST quiet data used in contrail paper 
cd,'/media/rutten/SSTDATA/alldata/SST/2014-06-21-quiet'
file1='constructs/width_kevin_6563_0.9.fits'
file2='constructs/minint_kevin_8542_0.6.fits'
;; file2='constructs/shift_kevin_6563_0.9.fits'
;; file2='constructs/width_kevin_8542_0.6.fits'

fileout='constructs/comb_HaW-CaI.fits'
;; fileout='constructs/comb_HaW+CaI.fits'
;; fileout='constructs/comb_HaWxHaS.fits'
;; fileout='constructs/comb_HaWdivCaW.fits'

;;;;; trimbox=[50,50,880,800]  ; NO LONGER
;; trange=[88,99]

combinespecial,file1,file2,fileout,'minus',$
  trange=trange,trimbox=trimbox,verbose=1

; check
showex,fileout,trimbox=trimbox

end
