; file: sst_findlocation.pro = use SDO to find where SST was pointing
; init: Apr 18 2017  Rob Rutten  Deil
; last: Aug 13 2022  Rob Rutten  Deil
; site: rridl/sdolib

;+
pro sst_findlocation,sstfile,x_sst,y_sst,angle_sst,px_sst,$
  sstrednorotate=sstrednorotate,$
  ssttimesav=ssttimesav,obsdate=obsdate,ssttimetai=ssttimetai,$
  summertime=summertime,tablecor=tablecor,$
  skipmetcalf=skipmetcalf,rotit0=rotit0,$
  range=range,sdowav=sdowav,$
  trimboxsst=trimboxsst,itsample=itsample,$
  smearsst=smearsst,histopsst=histopsst,flatten=flatten,$
  muckdarksdo=muckdarksdo,muckdarksst=muckdarksst,$
  muckbrightsdo=muckbrightsdo,muckbrightsst=muckbrightsst,$
  shiftfor=shiftfor,shiftrev=shiftrev,$
  nxfor=nxfor,nyfor=nyfor,nxrev=nxrev,nyrev=nyrev,$
  blink=blink,checkmetcalf=checkmetcalf,show=show

 ; find precise solar X,Y location and orientation of SST images
 ; this version should accept both "classic" LP files and SSTRED files
 ; (SSTRED: reduction pipeline of 2021A&A...653A..68L)
 ; 
 ; METHOD:
 ; - distill SST pointing parameters from turret log or specify better
 ; - call stx_findlocation.pro
 ;     advised comparison pairs: 
 ;       workhorse: Halpha wide band versus HMI continuum
 ;       often good: SST Halpha longest wavelength versus HMI continuum
 ;       alternatives: SST magnetogram versus HMI magnetogram
 ;                     Ca II 8542 longest wavelength versus AIA 1700
 ; examples at end of this file
 ;
 ; INPUTS:
 ;   sstfile = string filename SST image sequence in a fitscube file
 ;     or in a "La Palma" format cubefile (but NOT a multi-wav F-P file). 
 ;     Either  wide-band file or one wavelength of a multi-wav F-P file,
 ;     made with crisp2fits.pro (classic) or sst_mwfits2fitscube.pro (SSTRED)
 ;   x_sst, y_sst: SST (X,Y) pointing (arcsec; default -999 = turret value)
 ;   angle_sst: input solar FOV angle (deg NESW)
 ;      classic: default -999 => use turret value
 ;      SSTRED: default -999 => use 0 when rotated to North instead of turret
 ;   px_sst: SST pixel size (in arcsec, default -999 => 0.058 for CRISP)
 ;
 ; OPTIONAL KEYWORD INPUTS:
 ;   sstrednorotate = 1/0:  SSTRED did not rotate images to solar North
 ;   ssttimesav = string, file name classic SST sequence timings
 ;     IDL save file must contain array "time" with time strings without date;
 ;     required for pre-SSTRED "classic" files, not for SSTRED 
 ;   obsdate = string with observing date in mandatory format '2014.06.19'
 ;     required for classic to add date to the time strings, not for SSTRED
 ;   ssttimetai = string file name SST sequence timings in TAI
 ;     required for SSTRED = output of sst_reform5d.pro (may do just that)
 ;     or sst_mwfits2fitscube.pro.  Written as output for classic (if
 ;     not given written as sstfile_tai.dat).
 ;   summertime=1/0: SST log uses La Palma summertime (old habit, default 0)
 ;   tablecor = some fixed SST quantity (I don't know detail) (default 48.)
 ;   skipmetcalf: skip (slow) Metcalf improvement of angle and scale
 ;   rotit0=1/0: include solar rotation correction to add (X,Y) at it=0 
 ;     (needed for sdo_getdata_rr.pro, default=1)
 ;   range = X and Y search range in AIA px of 0.6 arcsec (default 50)
 ;   sdowav = SDO diagnostic to correlate with (default 'continuum')
 ;   trimboxsst:  4-elem int vector [xmin,ymin,xmax,ymax] to select subfield
 ;     NB: resetting trimbox may give other SST it selection: reget sdoimage
 ;     trimboxsst=-2: showex 1st SST image to define trimbox (click ll, ur)
 ;   itsample: it for SST sample (default = sharpest near midtime)
 ;   smearsst: smear SST image (default = -1.5 = 1.5*pxratio)
 ;   histopsst: hist_opt SST image with given value (clip limit < 1, top only)
 ;   flatten: subtract boxcar average from each, value = width in coarsest px
 ;    value 5 good for granulated scenes, larger for spots or limb in field
 ;   muckdark: set intensities below threshold (units mean=1) to mean
 ;   muckbright: set intensities above threshold (units mean=1) to mean
 ;   checkmetcalf: showex final Metcalf blink (may replace blink)
 ;   blink: Metcalf blink duration for [SDO cutout - SST image] (default 10)
 ;   show = 0/1/2/3 as findalignimages.pro (1 starts result showexes)

 ; OUTPUTS:
 ;   x_sst, y_sst: improved (X,Y) coordinates center of SST field of view
 ;   angle_sst: improved SST orientation angle NESW (counterclockwise)
 ;   px_sst: improved SST pixel size in arcsec 
 ;
 ; OPTIONAL KEYWORD OUTPUTS
 ;   nxfor,nyfor = central-cut common size in finest px
 ;   shiftrev = im2 [xshift,yshift] in finest-px units for SST > SDO
 ;   nxrev,nyrev = central-cut size in finest px, accommodating SST rotation
 ;
 ; RESTRICTIONS:
 ;   needs image at good SST seeing and no post-eclipse SDO defocus   
 ;   SST data should be derotated to angle at mid-sequence 
 ;
 ;   WARNING: 
 ;   parameters are sticky when rerunning within same session
 ;  
 ; HISTORY:
 ;   Apr 18 2017 RR: start   
 ;   May  3 2017 RR: angle and pixel improvement estimation
 ;   May 19 2017 RR: improved Metcalf per findalignimages.pro 
 ;   Nov 21 2017 RR: auto-detect bestitmidseq
 ;   Mar  3 2021 RR: itsample
 ;   Aug 12 2022 RR: accommodate cumbersome SSTRED 
;-

; answer no-parameter query 
if (n_params(0) lt 5) then begin
  sp,sst_findlocation
  return
endif

; defaults for keyword parameters
if (n_elements(x_sst) eq 0) then x_sst=-999
if (n_elements(y_sst) eq 0) then y_sst=-999
if (n_elements(angle_sst) eq 0) then angle_sst=-999
if (n_elements(px_sst) eq 0) then px_sst=-999
if (n_elements(sstrednorotate) eq 0) then sstrednorotate=0
if (n_elements(ssttimesav) eq 0) then ssttimesav='none'
if (n_elements(obsdate) eq 0) then obsdate='none'
if (n_elements(ssttimetai) eq 0) then ssttimetai='none'
if (n_elements(skipmetcalf) eq 0) then skipmetcalf=0
if (n_elements(rotit0) eq 0) then rotit0=1
if (n_elements(summertime) eq 0) then summertime=0
if (n_elements(tablecor) eq 0) then tablecor=48.
if (n_elements(range) eq 0) then range=50  ; NB default
if (n_elements(sdowav) eq 0) then sdowav='continuum'
if (n_elements(trimboxsst) eq 0) then trimboxsst=-1
if (n_elements(itsample) eq 0) then itsample=-1
if (n_elements(smearsst) eq 0) then smearsst=-1.5  ; NB default
if (n_elements(histopsst) eq 0) then histopsst=0
if (n_elements(flatten) eq 0) then flatten=0  ; changed default
if (n_elements(muckdarksdo) eq 0) then muckdarksdo=0 
if (n_elements(muckdarksst) eq 0) then muckdarksst=0 
if (n_elements(muckbrightsdo) eq 0) then muckbrightsdo=0 
if (n_elements(muckbrightsst) eq 0) then muckbrightsst=0 
if (n_elements(shiftfor) eq 0) then shiftfor=0
if (n_elements(shiftrev) eq 0) then shiftrev=0
if (n_elements(nxfor) eq 0) then nxfor=0
if (n_elements(nxrev) eq 0) then nxrev=0
if (n_elements(nyfor) eq 0) then nyfor=0
if (n_elements(nyrev) eq 0) then nyrev=0
if (n_elements(blink) eq 0) then blink=10
if (n_elements(checkmetcalf) eq 0) then checkmetcalf=0
if (n_elements(show) eq 0) then show=0

; get sstfile extension to find its type
dummy=cgrootname(sstfile,extension=ext)  ;RR needs coyotelib

; Jul 27 2022: split between classic and sstred
; icube = classic; fits may be sstred or my own production maybe from sstred
sstred=0
if (ext eq 'fits') then begin
  headsst=headfits(sstfile)
  solnet=fxpar(headsst,'SOLARNET')
  if (solnet eq 1) then sstred=1
endif

if (sstred eq 0 and (obsdate eq 'none' or ssttimesav eq 'none')) then begin
  print,' ##### sst_findlocation abort: '+$
    'classic (non-SSTRED) requires obsdate and ssttimesav'
  return
endif

if (sstred eq 1) and (ssttimetai eq 'none') then begin
  print,' ##### sst_findlocation abort: '+$
    'SSTRED requires file ssttimetai '+$
    ' from sst_reform5d.pro or sst_mwfits2fitscube.pro'
  return
endif

; get timing, classic = file ssttimesav, sstred = file ssttimetai
;RR ssttimesav must have array time = strings without date as they used to be
;RR since SSTRED there seem to be time = TAI and times = time strings?
;RR ssttimetai must be TAI from sst_reform5d.pro or sst_mwfits2fitscube.pro
if (sstred eq 0) then begin
  restore,ssttimesav
  sztime=size(time)
  if (sztime[0] ne 1) then begin
    print,' ##### sst_findlocation abort: '+$
      'variable "time" in ssttimesav not 1D array'
    return
  endif
  if (sztime[2] eq 5) then time=times ; funny new habit?
  sztime=size(time)
  if (sztime[2] ne 7) then begin
    print,' ##### sst_findlocation abort: ssttimesav time not strings'
    return
  endif
  nt_sst=sztime[1]
  timearr_sst_datetime=obsdate+'_'+time
  timearr_sst=anytim2tai(timearr_sst_datetime)
  cadence_sst=(timearr_sst[nt_sst-1]-timearr_sst[0])/(nt_sst-1)
  taistart=timearr_sst[0] ; for (X,Y) at it=0 printout
endif
if (sstred eq 1) then begin
  readcol,ssttimetai,timearr_sst,format='(D)'
  nt_sst=n_elements(timearr_sst)
  cadence_sst=(timearr_sst[nt_sst-1]-timearr_sst[0])/(nt_sst-1)
  taistart=timearr_sst[0] ; for (X,Y) at it=0 printout
endif

; if classic write ssttimetai file for future use
if (sstred eq 0) then begin
  timeonly=cgrootname(ssttimetai,directory=timepath,extension=timeext)
  if (timeext ne 'dat') then begin ; also includes 'none' for absent
    sstonly=cgrootname(sstfile,director=sstpath,extension=sstext) 
    ssttimetai=sstonly+'_tai.dat'
    print, ' !!!!! no file ssttimetai or no .dat extension'+$
      ' wrote ssttimetai = '+ssttimetai
  endif
  writecol,ssttimetai,timearr_sst,fmt='(5E21.12)'
endif

; get itsample = best rms in sequence midrange (length 2x 10 min)
if (itsample eq -1) then begin
  itmid=fix((nt_sst-1)/2.)        
  itrange=fix(600./cadence_sst)
  if (2*itrange+1 gt nt_sst) then itrange=fix((nt_sst-1)/2.)
  rms_sst=fltarr(2*itrange+1)
  if (ext eq 'icube' or ext eq 'fcube' or ext eq 'bcube') then $
    sstarr=readsstfile(sstfile,trange=[itmid-itrange,itmid+itrange])
  if (ext eq 'fits') then $
    sstarr=readfitscube(sstfile,trange=[itmid-itrange,itmid+itrange])
  for it=0,2*itrange do begin
    sstim=sstarr[*,*,it]
    if (trimboxsst[0] eq -2) then begin 
      print,' !!!!! select trimboxim2 by clicking lowerleft, upperright'
      print,'       results typed in terminal, enter in next run'
      showex,sstim
      return
    endif
    if (trimboxsst[0] eq -1) then $
      reformimage,sstim,sstimcut,/croptriangles else $
        reformimage,sstim,sstimcut,trimbox=trimboxsst
    momim=moment(sstimcut)
    rms_sst[it]=sqrt(momim[1])
  endfor
  itsample_sst=where(rms_sst eq max(rms_sst))+itmid-itrange
  if (show) then begin
    plot,indgen(2*itrange+1)+itmid-itrange,rms_sst,/ynozero,$
      title='rms around midpoint sequence at it = '+trimd(itmid)
    plots,itsample_sst,max(rms_sst),psym=6,symsize=3
  endif
  print,' ----- itsample_sst ='+trimd(itsample_sst)+$
    '  itmid ='+trimd(itmid)
endif else itsample_sst=itsample

; get image at itsample
if (ext eq 'fits') then $
  im_sst=readfitscube(sstfile,trange=[itsample_sst,itsample_sst])
if (ext eq 'icube' or ext eq 'fcube' or ext eq 'bcube') then $
  im_sst=readsstfile(sstfile,trange=[itsample_sst,itsample_sst])

; repeat trimbox selection display if itstart was given
if (trimboxsst[0] eq -2) then begin 
  print,' !!!!! select trimboxim2 by clicking lowerleft, upperright'
  print,'       results typed in terminal, enter in next run'
  showex,im_sst
  return
endif

; get datetime in RR format for sst_getturret.pro and stx_findlocation.pro
datetimecc=anytim2utc(timearr_sst[itsample_sst],/ccsds)
datetimeut=str_replace(datetimecc,'-','.')
;RR Jul 31 2022 next gave three days of error searching
;RR likely culprit: SSW read_sdo.pro expects file list = strarr
datetimeitsample=strarr(1)
datetimeitsample[0]=str_replace(datetimeut,'T','_')

; get SST turret info unless already given
if (x_sst eq -999 or y_sst eq -999 or angle_sst eq -999) then begin
  sst_getturret,datetimeitsample,x_turret,y_turret,angle_turret,$
    summertime=summertime ;; ,/quiet
  useturret=1
  if (x_sst eq -999) then x_sst=x_turret
  if (y_sst eq -999) then y_sst=y_turret
  if (angle_sst eq -999) then begin
    if (sstred eq 0) then angle_sst=angle_turret                 ; classic
    if (sstred eq 1 and sstrednorotate eq 0) then angle_sst=0.   ; North up
    if (sstred eq 1 and sstrednorotate eq 1) then angle_sst=angle_turret
  endif
endif else useturret=0

; default px size = CRISP value
if (px_sst eq -999) then px_sst=0.0575

; store input values for printout at end
x_sst_0=x_sst
y_sst_0=y_sst
angle_sst_0=angle_sst

; do the work
stx_findlocation,im_sst,datetimeitsample,$
  x_sst,y_sst,angle_sst,px_sst,$
  skipmetcalf=skipmetcalf,$
  xyrange=xyrange,sdowav=sdowav,$ ; "new"
  trimboxstx=trimboxsst,$
  smearstx=smearsst,histopstx=histopsst,$
  muckdarksdo=muckdarksdo,muckdarkstx=muckdarksst,$
  muckbrightsdo=muckbrightsdo,muckbrightstx=muckbrightsst,$
  flatten=flatten,$
  checkmetcalf=checkmetcalf,blink=blink,show=show,verbose=1

; optional solar rotation correction to get X at it=0 
; (start of image sequence needed for sdo_getdata_rr download and processing)
if (rotit0 ne 0) then begin
  taisample=anytim2tai(datetimeitsample)
  posit0=rot_xy(x_sst,y_sst,taistart-taisample)
  x_sst_t0=posit0[0]
  y_sst_t0=posit0[1]
;; ; check
;; print,' ----- datetimes = ',datetimeitsample,'   ',datetimeit0 
;; print,' ----- rotshift = ',x_sst-posit0[0],y_sst-posit0[1]
; print result
  print,' ----- above holds for itsample_sst ='+trimd(itsample_sst)
  print,' ----- at it=0 (specify in getting SDO): (X,Y) ='$
    +trimd([x_sst_t0,y_sst_t0],1)
endif

; print input values (delayed to appear after aia_prep spam)
if (skipmetcalf eq 0) then print,' ----- input was: '+$
  'useturret ='+trimd(useturret)+$
  '  (X,Y) ='+trimd([x_sst_0,y_sst_0],1)+$
  '  angle ='+trimd(angle_sst_0,2)

end


; =============== main for testing per IDLWAVE H-c ======================

; ---------------- first old "classic" tests

;; ; 2021 SOLNET school demo
;; ; SST files available https://robrutten.nl/rrweb/sdo-demo/instruction.html
;; cd,'/media/rutten/RRHOME/alldata/SST/2014-06-24-solnet/sst'
;; sstfile='wb.6563.corrected.aligned_cut.icube'
;; obsdate='2014.06.24'
;; ssttimesav='tseries.6563_cut.sav'
;; flatten=5 ; granulation
;; ;RR Aug 11 2022 here = fine: turret (X,Y) = -122.6 -116.9  angle = 63.9
;; ;   result: (X,Y) = -114.9 -112.4  angle_stx = 62.00  px_stx = 0.0567
;; ;   rinses up to 0.8 px
;; ;   final forshift 0.031 -0.110 revshift = -0.103 0.055

;; ;RR easy disk-center quiet-sun granulation scene
;; cd,'/media/rutten/RRHOME/alldata/SST/2013-05-03-krishna/sst' 
;; sstfile='ha_wb.fits'
;; obsdate='2013.05.03'
;; ssttimesav='tseries.6563.09:06:15.calib.sav'
;; flatten=5 ; granulation
;; histopsst=0.05 ; undo BPs
;; ;RR Aug 10 2022 redo fine: turret (X,Y) = 0.0 -0.6  angle = 49.5
;; ;   result: (X,Y) = -4.6 -5.2  angle_stx = 48.46  px_stx = 0.0576
;; ;   final forshift 2.350E-03 0.100 revshift = 0.055 0.135

;; ;RR rather difficult sunspot with dark umbra and many bright points in moat
;; cd,'/media/rutten/RRHOME/alldata/SST/2016-09-05-demo'
;; ;; sstfile='sst/ha_wb.fits'   ; has black edges top and right
;; sstfile='crispex/wb.6563.09:48:31.corrected.aligned.icube'
;; obsdate='2016.09.05'
;; ssttimesav='crispex/tseries.8542.09:48:31.calib.sav'
;; histopsst=0.1  ; muck moat BPs, tops get cut and then smeared
;; ;; flatten=0  ; final forshift = -2.551 2.738 revshift = 0.072 0.318
;; ;; flatten=5  ; final forshift = 0.057 -1.648 revshift = -0.262 1.521
;; ;; flatten=10 ; final forshift = -1.942 -0.380 revshift = -0.290 1.506
;; ;; flatten=20 ; final forshift = -1.447 -0.395 revshift = -0.299 1.478
;; ;RR Aug 12 2022 redo fine: turret (X,Y) = -112.9 23.4  angle = 98.4
;; ;   result: (X,Y) = -115.6 18.9  angle_stx = 98.18  px_stx = 0.0568

;; ;RR very difficult featureless near-limb scene including limb 
;; path='/media/rutten/RRHOME/alldata/SST/2014-06-19-limb/09:56:38/'
;; sstfile=path+'sst/ha_wb.fits'
;; sdowav='continuum'
;; obsdate='2014.06.19'
;; ssttimesav=path+'sst/tseries.6563.09:56:38.calib.sav'
;; summertime=1    ;RR old dates have silly summertime in turret files
;; flatten=20 ; does a good job 
;; trimboxsst=[175,58,915,594] ; via new option trimboxsst=-2, exclude limb
;; ;RR Aug 10 2022 redo: blinks amazingly good, rerun in sticky IDL session 
;; ; useturret = 1  (X,Y) = 9.5 891.1  angle = 58.6
;; ; run 1: result: (X,Y) = 32.8 924.0  angle_stx = 53.07  px_stx = 0.0575
;; ;   final forshift = -2.154 1.724 revshift = -0.252 0.227
;; ; run 2: result: (X,Y) = 15.7 923.9  angle_stx = 53.04  px_stx = 0.0574
;; ;   final forshift = -2.325 1.898 revshift = -0.248 0.401
;; ;   correction solar X OK?  (Multiple repeats yesterday also 15.4 924.2) 

;; ;RR Vasco Henriques data CHROMIS + CRISP quiet Sun
;; cd,'/media/rutten/RRHOME/alldata/SST/2018-06-10-vasco/crisp'
;; sstfile='wb.6563.12:21:09.corrected.icube'
;; sdowav='continuum'
;; obsdate='2018.06.10'
;; ssttimesav='tseries.6563.12:21:09.calib.sav'
;; histopsst=0.1
;; flatten=5
;; ;RR Aug 10 2022 redo fine: useturret = 1  (X,Y) = 3.3 -1.5  angle = 35.6
;; ;  result: (X,Y) = 5.6 -10.5  angle_stx = 35.41  px_stx = 0.0587
;; ;  final forshift = -1.038E-03 0.049 revshift = 0.071 -0.062

;; ;RR Vasco Henriques CHROMIS + CRISP quiet Sun now CaII 8542
;; cd,'/media/rutten/RRHOME/alldata/SST/2018-06-10-vasco/crisp'
;; sstfile='wb.8542.12:21:09.corrected.icube'
;; sdowav='continuum'
;; obsdate='2018.06.10'
;; ssttimesav='tseries.8542.12:21:09.calib.sav'
;; histopsst=0.1
;; ;RR OK result:  (X,Y) = 9.7 -11.9  angle_stx = 35.30  px_stx = 0.0588
;; ;RR Aug 10 2022 redo:  10.4 -9.9  angle_stx = 35.44  px_stx = 0.0587

;; ;RR Aditi Bhatnagar problem SDO-eclipse early 2021
;; cd,'/media/rutten/RRHOME/alldata/SST/2020-08-16-aditi/sst'
;; sstfile='3950_gran.fits'
;; obsdate='2020.08.16'
;; ssttimesav='times.3950.idlsave'
;; px_sst=0.0377 ; CHROMIS px so CRISP default wrong, add test?
;; itsample=123 ; nt=224, this is after the highest, had SDO eclipse problem
;; histopsst=0.1 ; likely not needed for this soupy scene
;; ; Aug 10 2022 redo: fine! useturret = 1  (X,Y) = 514.7 618.5  angle = 96.3
;; ;   result: (X,Y) = 512.6 611.4  angle_stx = 97.58  px_stx = 0.0377
;; ;   final forshift = -0.222 1.340 revshift = -0.025 -0.799

; ---------------- SSTRED tests

;RR Luc Rouppe vd V test data = difficult limb scene
cd,'/media/rutten/RRHOME/alldata/SST/2020-08-13-AR-5D/sst/crisp'
; 8542 red wing versus 1700 (IN acoustics + MC bright points)
sstfile='redwing_18_8542.fits'
ssttimetai='nb_8542_sstrr_tai.dat'
sdowav='1700'
flatten=20 ; without this totally non-robust for close startoff values
trimboxsst=[140,140,900,900] ; inside limb but auto-triangle-removal OK
; turret X = -750.5  Y = -502.5  angle_sst = 95.0 angle undone SSTRED
;RR Aug 13 2022 redo: blinks amazingly good, rerun in sticky IDL session
; itsample=43
; run1: final forshift 2.127 1.817 revshift 0.104 -0.490 (NB nonmucked)
;       result: (X,Y) = -763.9 -494.7  angle_stx = 0.14  px_stx = 0.0589
; run2: final forshift 2.151 1.811 revshift 0.069 -0.499
;       result: (X,Y) = -764.5 -494.6  angle_stx = 0.12  px_stx = 0.0589
; run3: final forshift 2.139 1.810 revshift 0.024 -0.703
;       result: (X,Y) = -763.8 -495.4  angle_stx = 0.06  px_stx = 0.0588
; run4: final forshift 2.148 1.814 revshift 0.085 -0.501
;       result: (X,Y) = -763.9 -495.6  angle_stx = 0.12  px_stx = 0.0589
; limbward height offsets between 8542 redwing and 1700 in shocks and MCs?
; MCs looses 8542 damping and 1700 metal edge opacities, holes as deep?
; shocks likely differ between top-down and slanted viewing, risky pair

;; ;RR above but Halpha wide band versus HMI cont (earlier no good anyhow)
;; cd,'/media/rutten/RRHOME/alldata/SST/2020-08-13-AR-5D/sst/crisp'
;; sstfile='wb_6563_2020-08-13T07:58:01_scans=0-82_corrected_im.aligned8542.icube'
;; ssttimesav='wb_6563_2020-08-13T07:58:01_scans=0-82_corrected_im_times.idlsave'
;; obsdate='2020.08.13'
;; sdowav='continuum'
;; itsample=49 ; from default plot, later than tmid since HMI bad at start
;; flatten=20 ; runaway without 
;; ;; ;RR initial try: terrible runaway; I then revampled findalignimages.pro
;; ;; ;; trimboxsst=-2 ; select no-limb box (draw box, click ll+ur corners)
;; trimboxsst=[50,150,909,973]
;; ;; ;RR Aug 13 2022: redo using 8542-1700 results as startoff; blinks fine
;; x_sst=-764 & y_sst=-496 & angle_sst=0.1 & px_sst=0.0588
;; ; run 1: final forshift -83.410 -28.122 revshift -0.140 -0.164 ; nonmuck!
;; ;        result: (X,Y) = -784.8 -459.9  angle_stx = 0.09  px_stx = 0.0587
;; ; run 2: final forshift -83.410 -28.122 revshift -0.140 -0.164
;; ;        result: (X,Y) = -784.8 -459.9  angle_stx = 0.09  px_stx = 0.0587
;; ;RR persistent but seems off in (X,Y) (turret -750, -502), why?

; -------------------------------------------- run current test

; output parameters
blink=6
checkmetcalf=1
show=3

 sst_findlocation,sstfile,x_sst,y_sst,angle_sst,px_sst,$
  sstrednorotate=sstrednorotate,$
  ssttimesav=ssttimesav,obsdate=obsdate,ssttimetai=ssttimetai,$
  summertime=summertime,tablecor=tablecor,$
  skipmetcalf=skipmetcalf,rotit0=rotit0,$
  range=range,sdowav=sdowav,$
  trimboxsst=trimboxsst,itsample=itsample,$
  smearsst=smearsst,histopsst=histopsst,flatten=flatten,$
  muckdarksdo=muckdarksdo,muckdarksst=muckdarksst,$
  muckbrightsdo=muckbrightsdo,muckbrightsst=muckbrightsst,$
  shiftfor=shiftfor,shiftrev=shiftrev,$
  nxfor=nxfor,nyfor=nyfor,nxrev=nxrev,nyrev=nyrev,$
  blink=blink,checkmetcalf=checkmetcalf,show=show

end

