; file: sdo_writeallpairsplines.pro
; init: Feb 21 2016  Rob Rutten  Deil
; last: Mar  2 2022  Rob Rutten  Deil
; note: muck parameter choices marked by !!!!

;+
pro sdo_writeallpairsplines,datadir,xalpair,$
  trimbox=trimbox,euvanchor=euvanchor,boxcarmin=boxcarmin,$
  verbose=verbose

 ; NAME:
 ;   sdo_writeallpairsplines
 ;   
 ; PURPOSE:
 ;   call sdo_writepairspline.pro to get shift splines for all desired pairs
 ;   
 ; INPUTS   
 ;   datadir = string path/dir containing /cubes to pair-spline
 ;
 ; OUTPUT
 ;   xalpair = last pair defined in pairs below, for alignment check plot
 ;
 ; OPTIONAL KEYWORD INPUTS:
 ;   trimbox = align only over selected subfield [xmin,ymin,xmax,ymax]
 ;   euvanchor =  anchor EUV default ['304','mag']
 ;   boxcarmin = [dx,dy,dt] in [px,px,min] smear all cubes, default none 
 ;   verbose = 1/0: yes/no show align plots on screen 
 ;
 ; OUTPUTS:
 ;   spline structure files and splinefit plots
 ;   NB: order the pairs needed in sdo_getsumsplineshift.pro + xalpair
 ; 
 ; HISTORY:
 ;   Feb 24 2016 RR: start
 ;   Apr 15 2016 RR: spline smooth parameter out
 ;   Feb 16 2017 RR: new pair sequence
 ;   Feb  9 2020 RR: euvanchor choices (but back to 1600-304)
 ;   Feb 13 2020 RR: output xalpair
 ;   May 30 2020 RR: default euvanchor 1700-304
 ;   Jun 12 2020 RR: boxcarmin, euvanchor mag-304, definable delay
;-

; answer no-parameter query 
if (n_params(0) lt 2) then begin
  sp,sdo_writeallpairsplines
  return
endif

; default keyword values
if (n_elements(trimbox) eq 0) then trimbox=[-1,-1,-1,-1]
if (n_elements(euvanchor) eq 0) then euvanchor=['304','mag']
if (n_elements(boxcarmin) eq 0) then boxcarmin=[0,0,0]
if (n_elements(verbose) eq 0) then verbose=0

; check euvanchor
if (not(euvanchor[1] eq '1600' and euvanchor[0] eq '304') and $
    not(euvanchor[1] eq '1700' and euvanchor[0] eq '304') and $
    not(euvanchor[1] eq '1700' and euvanchor[0] eq '131') and $
    not(euvanchor[1] eq 'mag' and euvanchor[0] eq '304')) then begin
  print, ' ##### WARNING do_writeallpairsplines: specified euvanchor'+$
    ' not (yet) implemented; falling back on 1700-304'
  print,'      implement in sdo_writepairspline, sdo_writeallpairsplines,'+$
    '  and sdo_getsumsplineshift'
  euvanchor=['304','mag'] ;RR is NOT UV but HMI but the name sticks
  wait,20
endif

; EUV anchor nrs
euvanchornrs=[0,0]
if (euvanchor[0] eq '304') then euvanchornrs[0]=6
if (euvanchor[1] eq '1600') then euvanchornrs[1]=8
if (euvanchor[1] eq '1700') then euvanchornrs[1]=9
if (euvanchor[1] eq 'mag') then euvanchornrs[1]=11

;  RR channel numbers and AIA telescopes
; 94,131,171,193,211,304,335,1600,1700,4500,mag,cont,dop
;  1  2   3   4   5   6   7   8    9    10   11  12   13
;  4  1   3   2   2   4   1   3    3    3    0   0    0

; desired pairs to co-align (see wrk/SDO/alignment/00-align-scheme.txt)
; NB: last pair is the xal alignment check pair
; NB: adapt sdo_getsumsplines summing sequences accordingly
; NB: euvanchor pair order is reverse in below definitions for historic reason
;     the usage is that pair[0] will be put onto pair[1]

; CURRENT DEFAULT euvanchor ['304','mag'] (with time manipulation [??])
; including (6,8) and (6,9) for comparison with (6,11)
; Mar  2 2022: 94 from >211 to > 335
if (euvanchor[1] eq 'mag' and euvanchor[0] eq '304') then $  
  pairs=[[6,11],[9,11],[8,9],[5,6],[3,6],[4,3],[7,5],[1,7],[2,7],$
         [2,4]] ; last = xal pair = two chains in sdo_getsumsplineshift.pro  

; default from May 31 2020 to Jun 11 2020 with euvanchor ['1700','304']
; including (6,8) for comparison with (6,9)
if (euvanchor[1] eq '1700' and euvanchor[0] eq '304') then $  
  pairs=[[9,11],[8,9],[6,8],[6,9],[5,6],[3,6],[1,5],[7,5],[4,3],[2,7],[2,4]]

; next was standard up to May 31 2020, with euvanchor ['1600','304'] 
if (euvanchor[1] eq '1600' and euvanchor[0] eq '304') then $  
  pairs=[[9,11],[8,9],[6,8],[5,6],[3,6],[1,5],[7,5],[4,3],[2,7],[2,4]]

;; ; some trial around Feb 2020 I think with euvanchor ['1700','131']
;; if (euvanchor[1] eq '1700' and euvanchor[0] eq '131') then $
;;   pairs=[[9,11],[8,9],[2,9],[6,2],[3,6],[5,6],[7,5],[7,5],[4,3],[1,5],[2,1]]

; before Apr 14 2016 
;; pairs=[[9,11],[8,9],[6,8],[3,6],[1,6],[5,3],[4,5],[7,1],[2,1]]
; since Apr 15 2016  
;; pairs=[[9,11],[8,9],[6,8],[5,6],[3,6],[1,5],[7,5],[2,5],[4,5],[1,6]]
; since May  7 2016 
;; pairs=[[9,11],[8,9],[6,8],[5,6],[3,6],[1,5],[7,5],[2,5],[4,3],[4,5],[1,6]]
; standard Feb 16 2017 - Feb 15 2020  
;; pairs=[[9,11],[8,9],[6,8],[5,6],[3,6],[1,5],[7,5],[4,3],[2,1],[2,3]]

; other trials?
;; if (euvanchor[1] eq '1700' and euvanchor[0] eq '304') then $  
;;   pairs=[[9,11],[8,9],[6,9],[5,6],[3,6],[1,5],[7,5],[4,3],[2,3],[2,9]] 
;; if (euvanchor[1] eq '1600' and euvanchor[0] eq '131') then $
;;   pairs=[[9,11],[8,9],[2,8],[6,2],[5,6],[3,6],[1,5],[7,5],[4,3],[2,1]]

npairs=n_elements(pairs)/2
xalpair=pairs[*,npairs-1]

; ============ boxcar business, still present but NO LONGER IN USE

; NB: tests Jun 9 2020 for data/SDO/2014-06-14-longer showed no
;     cross-align improvement, so back to default = no smearing
;     Jun 12 2020: time smearing and delay for euvanchor pairs only
boxcarlabel=''  ; default none
if (total(boxcarmin) gt 0) then begin
  boxcarlabel=trim(boxcarmin[0])+trim(boxcarmin[1])+trim(boxcarmin[2]) 
  
; convert boxcar minutes to nearest number of time steps
  cubefiles=file_search(datadir+'/cubes/*.fits',count=ncubes)
  if (ncubes eq 0) then begin
    print,' ###### sdo_writeallpairsplines finds no center/cubes files'
    STOP
  endif
  cubehead=headfits(cubefiles[0])
  cadence=fxpar(cubehead,'cadence')
  boxcarpx=boxcarmin
  boxcarpx[2]=fix((boxcarmin[2]*60.)/cadence)

; check that boxcar doesn't exceed cube dimensions
  nx=fxpar(cubehead,'naxis1')
  ny=fxpar(cubehead,'naxis1')
  nt=fxpar(cubehead,'naxis3')
  if (boxcarpx[0] gt nx/2 or boxcarpx[1] gt ny/2 or $
      boxcarpx[2] gt nt/2) then begin
    print,' ##### boxcarpx = '+trimd(boxcarpx)+$
      ' excessive for nx,ny,nt = '+trimd(nx)+trimd(ny)+trimd(nt)
    print,'       decrease boxcarmin in sdo_writeallpairsplines.pro or '+$
      'increase order duration or decrease center cadence'
    print,'       for now boxcarpx set to [3,3,3]'
    boxcarpx=[3,3,3]
  endif

; apply the same boxcar smearing to all cubes
  if (total(boxcarpx) gt 0) then begin
    print,' ----- sdo_writeallpairsplines smears center cubes with boxcar'+$
      trimd(boxcarpx)
    doallfiles,datadir+'/cubes',datadir+'/cubessmear'+boxcarlabel,$
      '.fits','.fits','boxcarfitscube',boxcar=boxcarpx
  endif else spawn,'rm -rf '+datadir+'/cubessmear'+boxcarlabel
endif

; ================= end boxcar business no longer in use

; make driftsdir and gendriftsdir in dir above if not there already 
driftsdir=datadir+'/../driftscenter'+boxcarlabel+'/'
spawn,'mkdir -p '+driftsdir
if (boxcarlabel ne '') then begin
  gendriftsdir=datadir+'/../driftscenter'
  spawn,'mkdir -p '+gendriftsdir
endif

; delete earlier content
spawn,'rm -f '+driftsdir+'/*'

; do all except last pair = xalpair (for check xal plot)
for ipair=0,npairs-2 do begin
  thispair=reform(pairs[*,ipair])

;RR Dec 11 2020 apparently I don't delay anymore, good necause diffderot clash
;; ; set delaymin for euvanchor = !!!!! RR-muck choice 
;;   if (thispair[0] eq euvanchornrs[0] and thispair[1] eq euvanchornrs[1]) $
;;     then delaymin=0 else delaymin=0
;; ; set delaymin for all potential euvanchors = !!!!! RR-muck choice 
;;  if (thispair[0] eq 6 and thispair[1] gt 7) then delaymin=5 else delaymin=0
  delaymin=0
  
; get spline stuff for this pair
  sdo_writepairspline,datadir,thispair,$
    trimbox=trimbox,boxcarlabel=boxcarlabel,delaymin=delaymin,verbose=verbose
endfor

; boxcar smearing: remove smeared cubes, copy smeared driftsdir to generic one
; for further target cube making
if (boxcarlabel ne '') then begin
  spawn,'rm -rf '+datadir+'/cubessmear'+boxcarlabel
  spawn,'cp -upr '+driftsdir+'/* '+ gendriftsdir+'/.'
end

end


; =============== main for testing per IDLWAVE H-c ======================

;; datadir='/media/rutten/SSTDATA/alldata/SST/2014-09-06/sdo/center/'
; this set has very large SDO drifts after eclipse
; also wavpair=[8,9] 1600 > 1700 outlier in X at start 

;; datadir='/home/rutten/data/SDO/2014-06-14-longer/center/'
datadir='/home/rutten/data/SDO/2014-06-14-small/center/' ; standard test

sdo_writeallpairsplines,datadir,xalpair,boxcarmin=boxcarmin,/verbose

end
