; file: sdo_stx_findalignfiles.pro = find shifts SDO-STX time sequences
; init: Dec 31 2017  Rob Rutten  Deil from sdo_sst_findalignfiles.pro
; last: Aug 13 2022  Rob Rutten  Deil

;+
pro sdo_stx_findalignfiles,$
; ----- parameters (mostly for findalignimages.pro)
  obsdate,sdoalignfile,stxalignfile,stxtimesfile,$
  px_stx,angle_stx,cadence_stx,$
  px2asym,driftshifts,wobbleshifts,shiftfor,shiftrev,$
  nxfor,nyfor,nxrev,nyrev,euvcor,$
  stxdriftfile=stxdriftfile,itsample_stx=itsample_stx,$
  skipmetcalf=skipmetcalf,applypx2asym=applypx2asym,$
  inshift12=inshift12,trimboxstx=trimboxstx,$
  smearstx=smearstx,histopstx=histopstx,flatten=flatten,$
  muckdarksdo=muckdarksdo,muckdarkstx=muckdarkstx,$
  muckbrightsdo=muckbrightsdo,muckbrightstx=muckbrightstx,$
  norinse=norinse,$
; ----- extra keywords here
  stxname=stxname,$ 
  driftspline=driftspline,wobblespline=wobblespline,$
  aligndir=aligndir,splinip=splinip,$
  check304ha=check304ha,splineha=splineha,$
  halcfile=halcfile,aia304file=aia304file,$
  undo304_1600=undo304_1600,blink=blink,show=show,verbose=verbose
 ;
 ; find shifts to align SDO target cubefiles to STX image sequence
 ;
 ; DESCRIPTION:
 ;   wrapper calling multiple programs: first findalignimages.pro for
 ;   a selected image pair, including (slow) STX angle and pixel
 ;   corrections unless skipmetcalf=1, then sdo_shiftaligncubefiles.pro to
 ;   determine and correct time-dependent offsets between SDO and STX
 ;   guiding, with "drifts" typically due to STX local-feature
 ;   tracking and "wobble" remaining 0.1 arcsec modulation in SDO
 ;   resulting from the JSOC 1 arcsec cutout sawtooth, then optionally
 ;   time-dependent offsets between AIA-304 and STX Halpha_linecenter
 ;   with question (in the terminal) whether to apply these to all AIA
 ;   EUVs or not (do only if that blink and offset graph look good: it
 ;   works only for scenes with very much activity and good fibrilar
 ;   or filamentary matching which is rare).
 ; EXAMPLES:
 ;   at the end of this file
 ; INPUT PARAMETERS: 
 ;   obsdate = observing date, mandatory format '2016.09.05'
 ;   sdoalignfile = string name SDO fits cubefile for alignment
 ;     (default 'sdo/target/cubes/hmicont.fits')
 ;   stxalignfile = string name STX file for Metcalf alignment
 ;     SST/CRISP: default stxname+'/ha_wb.fits' but may also be La Palma
 ;       cubefile as 'wb.6563.hh:mm:ss.corrected.aligned.icube'
 ;     DST/IBIS: make fitscube file from reddest Halpha wavelength
 ;   stxtimesfile = string name STX timing file (SST idlsave or tai)
 ;   px_stx = pixel size STX data in arcsec
 ;   angle_stx = STX data rotation from solar (X,Y) in degrees NESW (CCW)
 ; OUTPUT PARAMETERS:
 ;   px_stx and angle_stx get improved if skipmetcalf=0
 ;   cadence_stx = mean value of temporal image separation in seconds
 ;   px2asym = px_stx asymmetry (x,y) also gets improved if skipmetcalf=0
 ;   driftshifts,wobbleshifts,shiftfor,shiftrev,nxfor,
 ;     nyfor,nxrev,nyrev,euvcor = alignment results, 
 ;     some also written in aligndir file sdo-stxname-align.dat
 ;   undo304_1600 = 1/0: undo the EUV anchoring to measured 304 > 1600 shifts
 ; OPTIONAL INPUTS:
 ;   ---- first group = keywords of findalignimages.pro
 ;   itsample_stx = best it (sharp STX mid-sequence); default -1 = find)
 ;   stxdriftfile=stxdriftfile: string name STX file for guiding alignment
 ;     SST = stxalignfile (default; MOMFBD gives wb-FP coherence)
 ;     DST/IBIS: despeckled wide-band file (may be angled to IBIS) 
 ;   skipmetcalf = 1/0: determine shift only, no scale + rotate correction
 ;   applypx2asym = 1/0: apply Metcalf-found px2 asymmetry in iteration
 ;     NB: advised for DKIST/IBIS, nor for SST/CRISP
 ;   inshift12 = [xshift,yshift] initial shift to accommodate large offsets;
 ;      see findalignimages.pro
 ;   trimboxstx = integer vector [xmin,ymin,xmax,ymax] to select subfield
 ;   smearstx: smear STX image (default -1 = 1.5*pxratio)
 ;   histopstx: hist_opt STX image with given value (clip limit < 1, top only)
 ;   flatten: subtract boxcar average from each, value = width in coarsest px
 ;     value 5 good for spots or limb in the field.  Default here 0.
 ;   muckdarkstx: set intensities below threshold (units mean=1) to mean
 ;   muckbrightstx: set intensities above threshold (units mean=1) to mean
 ;   norinse = 1/0: do not apply shift-together after Metcalf iteration
 ;   ----- second group = for this program
 ;   stxname = string with STX name, default 'sst'
 ;   driftspline = smooth value (default 1.E3; -1 = no spline, 1.E8 = linear)
 ;   wobblespline = smooth value (default 0.1)
 ;   aligndir = string dir to put results in (default 'sdo_stxname_align')
 ;   splinip = 0: nearest-neighbour = retain coarse SDO pixels (suffix _px) 
 ;           = 1: cubic spline interpolation (suffix _ip)
 ;   check304ha = get alignment correction for aia304 versus STX Ha_lc
 ;   halcfile = string with path/name of STX Ha-linecenter file file
 ;   aia304file =  string with path/name of AIA 304 file file
 ;     (default 'sdo/target/cubes/aia304.fits')
 ;   splineha = smooth value for Ha_lc (default 1.E3*driftspline)
 ;   blink: flickrr initial results (value = duration in seconds, default 0)
 ;   show = 1/0: start forward and reverse showex result blinkers
 ;   verbose = 1/0: more output
 ; OUTPUTS:
 ;   alignment data in dir sdo_stx_align
 ;   aligned cubefiles in /tmp if showex demo requested (show)
 ;   undo304_1600 = 1 if set in answer to question check Ha_lc versus 304
 ; HISTORY:
 ;   Jun  1 2017 RR: from sdo2stx_v1.pro
 ;   Jan  2 2018 RR: option applypx2asym 
 ;   Feb 11 2018 RR: undo304_1600 as answer option
 ;   Dec 29 2019 RR: split drift (STX) and wobble (SDO)
;-

; answer no-parameter query 
if (n_params() lt 14) then begin
  sp,sdo_stx_findalignfiles
  return
endif

; set wall-clock timer (seconds)
timestart=systime(1) 

; print pacifier
if (verbose ne 0) then print,' ----- sdo_stx_findalignfiles starts'

; keyword defaults
; ----- keywords for using findalignimages.pro
if (n_elements(skipmetcalf) eq 0) then skipmetcalf=0
if (n_elements(applypx2asym) eq 0) then applypx2asym=0
if (n_elements(inshift12) eq 0) then inshift12=[0.,0.]
if (n_elements(trimboxstx) eq 0) then trimboxstx=-1 ; NB default
if (n_elements(smearstx) eq 0) then smearstx=-1.5   ; NB default
if (n_elements(histopstx) eq 0) then histopstx=0
if (n_elements(flatten) eq 0) then flatten=0   ; NB default
if (n_elements(muckdarksdo) eq 0) then muckdarksdo=0 
if (n_elements(muckdarkstx) eq 0) then muckdarkstx=0 
if (n_elements(muckbrightsdo) eq 0) then muckbrightsdo=0 
if (n_elements(muckbrightstx) eq 0) then muckbrightstx=0 
if (n_elements(norinse) eq 0) then norinse=0
; ----- extra keywords here
if (n_elements(stxname) eq 0) then stxname='sst'
if (n_elements(itsample_stx) eq 0) then itsample_stx=-1 ; NB default
if (n_elements(stxdriftfile) eq 0) then stxdriftfile=stxalignfile
if (n_elements(sdoalignfile) eq 0) then $
  sdoalignfile='sdo/target/cubes/hmicont.fits'
if (n_elements(aligndir) eq 0) then aligndir='sdo_'+stxname+'_align'
if (n_elements(driftspline) eq 0) then driftspline=1.E3   ; NB default
if (n_elements(wobblespline) eq 0) then wobblespline=0.1  ; NB default
if (n_elements(splinip) eq 0) then splinip=0
if (n_elements(check304ha) eq 0) then check304ha=0
if (n_elements(halcfile) eq 0) then halcfile=''
if (n_elements(aia304file) eq 0) then $
  aia304file='sdo/target/cubes/aia304.fits'
if (n_elements(splineha) eq 0) then splineha=driftspline*1.E3
if (n_elements(undo304_1600) eq 0) then undo304_1600=0
if (n_elements(blink) eq 0) then blink=10
if (n_elements(show) eq 0) then show=0
if (n_elements(verbose) eq 0) then verbose=0

; make aligndir if not yet existing
spawn,'mkdir -p '+aligndir

; get STX timearr_stx in TAI from stxtimesfile
dummy=cgrootname(stxtimesfile,extension=timefileext)  ;RR needs coyotelib
; classical SST 
if (timefileext eq 'sav' or timefileext eq 'idlsave') then begin
  restore,stxtimesfile 
  if (n_elements(time) ne 0) then datetime_stx=obsdate+'_'+time else $
    if (n_elements(csday) ne 0) then begin
    tai_stx=anytim2tai(obsdate)+double(csday)
    datetime_stx=anytim2utc(tai_stx,/ccsds)
  endif
  timearr_stx=anytim2tai(datetime_stx)
endif
; older DST 
if (strmatch(stxtimesfile,'*ibis*') or $
    strmatch(stxtimesfile,'*mxis*')) then begin
  readcol,stxtimesfile,datetime_stx,format='(A)' 
  timearr_stx=anytim2tai(datetime_stx)
endif
; SSTRED, GONG, ALMA: TAI with 'tai' in filenames
if (strmatch(stxtimesfile,'*tai*')) then $
  readcol,stxtimesfile,timearr_stx,format='(D)'
nt_stx=n_elements(timearr_stx)
cadence_stx=(timearr_stx[nt_stx-1]-timearr_stx[0])/(nt_stx-1)
if (nt_stx eq 0) then begin
  print,' ##### sdo_stx_findalignfiles abort: STX timing sequence empty' 
  retall
endif

; get stxalignfile extension to find its type
dummy=cgrootname(stxalignfile,extension=ext)  ;RR needs coyotelib

; fits file: check 3 dimensions (Apr 8 2022 Kuridze CHROMIS problem)
if (ext eq 'fits') then begin
  stxhead=headfits(stxalignfile)
  ndims=fxpar(stxhead,'naxis') 
  if (ndims ne 3) then begin
    print,' ##### sdo_stx_alignfiles aborts: '+$
      stxalignfile+' has'+trimd(ndims)+$
      'instead of 3 - if SSTRED first sanitize'
    return
  endif
  nt_file=fxpar(stxhead,'naxis3') 
  if (nt_file ne nt_stx) then begin
    print,' ##### sdo_stx_alignfiles aborts: '+$
      stxalignfile+' has nxaxis3 ='+trimd(nt_file)+$
      'but stxtimesfile has nt_stx ='+trimd(nt_stx)
  endif
endif

; if default find itsample_stx from STX rms 
  if (itsample_stx eq -1) then begin
    if (verbose ne 0 and nt_stx gt 100) then print,$
      ' ----- find itsample_stx takes long; better note result'
    findrmscubefile,stxalignfile,maxrms,itsample_stx,$
      trimbox=trimboxstx,rmsplot=show,show=show
  endif

; check itsample_stx
  if (itsample_stx gt nt_stx-1) then begin
    print,' ##### sdo_stx_findalignfiles abort: itsample_stx > nt_stx-1'
    retall
  endif

; get STX align image from stxalignfile (fits or La Palma)
  if (ext eq 'fits') then $
    im_stx=readfitscube(stxalignfile,trange=[itsample_stx,itsample_stx])
  if (ext eq 'icube' or ext eq 'fcube' or ext eq 'bcube') then $
    im_stx=readsstfile(stxalignfile,trange=[itsample_stx,itsample_stx])

; get SDO parameters from cube file header
  header_sdo=headfits_rr(sdoalignfile)
  header_sdosize=(1+fix(n_elements(header_sdo)/36.))*2880
  bitpix_sdo=fxpar(header_sdo,'bitpix')
  nx_sdo=fxpar(header_sdo,'naxis1') 
  ny_sdo=fxpar(header_sdo,'naxis2') 
  nt_sdo=fxpar(header_sdo,'naxis3') 
  xcen_sdo=fxpar(header_sdo,'xcen')  ; at cube start, rotates differentially
  ycen_sdo=fxpar(header_sdo,'ycen')
  rsun_obs=fxpar(header_sdo,'rsun_obs')

;  get SDO time array from keywords (present in RJR-made SDO cubes)
  starttim_sdo=fxpar(header_sdo,'starttim')
  cadence_sdo=fxpar(header_sdo,'cadence')
  if (starttim_sdo eq 0 or cadence_sdo eq 0) then begin
    print, ' ##### SDO input file has no STARTTIM or CADENCE in header'
    return
  endif
  timearr_sdo=indgen(nt_sdo)*cadence_sdo+anytim2tai(starttim_sdo)

; check that timearr_stx sits within timearr_sdo
  if (timearr_stx[0]-timearr_sdo[0] lt 0) then $
        print,' !!!!! sdo_stx_findalignfiles WARNING: SDO started after STX'
  if (timearr_stx[nt_stx-1]-timearr_sdo[nt_sdo-1] lt 0) then $
          print,' !!!!! sdo_stx_findalignfiles WARNING: SDO ended before STX'

; find closest SDO to itsample_stx and interpolate SDO straddle image pair
;RR straddle overdoes, eg hmicont virtually no change since derotated
  tstxbest=timearr_stx[itsample_stx]
  dummy=min(abs(timearr_sdo-tstxbest[0]),itsample_sdo)  ;f##k IDL: [0]
  if (timearr_sdo[itsample_sdo] lt tstxbest) then begin
    it1sample_sdo=itsample_sdo
    it2sample_sdo=itsample_sdo+1
  endif else begin
    it1sample_sdo=itsample_sdo-1
    it2sample_sdo=itsample_sdo
  endelse
  impair=readfitscube(sdoalignfile,trange=[it1sample_sdo,it2sample_sdo])
  time1=timearr_sdo[it1sample_sdo]
  time2=timearr_sdo[it2sample_sdo]
  p=[[[impair[*,*,0]]],[[impair[*,*,1]]]]
  timefrac=(tstxbest[0]-time1)/(time2-time1)
  im_sdo=interpolate(p,timefrac)  ;RR bilinear, cubic no difference for 2)
  datetime_sdo=anytim2utc(tstxbest[0],/ccsds)

; check this is good SDO quality; if not break with warning
  momim_sdo=moment(im_sdo)
  rms_sdo=sqrt(momim_sdo[1])
  findrmscubefile,sdoalignfile,maxrms_sdo,itbest_sdo,$
    itskip=fix(nt_stx/20.),rmsrms=rmsrms_sdo
  if (rms_sdo lt maxrms_sdo-0.5*rmsrms_sdo) then begin
    findrmscubefile,sdoalignfile,maxrms_sdo,itbest_sdo,$
      itskip=fix(nt_stx/20.),rmsrms=rmsrms_sdo,rmsplot=1,show=1
    print,' ##### abort: seems SDO was not sharp at itsample_sdo ='+$
      trimd(itsample_sdo)+' corresponding to itsample_stx ='+$
      trimd(itsample_stx)
    print,' @@@@@ inspect both manually and specify better itsample_stx'
    return
  endif

  if (verbose) then print,' ----- itsample_stx ='+trimd(itsample_stx)+$
    '  itsample_sdo ='+trimd(itsample_sdo) 

;; ; write sample images to subdir tests for trial findalignimages.pro
;; writefits,'/tmp/im_sample_sst.fits',im_stx
;; writefits,'/tmp/im_sample_sdo.fits',im_sdo
;; sv,im_stx
;; sv,im_sdo
;; STOP

; do full alignment for this "best" image pair
  px_sdo=0.6
  findalignimages,im_sdo,im_stx,px_sdo,px_stx,angle_stx,px2asym,$
    shiftfor,shiftrev,nxfor,nyfor,nxrev,nyrev,$    
    nxmuckfor=nxmuckfor,nymuckfor=nymuckfor,$
    nxmuckrev=nxmuckrev,nymuckrev=nymuckrev,$
    skipmetcalf=skipmetcalf,applypx2asym=applypx2asym,$
    trimboxim2=trimboxstx,inshift12=inshift12,$$
    smearim2=smearstx,histopim2=histopstx,$
    muckdarkim1=muckdarksdo,muckdarkim2=muckdarkstx,$
    muckbrightim1=muckbrightsdo,muckbrightim2=muckbrightstx,$
    norinse=norinse,$
    flatten=flatten,blink=blink,show=show,verbose=verbose

; get location parameters to write in sdo-stxname-align.dat
  nx_stx=nxfor
  ny_stx=nyfor
  deltat=timearr_stx[itsample_stx]-timearr_sdo[0]
  rotated=rot_xy(xcen_sdo,ycen_sdo,deltat)
  solx_stx=rotated[0]+inshift12[0]*0.6    ; STX (X,Y) at sample time
  soly_stx=rotated[1]+inshift12[1]*0.6
  angle_limb=limbangle(solx_stx,soly_stx)
  robs=sqrt(solx_stx^2+soly_stx^2)
  if (robs lt rsun_obs) then theta=asin(robs/rsun_obs) else theta=!pi/2
  muobs=abs(cos(theta))

; print results
  if (verbose ne 0) then print,' ----- results:'+$
    '  px_stx ='+trimd(px_stx,4)+'  angle_stx ='+trimd(angle_stx,2)+$
    '  (X,Y) = '+trimd(solx_stx,1)+','+trimd(soly_stx,1)+$
    '  mu = '+trimd(muobs)+'  angle_limb ='+trimd(angle_limb,1)

; set congridfactors as under @@@@@ demo in findalignimages.pro
  px1=0.6  ; AIA px
  px2=px_stx
  px2d=px2*px2asym
  pxratio=float(px1)/px2
  pxrat2d=float(px1)/px2d
  pxscale1=[1.0,1.0]
  pxscale2=[1.0,1.0]
  if (pxratio lt 0.999) then begin
    if (applypx2asym eq 0) then pxscale2=1./[pxratio,pxratio] $
    else pxscale2=1./pxrat2d ; px2 smaller than px1
  endif
  if (pxratio gt 1.001) then begin
    ; rescale im1 to mean im2 pixel size 
    pxscale1=[pxratio,pxratio]  
    if (applypx2asym eq 1) then pxscale2=px2asym   
  endif

; if applypx2asym write dewarped version of stxalignfile into /tmp
  if (applypx2asym eq 0) then stxalignforfile=stxalignfile else begin
    stxalignforfile=file_basename(stxalignfile,'.'+ext)
    stxalignforfile='/tmp/'+stxalignforfile+'_for_ip.fits'
    reformcubefile,stxalignfile,stxalignforfile,scale=pxscale2,splinip=1
  endelse

; reform both files forward to temp files with splinip=1 for 
; subpixel cross-alignment in drift determination

; get two unique muck filenames
  rannr=fix(abs(randomn(seed))*10000)
  sdomuckfile='/tmp/sdo_muck'+$
    strmid(string(rannr+1E5,format='(i6)'),2)+'.fits'
  stxmuckfile='/tmp/'+stxname+'_muck'+$
    strmid(string(rannr+1E5,format='(i6)'),2)+'.fits'

; reform sdoalignfile = rotate, shift, cut, put on STX timing, muck
; first check no /tmp file already (less likely since intro random)
  if (file_test(sdomuckfile)) then begin
    print,' ##### WARNING sdo_stx_findalignimages'
    print,' ##### file '+sdomuckfile+' exists: runnning two instances?'
    print,' ##### the older such /tmp files get overwritten now!'
  endif
  if (verbose) then print,' ----- write '+sdomuckfile+$
    ' may take long; faster with trimboxstx?'
  reformcubefile,sdoalignfile,sdomuckfile,$
    congridfactor=pxscale1,rotate=angle_stx,shift=shiftfor,$
    cutcentralx=nxmuckfor,cutcentraly=nymuckfor,missingvalue=-1,$
    muckdark=muckdarksdo,muckbright=muckbrightsdo,flatten=flatten*pxratio,$
    timearr_out=timearr_stx,splinip=1

; reform stxdriftfile = dewarp, cut and muck for better scene similarity
; (nxmuckfor,nymuckfor from findalignimages takes care of trimbox)
  if (verbose) then print,' ----- write '+stxmuckfile+$
    ' may take yet longer'  ;RR finer pixels
  if (smearstx lt 0) then smearstx=abs(smearstx)*pxratio
  if (stxdriftfile eq stxalignfile) then stxdriftforfile=stxalignforfile $
  else begin
    stxdriftforfile='/tmp/'+stxname+'driftforfile.fits'
    reformcubefile,stxdriftfile,stxdriftforfile,scale=pxscale2,splinip=1
  endelse
  reformcubefile,stxdriftforfile,stxmuckfile,$
    cutcentralx=nxmuckfor,cutcentraly=nymuckfor,$
    smear=smearstx,histopttop=histopstx,flatten=flatten*pxratio,$
    muckdark=muckdarkstx,muckbright=muckbrightstx,splinip=1,$  ;; ,$
    flt2int=1  ;RR Dec 23 2019 problem SST/2018-06-10-vasco

; cross-align these temporary cubefiles to find time-dependent drifts
; if no optional output files then no inspection below
  if (verbose) then print,' ----- sdo_shiftaligncubefiles may take long' 
  sdo_shiftaligncubefiles,stxmuckfile,sdomuckfile,$
    driftshifts,wobbleshifts,$
;;  outfile1='/tmp/'+stxname+'_xal.fits',outfile2='/tmp/sdo_xal.fits',$
    niter=3,cutiter=0.05,$
    driftspline=driftspline,wobblespline=wobblespline,nspliter=2,$
    plotdir=aligndir

;; ; check mucked-pair alignment (needs getting output files above) 
;; showex,'/tmp/'+stxname+'_xal.fits','/tmp/sdo_xal.fits'
;; STOP

; get total forward shift array on the rescaled STX pixel scale
; correct original it_sample shift which was total instead of to drift only
  sdoshifts=fltarr(nt_stx,2)
;; shiftfordrift=shiftfor-driftshifts[itsample_stx,*]
  shiftfordrift=shiftfor-wobbleshifts[itsample_stx,*]
  for it=0,nt_stx-1 do begin
    driftshifts[it,0]=driftshifts[it,0]+shiftfordrift[0]
    driftshifts[it,1]=driftshifts[it,1]+shiftfordrift[1]
    sdoshifts[it,0]=driftshifts[it,0]+wobbleshifts[it,0]
    sdoshifts[it,1]=driftshifts[it,1]+wobbleshifts[it,1]
  endfor

; ======== option: find EUV corrections from cross-alignment aia304 to ha_lc

  euvcor=fltarr(nt_stx,2)
  if (check304ha ne 0 and halcfile ne '') then begin

; reform and muck aia304 for crossalignment (no flatten)
; also apply the above time-dependent drift correction
    reformcubefile,aia304file,$
      '/tmp/aia304_muck.fits',$
      congridfactor=pxscale1,rotate=angle_stx,shift=sdoshifts,$
      cutcentralx=nxmuckfor,cutcentraly=nymuckfor,$
      missingvalue=-1,$
      muckdark=muckdarksdo,muckbright=muckbrightsdo,flatten=0,$
      timearr_out=timearr_stx,splinip=1

; if applypx2asym write dewarped Ha_lc _for_ip file in /tmp
    if (applypx2asym eq 0) then halcforfile=halcfile else begin
      halcforfile=file_basename(halcfile,'.'+ext)
      halcforfile='/tmp/'+halcforfile+'_for_ip.fits'
      reformcubefile,halcfile,halcforfile,scale=pxscale2,splinip=1
    endelse

; muck (dewarped) Ha_lc cube for better scene similarity (no flatten)
    if (smearstx lt 0) then smearstx=abs(smearstx)*pxratio
    reformcubefile,halcforfile,'/tmp/halc_muck.fits',$
      trimbox=trimboxstx,$
      cutcentralx=nxmuckfor,cutcentraly=nymuckfor,$
      smear=smearstx,histopttop=histopstx,flatten=0,$
      muckdark=muckdarkstx,muckbright=muckbrightstx,splinip=1

; blink input
    if (blink ne 0) then begin
      print,' ----- showblink: input mucked STX halc versus AIA 304'
      showex,'/tmp/halc_muck.fits',$
        '/tmp/aia304_muck.fits'
    endif

; shiftalign the two to get time-dependent shifts and spline plot
; get the optional outputs for inspection
    sdo_shiftaligncubefiles,'/tmp/halc_muck.fits','/tmp/aia304_muck.fits',$
      xal304hashifts,nowobbleshifts,$
      outfile1='/tmp/halc_xal.fits',outfile2='/tmp/aia304_xal.fits',$
      niter=3,cutiter=0.05,$
      driftspline=splineha,wobblespline=0,nspliter=2,plotdir=aligndir

; blink result
    if (blink ne 0) then begin
      print,' ----- showblink: aligned mucked STX halc versus AIA 304'
      showex,'/tmp/halc_xal.fits',$
        '/tmp/aia304_xal.fits'
    endif

; correct these shifts from trimbox to original frame ;RR ?? may be wrong
    if (trimboxstx[0] ne -1) then begin
      corha304=[trimboxstx[0],trimboxstx[1]]$
        -[(nxfor-nxmuckfor)/2.,(nyfor-nymuckfor)/2.]
      for it=0,nt_stx-1 do xal304hashifts[it,*]=xal304hashifts[it,*]+corha304[*]
    endif

; ask how to continue (seems fairly stupid IDL manners)
ASKUSER:
    read,answer,prompt=' !!!!! type to do: 1 = apply splined 304-ha_lc, 2 = apply mean of splined 304-ha_lc, 3 = do not apply 304-ha_lc; 4 = apply undo304_1600 instead, 5 = quit here?  TYPE HERE: '
    if (answer eq 1) then euvcor=xal304hashifts
    if (answer eq 2) then begin 
      euvcor[*,0]=avg(xal304hashifts[*,0])
      euvcor[*,1]=avg(xal304hashifts[*,1])
    endif
    if (answer eq 3) then goto, JUSTCONTINUE 
    if (answer eq 4) then begin
      undo304_1600=1
      goto, JUSTCONTINUE
    endif
    if (answer eq 5) then retall
    if (answer ne 1 and answer ne 2 and answer ne 3 and answer ne 4) then begin
      print,' ##### invalid answer; try again'
      goto, ASKUSER 
    endif

JUSTCONTINUE:
    print,' ------ sdo_stx_findalignfiles continues'
  endif ; ==============  end of optional 304 to ha_lc shift correction

; ============== write results (always)

; resulting shifts yet without euvcor (write separately for split UV and EUV)
  shiftfor=sdoshifts  
  shiftrev=-rotatevec(shiftfor,-angle_stx)

; write results in align file for future usage (use unit nr to append)
; add explanation text after variables per line for direct reading
  outfile=aligndir+'/sdo-'+stxname+'-align.dat'
  openw,42,outfile   ;RR answer to everything
  printf,42,trimd(angle_stx)+trimd(px_stx,4)+trimd(px2asym,4)+$
    ' = angle_stx  px_stx  xasym ysasym'
  printf,42,trimd(nt_stx)+trimd(cadence_stx)+trimd(nxfor)+trimd(nyfor)+$
    trimd(nxrev)+trimd(nyrev)+$
    ' = nt_stx  cadence_stx  nxfor nyfor  nxrev nyrev'
  printf,42,trimd(solx_stx)+trimd(soly_stx)+trimd(angle_limb)+trimd(muobs)+$
    ' = solX  solY  angle_limb  mu'
  printf,42,trimd(itsample_stx)+trimd(itsample_sdo)+$
    ' = itsample_stx  itsample_sdo; next line datetime[itsample_sdo]'
  printf,42,string(datetime_sdo)
  xdrift=driftshifts[*,0]
  ydrift=driftshifts[*,1]
  xwobble=wobbleshifts[*,0]
  ywobble=wobbleshifts[*,1]
  xshiftfor=shiftfor[*,0] ; this is total shift
  yshiftfor=shiftfor[*,1]
  xeuvcor=euvcor[*,0]
  yeuvcor=euvcor[*,1]
  writecol,outfile,timearr_stx,xdrift,ydrift,xwobble,ywobble,$
    xshiftfor,yshiftfor,xeuvcor,yeuvcor,$
    fmt='(1E20.12,8F10.4)',filnum=42
  close,42

; ========= essentially this program is now done =================

;   rest: write forward and reverse align fitscubes and showblink them;
;     no 304_Ha or undo_304_1600 corrections assuming VUV sdoalignfile 
;     (hmicont, hmimag, aia1700, aia1600).

; ======= forward result 

; showblink forward result
  if (show ne 0) then begin

; write splinipped forward SDO alignfile in /tmp
    sdoalignforfile=file_basename(sdoalignfile,'.fits')
    sdoalignforfile='/tmp/'+sdoalignforfile+'_for_ip.fits'
    if (verbose) then print,' ----- writing show sdoalignforfile = '+$
      sdoalignforfile
    reformcubefile,sdoalignfile,sdoalignforfile,$
      rotate=angle_stx,congridfactor=pxscale1,$
      shift=shiftfor,cutcentralx=nxfor,cutcentraly=nyfor,$
      timearr_out=timearr_stx,missingvalue=-1,splinip=1

    print,' ----- showblink: forward result align files'
    showex,sdoalignforfile,stxalignforfile,trimbox=trimboxstx,/blink
  endif

; ===== inspect reverse result

; showblink reverse result
  if (show eq 1) then begin

; complication: split reverse shifts between SDO and STX in order to
;   maintain STX field center in the rotated reverse display: 
;   - backshift SDO over temporal mean (integer for splinip=0 usage) 
;   - shift STX only over the remaining modulation

; get temporal mean reverse shift and modulation around mean
    meanshiftrev=[fix(mean(shiftrev[*,0])+0.5),fix(mean(shiftrev[*,1])+0.5)]
    modshiftrev=[[shiftrev[*,0]-meanshiftrev[0]],$
                 [shiftrev[*,1]-meanshiftrev[1]]]

; Apr  8 2022: check SDO cutout not too small (Kuridze CHROMIS problem)
    headsdo=headfits(sdoalignfile)
    nxsdo=fxpar(headsdo,'naxis1')
    nysdo=fxpar(headsdo,'naxis2')
    if (nxsdo lt nxrev or nysdo lt nyrev) then $
      print,' !!!!! sdo_stx_findalignimages has images in SDO file '+$
      sdoalignfile+' too small for reverse STX image (nx,ny) ='+$
      trimd(nxrev)+','+trimd(nyrev)
    
; write splinipped reverse SDO alignfile in /tmp
    sdoalignrevfile=file_basename(sdoalignfile,'.fits')
    sdoalignrevfile='/tmp/'+sdoalignrevfile+'_rev_ip.fits'
    if (verbose) then print,' ----- writing show sdoalignrevfile = '+$
      sdoalignrevfile
    reformcubefile,sdoalignfile,sdoalignrevfile,shift=-meanshiftrev,$ 
      congridfactor=pxscale1,cutcentralx=nxrev,cutcentraly=nyrev,$
      timearr_out=timearr_stx,missingvalue=-1,splinip=1

; write (if applypx2asym then dewarped) STX_rev file in /tmp
    stxalignrevfile=file_basename(stxalignfile,'.fits')
    stxalignrevfile='/tmp/'+stxalignrevfile+'_rev_ip.fits'
    if (verbose) then print,' ----- writing show stxalignrevfile = '+$
      stxalignrevfile
    reformcubefile,stxalignforfile,stxalignrevfile,$
      shift=modshiftrev,rotate=-angle_stx,$  
      nxlarge=nxrev,nylarge=nyrev,cutcentralx=nxrev,cutcentraly=nyrev,$
      timearr_in=timearr_stx,missingvalue=-1,splinip=1

    print,' ----- showblink: reverse result for align files'
    showex,sdoalignrevfile,stxalignrevfile,trimbox=trimboxstx,/blink
  endif

; print elapsed time
  timedone=systime(1)
  timelaps=ntostr((timedone-timestart)/60.,format='(F11.1)')
  if (verbose ne 0) then $
    print,' ----- sdo_stx_findalignfiles took '+timelaps+' minutes'

end

; ================= main for testing per IDLWAVE hyper ==================

;; ; data DST/IBIS from Tom Schad
;; cd,'/media/rutten/RRDATA/alldata/DST/2017-06-14-schad/'
;; stxname='ibisA'
;; obsdate='2017.06.14'
;; stxalignfile='ibisA/ibis_wb_6563_20170614_151541.fits'
;; stxdriftfile='ibisA/
;; stxtimesfile='ibisA/ibis_times_6563_20170614_151541.txt'
;; angle_stx=-0.53        ; after Metcalf
;; px_stx=0.0956          ; after Metcalf 
;; px2asym=[0.9888,1.0112]  ; from Metcalf
;; histopstx=0.1          ; muck: cut Halpha network and moat bright points
;; trimboxstx=[200,200,800,800]   ; conservative, from ibisscans2files.pro
;; splinestx=1.E1         ; low value to follow erratic DST guiding jumps
;; skipmetcalf=1   ; after Metcalfs
;; applypx2asym=1  ; NEEDED for IBIS     
;; check304ha=0        
;; blink=0
;; show=1
;; verbose=1 

;; ; limb data Gregal (takes ages since very long duration)
;; cd,'/media/rutten/SSTDATA/alldata/SST/2014-08-29-gregal'
;; obsdate='2014.08.29'
;; stxtimesfile='crispex/tseries.6563.15:40:09.calib.sav'
;; stxalignfile='crispex/wb.6563.15:40:09.corrected.fits'
;; angle_stx=131.57  ; Dec  8 2017 from 4 Metcalfs
;; px_stx=0.0575     ; Dec  8 2017 from 4 Metcalfs
;; inshift12=[-2.6,-12.8] ; output from earlier run without inshift12
;; ;;; histopstx=0.1  ; do not muck Ha BPs since cont plage bright at limb
;; ;;; trimboxstx=[330,330,730,740] ; no longer since autodetect triangles
;; splinestx=1.E2      ; needs lower value for shorter duration?
;; check304ha=0        ; trial; you get question to do or not to do
;; undo304_1600=0      ; check driftplot
;; skipmetcalf=1       ; do not refine angle and px per Metcalf since earlier
;; blink=3
;; verbose=1
;; show=1

;; ;RR very difficult featureless near-limb scene including limb
;; cd,'/media/rutten/SSTDATA/alldata/SST/2014-06-19-limb/07:27:10'
;; stxname='sst'
;; obsdate='2014.06.19'
;; stxalignfile='sst/ha_wb.fits'
;; halcfile='sst/ha_lc.fits'
;; stxtimesfile='sst/tseries.6563.07:27:10.calib.sav'
;; angle_stx=63.8
;; px_stx=0.0577
;; histopstx=0.01
;; splinestx=1.E2
;; skipmetcalf=0
;; check304ha=0
;; itsample_stx=495 ; from earlier run without it_sample
;; blink=10
;; show=1
;; verbose=1

; standard SST testdata = sunspot
;; cd,'/home/rutten/data/SST/2016-09-05-demo/'
cd,'/media/rutten/RRHOME/alldata/SST/2016-09-05-demo'
stxname='sst'
obsdate='2016.09.05'
stxalignfile='crispex/wb.6563.09:48:31.corrected.aligned.icube'
halcfile='sst/ha_lc.fits'  ; already greybordered
stxtimesfile='crispex/tseries.8542.09:48:31.calib.sav'
angle_stx=98.16    ; improved from from run with skipmetcalf=0
px_stx=0.0567
histopstx=0.1             ; cut network and moat BPs
driftspline=0
wobblespline=0

skipmetcalf=1
check304ha=0
itsample_stx=29 ; from earlier run without it_sample
blink=5
show=0
verbose=1

sdo_stx_findalignfiles,$
; ----- parameters findalignimages
  obsdate,sdoalignfile,stxalignfile,stxtimesfile,$
  px_stx,angle_stx,$
  cadence_stx,px2asym,driftshifts,wobbleshifts,shiftfor,shiftrev,$
  nxfor,nyfor,nxrev,nyrev,euvcor,$
  itsample_stx=itsample_stx,stxdriftfile=stxdriftfile,$
  skipmetcalf=skipmetcalf,applypx2asym=applypx2asym,$
  inshift12=inshift12,trimboxstx=trimboxstx,$
  smearstx=smearstx,histopstx=histopstx,flatten=flatten,$
  muckdarksdo=muckdarksdo,muckdarkstx=muckdarkstx,$
  muckbrightsdo=muckbrightsdo,muckbrightstx=muckbrightstx,$
  norinse=norinse,$
; ----- extra keyword inputs of this program
  stxname=stxname,$
  driftspline=driftspline,wobblespline=wobblespline,$
  aligndir=aligndir,splinip=splinip,$
  check304ha=check304ha,splineha=splineha,$
  halcfile=halcfile,aia304file=aia304file,$
  undo304_1600=undo304_1600,blink=blink,show=show,verbose=verbose

end


