; file: sdo_shiftjsoc.pro
; init: Jul  3 2020  Rob Rutten  Deil
; last: Nov 30 2020  Rob Rutten  Deil

;+
pro sdo_shiftjsoc,image,index,$
  tairef,xcenarcref,ycenarcref,cenxpxref,cenypxref,sdoshift,$
  outimage,outshift,notrack=notrack,diffdesolrot=diffdesolrot

 ;  shift JSOC im_patch cutout to:
 ;   - undo JSOC tracked im_patch cutout pixel quantization
 ;   - shift over the border cropping in sdo_makereffitscube.pro
 ;   - correct sdoshift evaluated from driftscenter co-align splines
 ;   - optional diffdesolrot: differentially derotate target field 
 ;
 ; INPUTS:
 ;   image = "level2" SDO image (NB: RR "level2" = SDO level 1.5)
 ;   index = "level2" index structure
 ;   tairef = reference ("ref", default 171) cube start time in tai
 ;   xcenarcref, ycenarcref: solar (X,Y) of ref (HMI) image center at start
 ;   cenxpxref,cenypxrev: nx/2, ny/2 of border-cropped ref cube
 ;   sdoshift: 2-elem SDO crossalign vector from driftscenter splines
 ;
 ; KEYWORD INPUT:
 ;   notrack = 1/0: JSOC did not track rotation in im_patch cutting out
 ;   diffdesolrot = 1/0 differential derotation
 ;
 ; OUTPUTS:
 ;   outimage: (differentially) derotated, SDO-shifted, border-shifted image
 ;   outshift: 2-elem shift vector for image center
 ;
 ; HISTORY:
 ;   Jul  3 2020 Rob Rutten start   
;-

; answer no-parameter query 
if (n_params(0) lt 10) then begin
  sp,sdo_shiftjsoc
  return
endif

; keyword defaults
if (n_elements(notrack) eq 0) then notrack=0
if (n_elements(diffdesolrot) eq 0) then diffdesolrot=0

; image dimensions
szim=size(image)
nx=szim[1]
ny=szim[2]

; more starters
sdopx=0.6
taiim=anytim2tai(index.t_obs)
blank=min(image)+0.1*(max(image)-min(image)) ; shift-in dark grey

; ========== shift whole field 

wholerot=[0,0]
outshift=fltarr(2)
xsolim=xcenarcref ; 1st reference image center solar (X,Y) coordinates
ysolim=ycenarcref
if (taiim ne tairef and notrack eq 0) then begin
  newpos=rot_xy(xcenarcref,ycenarcref,tstart=tairef,tend=taiim)
  newpos=reform(newpos)
  if (min(newpos) lt -3000) then begin ; off_limb rot_xy return 
    print,' ##### sdo_shiftjsoc warning: rotated image center off limb'
    newpos=[xsolim,ysolim]
  endif
  wholerot=newpos-[xsolim,ysolim]  ; rotation part
  xsolim=newpos[0]                 ; rotated image center (X,Y) coordinates 
  ysolim=newpos[1]
endif
cenxpxcorr=xsolim/index.cdelt1+index.crpix1-0.5 
    ; xsol/index.cdelt1 = current solar X of image center in px
    ; index.crpix1 = x px distance image llpx (1,1) to sun center
    ; (when JSOC tracks its im_patch cutting this jumps accordingly) 
    ; -0.5 for WCS definition (1.1) = center first lower-left pixel
    ; sign is reversed between the two
    ; sum = sum rotation correction to JSOC cutout + half image width
cenypxcorr=ysolim/index.cdelt2+index.crpix2-0.5
outshift[0]=cenxpxcorr-cenxpxref+sdoshift[0]
    ; JSOC tracked cutout taking follows rotation with full pixel jumps
    ; cenxpxcorr-cenxpxref = sub-pixel correction JSOC cut + border shift
    ; border shift = 2x border = difference nx ref and nx this image
    ; sdoshift = offset given by driftscenter offset this wav from HMI
outshift[1]=cenypxcorr-cenypxref+sdoshift[1]
fixshift=outshift-wholerot  ; subtract rotation part
if (diffdesolrot eq 0) then $
  outimage=shift_img(image,-outshift,missing=blank) else $
  shiftim=shift_img(image,-fixshift,missing=blank)

; ========= optional diffdesolrot = derotate shiftim differentially

if (diffdesolrot ne 0 and notrack eq 0) then begin

; get fixshifted xcen, ycen
  xcen=xcenarcref+fixshift[0]*sdopx
  ycen=ycenarcref+fixshift[1]*sdopx
  
; diffdesolrotate  ;RR Dec  1 2020 does this need /dedesolrot??
  diffim=diffdesolrotate(shiftim,0.6,xcen,ycen,taiim,tairef)
  
  ;; ; check differential derot against fixed derot (undo split above)
  ;; showex,outimage,diffim
  ;; STOP

; overwrite the non-diff result above
  outimage=diffim

endif ; end of diffdesolrot derotation

end


; =============== main for testing per IDLWAVE H-c ======================

;; cd,'/home/rutten/data/SDO/2014-06-14-small/center'
;; cd,'/home/rutten/data/SDO/2014-06-14-small/target'
;; cd,'/home/rutten/data/SDO/2019-11-11-transit/midpointfull/target'
;; cd,'/home/rutten/data/SDO/2019-11-11-transit/complete_notrack/target'
;; cd,'/media/rutten/RRHOME/alldata/SDO/2019-11-11-transit/full_long_notrack/target'
;; cd,'/media/rutten/RRHOME/alldata/SDO/2019-11-11-transit/full_long/target'
cd,'/media/rutten/RRHOME/alldata/SDO/2019-11-11-transit/full_long/target'

; input params
notrack=0
diffdesolrot=1
sdoshift=[1,1] 

; reference data
refheader=headfits_rr('cubes/aia171.fits')
nxref=fxpar(refheader,'naxis1')
nyref=fxpar(refheader,'naxis2')
cenxpxref=nxref/2.-0.5   ; center of FOV center px in IDL px index
cenypxref=nyref/2.-0.5   
starttim=fxpar(refheader,'starttim')
tairef=anytim2tai(starttim)
xcenarcref=fxpar(refheader,'xcen')  
ycenarcref=fxpar(refheader,'ycen')  

; get "slave" file 
;; files=file_search('level2/*_0304.fits*')
;; files=file_search('level2/*_cont.fits*')
files=file_search('level2/*_0193.fits*')
nfiles=n_elements(files)
spawn,'funpack -D '+files[nfiles-1]
;; spawn,'funpack -D '+files[0]
;; files=file_search('level2/*_0304.fits') ; uncompressed ones
;; files=file_search('level2/*_cont.fits') ; uncompressed ones
files=file_search('level2/*_0193.fits') ; uncompressed ones
nfiles=n_elements(files) 
read_sdo,files[nfiles-1],index,image,/uncomp_delete
;; read_sdo,files[0],index,image,/uncomp_delete

; rescale standard
;; image=sdo_intscale(index,image,'304',clipmin=-1,clipmax=-1)
;; image=sdo_intscale(index,image,'cont',clipmin=-1,clipmax=-1)
image=sdo_intscale(index,image,'193',clipmin=-1,clipmax=-1)

sdo_shiftjsoc,image,index,$
  tairef,xcenarcref,ycenarcref,cenxpxref,cenypxref,sdoshift,$
  outimage,outshift,notrack=notrack,diffdesolrot=diffdesolrot

; check
print,' ----- outshift ='+trimd(outshift)
showex,image,outimage,/blink

; center 750x750 arcsec = 1250x1250 px but image 1488x1488, border 50
; AHA HMI 750x750 = 1500x1500 px, somewhere down to 1488x1488
; these are aia_prepped so px scale is 0.6 but image file kept px size?
; at end of duration whole sun is shifted left from JSOC tracking
; therefore black shift-in blanks strips only at left side
; better with notrack but then no derot since no derot

end

