; file: sdo_prep.pro
; init: Feb 25 2016  Rob Rutten  Deil
; last: Apr 13 2022  Rob Rutten  Deil

;+
pro sdo_prep,datadir,$
  refwav=refwav,verbose=verbose

 ;  aia_prep all SDO fits files in datadir/level1
 ;  using the refwav files as successive sliding pointing reference
 ;  for spatial co-alignment of the different channels (wavelengths)
 ;
 ; INPUTS:
 ;   datadir: string with path/dirname, e.g. center, with subdir level1
 ;
 ; OPTIONAL KEYWORD INPUTS:
 ;   refwav: string defining reference SDO channel, default '171'
 ;   verbose=1/0: more printout
 ;
 ; OUTPUTS:
 ;   level 1.5 fits files in datadir/level2
 ;   
 ; HISTORY:
 ;   Feb 25 2016 RR: start = merge sdo_prep_cuts and sdo_prep_sets
 ;   Aug 26 2016 RR: add noheadpnt = crpix shifts from MP table 
 ;   Jan  3 2017 RR: scale and rotation always from MP master table
 ;   Mar  3 2017 RR: no MP scale and rotation for HMI
 ;   Feb 10 2020 RR: option useheadpnt/noheadpnt out
 ;   Feb 19 2020 RR: temporary Slater fix aia_prep, then out
 ;   Sep 15 2020 RR: Gregal Vissers fix call aia_prep: /no_mpo_update
 ;   Mar  8 2022 RR: /no_mpo_update out, /all_versions
 ;   Apr 13 2022 RR: /no_mpo_update back into call aia_prep.pro
;-

; answer no-parameter query 
if (n_params(0) lt 1) then begin
  sp,sdo_prep
  return   
endif

; defaults for keywords
if (n_elements(refwav) eq 0) then refwav='171'
if (n_elements(verbose) eq 0) then verbose=0

; set wall-clock timer (seconds)
timestartprep=systime(1) 

; print pacifier
if (verbose ne 0) then print,' ===== sdo_prep.pro started at UT = '+$
  anytim2utc(timestartprep,/vms)

; define SDO file subdirs
level1dir=datadir+'/level1/'
level2dir=datadir+'/level2/'

; delete level2 (maybe still old files there) and make anew
spawn,'rm -rf '+level2dir
spawn,'mkdir -p '+level2dir

; define permitted SDO level1 fitsfilename strings
wavnames=['94','131','171','193','211','304','335',$
          '1600','1700','4500',$ 
          'magnetogram','continuum','Dopplergram'] 
nwavnames=n_elements(wavnames)

; check refwav specification 
iref=-1
refnamelevel1=refwav
if (refwav eq 'cont') then refnamelevel1='continuum'
if (refwav eq 'mag') then refnamelevel1='magnetogram'
if (refwav eq 'dop') then refnamelevel1='Dopplergram'
for iwav=0,nwavnames-1 do if (wavnames[iwav] eq refnamelevel1) then iref=iwav
if (iref eq -1) then begin
  print,' ##### sdo_prep abort: refwav invalid = ',refwav
  print,' ===== valid: ',wavnames
  return
endif

; re-order wavnames name array to put refwav first
wavnames2=wavnames
wavnames2[0]=wavnames[iref]
wavnames2[iref]=wavnames[0]
wavnames=wavnames2
iref=0

; define anonymous structure file_info following Harry Warren
file_info={filenames:ptr_new(),$
           nfiles:0L,$
           wavnames:''}
file_info=replicate(file_info,nwavnames)
file_info.wavnames=wavnames

; make a file inventory, also following Harry Warren
for i=0,nwavnames-1 do begin

; search for files with JSOC naming habit
;RR add "/quote" for \ flagged spaces in paths (eg Mac "My Passport" disk)
  filenames=file_search(level1dir+'*'+file_info[i].wavnames+'.*',$
                        count=nfiles,/quote)

; search instead for files with LMSAL cutout naming habit
  if (nfiles eq 0) then $
    filenames=file_search(level1dir+'/*_'+file_info[i].wavnames+'_*',$
                          count=nfiles,/quote)

; search instead for files with my naming habit (former set collections)
  if (nfiles eq 0) then $
    filenames=file_search(level1dir+'/*'+file_info[i].wavnames+'.fits',$
                          count=nfiles,/quote)

; fill structure info
  if (nfiles gt 0) then begin
    file_info[i].filenames=ptr_new(filenames)
    file_info[i].nfiles=nfiles
    reftimelist=dblarr(nfiles)
    if (verbose) then print,strpad('SDO '+file_info[i].wavnames,20)$
      +': files found = '+trim(nfiles)
  endif
endfor

; check that reference is present
if (file_info[0].nfiles eq 0) then message,' ##### ERROR: ref not found'

; exit if there are no files at all
good = where(file_info.nfiles gt 0,ngood)
if (ngood eq 0) then message,' ##### ERROR: no input files found'

; start big loop over SDO channels 
; --------------------------------
imscale=fltarr(10)
instrot=fltarr(10)
for iwav=0,nwavnames-1 do begin

;; ; get filelist
;; ;RR add "/quote" for \ flagged spaces in paths (eg Mac "My Passport" disk)
;;   filenames=file_search(level1dir+'/*'+wavnames[iwav]+'.fits',$
;;                         count=nfiles,/quote)

; skip if no files for this channel
  if (file_info[iwav].nfiles eq 0) then goto, ENDWAVLOOP

; get names and filename times for this channel
  filenames=*file_info[iwav].filenames

; read all file indices for this channel (no images)
  nfiles=n_elements(filenames)
  read_sdo,filenames[0:nfiles-1],indexlev1,/nodata,/noshell 

; in the first iwav pass set the reference index (for all time steps)
; and fix the reference pixel scale and angle (following Harry Warren)
  if (iwav eq 0) then begin
    index_ref=indexlev1
    index_ref.cdelt1=0.60d0
    index_ref.cdelt2=0.60d0
    index_ref.crota2=0.00d0
    reftimelist=anytim2tai(index_ref.t_obs)
  endif

; start loop over all files for this channel 
; ---------------------------------------------
  for j=0,nfiles-1 do begin
    
; read this image
    read_sdo,filenames[j],index,image,/uncomp_delete

;  nonref: find index p of the closest level1 reference image
    if (iwav eq 0) then p=j else begin
      obstime=anytim2tai(indexlev1[j].t_obs)
      dummy=min(abs(obstime-reftimelist),p) 
      ; verbose: print the two observing times and their difference
      if (verbose) then begin
        dt=obstime-reftimelist[p]
        print,file_info[iwav].wavnames,': '+$
          anytim(obstime,/yohkoh)+$
          ' <===> '+refwav+': '+anytim(reftimelist[p],/yohkoh)+$
          '  dt='+trim(dt,'(f10.2)')
      endif
    endelse

; Jan 3 2017 reset AIA scale and rotation to 
;   current Master Pointing table values (stuff from aia2wcsmin.pro)
; Feb 10 2020 always, no usheheadpnt/noheadpnt anymore
;;    if (iwav lt 10 and $  ;RR no sense
    if (wavnames[iwav] ne 'continuum' $
        and wavnames[iwav] ne 'magnetogram' $
        and wavnames[iwav] ne 'Dopplergram') $
    then begin
; get scale and rotate from latest Master Pointing table for first timestep
;RR May 21 2018: discussion with Greg Slater and John Serafin at LMSAL
;RR added 3h MPO read below, but it took ages and has no use here since
;RR angle and scale are very rarely updated (when Dick Shine does transit?);
;RR center shifts crpix1/2 better in 3h limb fits but give JSOC cutout errors,
;RR I have to use the JSOC header values with /use_hdr_pnt
;RR  (also gets crpix1/2 from header instead of from MP file, not only
;RR   scale (cdelt1/2 and rotate (crota2) as Slater wrote in aia_prep.pro,
;RR   see aia_prep.pro > aia_reg.pro > aia2wcsmin.pro).  
;RR With rotating JSOC cutouts something via MP goes bad, likely crpix1/2?.
;RR NB: problem case 304 (align to 1600) is also big problem in limb fit
      if (j eq 0) then begin
        mpo_prefix = 'a_' + wavnames[iwav]
        if (mpo_prefix eq 'a_94') then mpo_prefix='a_094'
;????        mpo_str=ssw_sdo_master_pointing(index.t_obs)
;RR Mar 1 2022 /all_versions for problems NISP runs 2011, 2012:
        mpo_str=ssw_sdo_master_pointing(index.t_obs,/all_versions)
 ;;         ,ds='aia_test.master_pointing3h')  ; May 21 2018 trials LMSAL
        t_names_mpo = strlowcase(tag_names(mpo_str))
        ss_imscale=where(strpos(t_names_mpo, mpo_prefix + $
                                '_imscale') ne -1,  n_match_imscale)
        ss_instrot = where(strpos(t_names_mpo, mpo_prefix + $
                                  '_instrot') ne -1, n_match_instrot)
        imscale[iwav]=mpo_str.(ss_imscale)
        instrot[iwav]=mpo_str.(ss_instrot)

;RR Feb 13 2020 maybe treat crpix12 similarly??
;RR   tests 2014 same as index.crpix12, tests 2019 weird differences
        ;; ss_x0=where(strpos(t_names_mpo, mpo_prefix + '_x0') ne -1, $
        ;;             n_match_x0)
        ;; ss_y0=where(strpos(t_names_mpo, mpo_prefix + '_y0') ne -1, $
        ;;             n_match_y0) ;?? x0 > y0 confirmed by Slater
        ;; crpix1_mp=mpo_str.(ss_x0)+1
        ;; crpix2_mp=mpo_str.(ss_y0)+1
      endif
      
; reset scale and rotate keywords for this image to first-timestep MP values
      index.cdelt1=imscale[iwav]
      index.cdelt2=imscale[iwav]
      index.crota2=instrot[iwav]
    endif 

;   aia_prep the image using the closest p reference (current for reference)
;RR   do NOT add ,/use_shared_lib or HMI-B BSCALE may get divided by 10
;RR   and I saw no speed increase (also not for /noshell)
;RR Feb 19 2020 temporary beta fix from Greg Slater at top above
;RR Apr  2 2020 no, aia_prep was standard version all along
;RR adapt the below for a notrack option didn't work: error in level1
;RR is at JSOC already for 304 and 94; changed default refwav to '171'
    ;; aia_prep,index,image,/quiet,/use_hdr_pnt,$
    ;;   /use_ref,index_ref=index_ref[p],/do_write_fits,outdir=level2dir
;RR Sep 12 2020 discard?  No-track problem level1 already?  But no-no:
;RR test '2018.04.12_15:40',25,-293,-298 very bad after SDO reset jump
    ;; aia_prep,index,image,/quiet,$   ;;; /no_mpo_update,$
    ;;   /use_ref,index_ref=index_ref[p],/do_write_fits,outdir=level2dir
;RR Sep 11 2020: Gregal Vissers add /no_mpo_update after he and Koza got stuck
;;     aia_prep,index,image,/quiet,/no_mpo_update,$
;;       /use_ref,index_ref=index_ref[p],/do_write_fits,outdir=level2dir
;RR Mar 8 2022: above out again, gives much tighter driftplots - hope robust.
;RR             Likely better by using newer LMSAL 3h limb fits
    ;; aia_prep,index,image,/quiet,$
    ;;   /use_ref,index_ref=index_ref[p],/do_write_fits,outdir=level2dir
;RR Apr 13 2022: /no_mpo_update back in, Reetika Joshi again index2mpo problem
    aia_prep,index,image,/quiet,/no_mpo_update,$
      /use_ref,index_ref=index_ref[p],/do_write_fits,outdir=level2dir

; end of loop over files for this channel
  endfor

ENDWAVLOOP:
endfor   ; end of loop over all channels

; print MP scale and rotate values when verbose
if (verbose ne 0) then begin
  for iwav=0,9 do print,' ----- wav = '$
    +wavnames[iwav]+':  imscale = '+trim(imscale[iwav])+$
    '  instrot = '+trim(instrot[iwav])
endif

; print elapsed time
timedone=systime(1)
timelaps=ntostr((timedone-timestartprep)/60.,format='(F11.1)')
if (verbose ne 0) then print,' ===== sdo_prep.pro took '+timelaps+' minutes'

end


; =============== main for testing per IDLWAVE H-c ======================

;; datadir='/home/rutten/data/SDO/2019-11-11-transit/midpoint/target/'
;; datadir='/home/rutten/data/SDO/2019-11-11-transit/midpoint-notrack/target/'
datadir='/home/rutten/data/NISP/2011-06-24-quiet/sdo/center/.

sdo_prep,datadir,/verbose

end
