; file: sdo_muckimagepair.pro
; init: Jun 28 2020  Rob Rutten  Deil from sdo_writepairspline.pro
; last: Aug  1 2020  Rob Rutten  Deil

;+
pro sdo_muckimagepair,im1,im2,wavpair,$
  muckim1,muckim2,astile,tilecut,heightdiff

 ; muck a pair of SDO images to better alikeness for alignment 
 ;   
 ; INPUTS:
 ;   im1, im2: SDO images
 ;   wavpair: 2-elem integer array defining sdo wavs ("channels") my way
 ;     or 2-elem string array with SDO level1 channel names
 ;
 ; OUTPUTS:
 ;   muckim1, muckim2, astile, heightdiff, tilecut
 ;
 ; METHOD;
 ;   trumpian tricky-dicky fakes
 ; 
 ; HISTORY:
 ;   Jun 28 2020 RR: extract from sdo_writepairspline.pro  
;-

; answer no-parameter query 
if (n_params(0) lt 8) then begin
  sp,sdo_muckimagepair
  return 
endif

; check im size equality
szim1=size(im1)
szim2=size(im2)
if (szim1[1] ne szim2[1] or szim1[2] ne szim2[2]) then begin
  print,' ##### sdo_muckimagepair: dimensions im1, im2 inequal'
  STOP
endif

; initialize muck versions
muckim1=im1
muckim2=im2

; check wavpair and what type

; define permitted level1 SDO name strings (rrwavnr = index+1)
sdowavs=['94','131','171','193','211','304','335',$
         '1600','1700','4500',$  
         'magnetogram','continuum','Dopplergram'] 
nsdowavs=n_elements(sdowavs)
;  rrwav numbers 
; 94,131,171,193,211,304,335,1600,1700,4500,mag,cont,dop
;  1  2   3   4   5   6   7   8    9    10   11  12   13

; check wavpair
szpair=size(wavpair)
if (szpair[0] ne 1 or szpair[1] ne 2) then begin
  print,' ##### sdo_muckimagepair wavpair not 2-elem array:'+trimd(wavpair)
  STOP
endif

; get rrpair values
rrpair=[0,0]
if (szpair[2] eq 2) then rrpair=wavpair ; integers = rrnumber pair
if (szpair[2] eq 7) then begin             ; strings = SDO level1 name pair
  for iwav=0,nsdowavs-1 do begin
    if (sdowavs[iwav] eq wavpair[0]) then rrpair[0]=iwav+1
    if (sdowavs[iwav] eq wavpair[1]) then rrpair[1]=iwav+1
  endfor
endif

; check
if (rrpair[1] eq rrpair[0]) then begin
  print,' ##### sdo_muckimagepair abort: rrpair ='+trimd(rrpair)
  STOP
endif

; histo_opt top of all AIA images
;RR they have been int_scaled already in the center cube making
;RR so some are already sqrt and log for better display
if (rrpair[0] lt 10) then muckim1=histo_opt_rr(muckim1,1.E-4,/top_only)
if (rrpair[1] lt 10) then muckim2=histo_opt_rr(muckim2,1.E-4,/top_only)

; set default tile parameters 
astile=30 ; tilesize in arcsec
tilecut=2  ; nr of iterations 
heightdiff=0 ; in km 

; ===== 89 = 1600 > 1700 ("expanding-universe" vectors when tiling) 
if (rrpair[0] eq 8 and rrpair[1] eq 9) then begin
  muckim1=muckim1>1.2*avg(muckim1)     ; remove internetwork acoustics
  muckim2=muckim2>1.2*avg(muckim2)     ; remove internetwork acoustics
  tilecut=3                 ; precise measurement in many tiles
  heightdiff=220.           ; much smaller than Vienna, see emwrkdsojot
endif

; ===== 911 = 1700 > mag  (UV to HMI anchor)
if (rrpair[0] eq 9 and rrpair[1] eq 11) then begin
  muckim1=muckim1>1.2*avg(muckim1)     ; remove internetwork acoustics
  imclip=0.*muckim2
  nospot=where(abs(muckim2) lt 7500)      ; 750 Gauss for disk-center plage
  imclip[nospot]=abs(muckim2[nospot]) 
  muckim2=imclip>mean(imclip)             ; take out magnetogram noise
  tilecut=3                           ; precise measurement in many tiles
  heightdiff=220.
endif

; ===== 811 = 1600 > mag  (for check or expanding universe vector chart)
if (rrpair[0] eq 8 and rrpair[1] eq 11) then begin
  muckim1=muckim1>1.2*avg(muckim1)     ; remove internetwork acoustics
  imclip=0.*muckim2
  nospot=where(abs(muckim2) lt 7500)      ; cut umbrae 
  imclip[nospot]=abs(muckim2[nospot]) 
  muckim2=imclip>mean(imclip)             ; take out magnetogram noise
  tilecut=3                           ; precise measurement in many tiles
  heightdiff=420.
endif

; ===== 611 = 304 > mag  (EUV => HMI anchor, very difficult)
if (rrpair[0] eq 6 and rrpair[1] eq 11) then begin
  muckim1=muckim1>0.5*avg(muckim1)     ; remove IN nose       
  muckim1=histo_opt(muckim1,1E-4)   ; again? also bottom      
  reformimage,muckim1,muckim1s,smear=25.
  muckim1=sqrt(muckim1s)
  imclip=0.*muckim2                    ; zero mag image
  nospot=where(abs(muckim2) lt 5000)   ; cut umbrae keeping disk-center plage
  imclip[nospot]=abs(muckim2[nospot]) 
  muckim2=imclip>0.01*avg(imclip)     ; take out magnetogram noise
  reformimage,muckim2,muckim2s,smear=25.   ; same as above for muckim1
  muckim2=alog10(sqrt(muckim2s))           ; seems better
  tilecut=2
  heightdiff=3600.
  astile=45
endif

; ===== 68 = 304 > 1600 (EUV to UV anchor, very difficult)
if (rrpair[0] eq 6 and rrpair[1] eq 8) then begin
  if (min(muckim1) lt 0) then muckim1=float(muckim1)-min(muckim1) 
  muckim1=muckim1>0.5*avg(muckim1)           ; May 30 2020: remove IN noise
  muckim1=histo_opt(muckim1,1E-3)         ; clip
  reformimage,muckim1,muckim1s,smear=25. 
  muckim1=sqrt(muckim1s)
  muckim2=muckim2>1.2*avg(muckim2)           ; remove IN acoustics
  muckim2=histo_opt(muckim2,1E-3)         ; clip 
  reformimage,muckim2,muckim2s,smear=25. 
  muckim2=muckim2s
  heightdiff=3200.
endif

; ===== 69 = 304 > 1700 (EUV to UV anchor, very difficult)
if (rrpair[0] eq 6 and rrpair[1] eq 9) then begin
  if (min(muckim1) lt 0) then muckim1=float(muckim1)-min(muckim1) 
  muckim1=muckim1>0.5*avg(muckim1)
  muckim1=histo_opt(muckim1,1E-3)
  reformimage,muckim1,muckim1s,smear=25.
  muckim1=sqrt(muckim1s)
  muckim2=muckim2>1.2*avg(muckim2)
  muckim2=histo_opt(muckim2,1E-3)
  reformimage,muckim2,muckim2s,smear=25.
  muckim2=muckim2s
  heightdiff=3400.
endif

; ===== 28 = 131 > 1600 (trial EUV to UV anchor, very difficult)
if (rrpair[0] eq 2 and rrpair[1] eq 8) then begin
  muckim1=histo_opt(muckim1,1E-3)
  reformimage,muckim1,muckim1s,smear=25.
  muckim1=muckim1s
  muckim2=muckim2>1.2*avg(muckim2)
  muckim2=histo_opt(muckim2,1E-3)
  reformimage,muckim2,muckim2s,smear=25.
  muckim2=muckim2s
  heightdiff=2000.
endif

; ===== remaining EUV-EUV rrpairs
if (rrpair[0] eq 3 and rrpair[1] eq 6) then heightdiff=300. ; 171 > 304
if (rrpair[0] eq 5 and rrpair[1] eq 6) then heightdiff=75.  ; 211 > 304
  ; rest are all coronal and should have no systematic height difference
  ; (multi-temperature loops adjacent in bundles)

;RR Feb 14 2020 smearing 5 changed all drift plots but not obviously better
;; ; ===== all EUV to EUV (they already are log or sqrt from intscale)
;;   if (rrpair[0] lt 8 and rrpair[1] lt 8) then begin
;;     reformimage,muckim1,muckim1s,smear=5.
;;     muckim1=muckim1s
;;     reformimage,muckim2,muckim2s,smear=5.
;;     muckim2=muckim2s
;;   endif

;; ; ===== 25 = 131 > 211 (difficult, many trials; also Pradeep; extra check)

; bytscale both
muckim1=bytscl(muckim1) 
muckim2=bytscl(muckim2) 

end

; =============== main for testing per IDLWAVE H-c ======================

; Mercury fulldisk
  datadir='~/data/SDO/2019-11-11-transit/midpointfull/target/cubes_noalign/'
  wavpair=['304','magnetogram']
  im1=readfitscube(datadir+'aia304.fits',trange=[3,3])
  im2=readfitscube(datadir+'hmimag.fits',trange=[3,3])
 
  sdo_muckimagepair,im1,im2,wavpair,$
    muckim1,muckim2,astile,tilecut,heightdiff

; check
  print,' ----- astile,tilecut,heightdiff = ',astile,tilecut,heightdiff
  showex,im1,im2,muckim1,muckim2
  
end
