; file: sdo_makenosyncfitscubes
; init: Apr  3 2020  Rob Rutten  Deil from sdo_makeslavefitscube.pro
; last: Jul  6 2020  Rob Rutten  Deil
; note: Made for Mercury transit to have non-time-interpolated image cubes
;         so that Mercury retains its round disk shape.
;       Works standalone but needs previous normal run producing
;         /driftscenter and ../cubes/<refwav>.fits to read its header.
;       Includes fixing SDO's hourly 1700 sampling skips.

;+

pro sdo_makenosyncfitscubes,datadir,outdir,refwav=refwav,$
  nottrack=notrack,diffdesolrot=diffdesolrot,$
  nocrossalign=nocrossalign,driftsdir=driftsdir,$
  clipmin=clipmin,clipmax=clipmax,no_intscale=no_intscale,$
  verbose=verbose

 ; NAME:
 ;   sdo_makenosyncfitscubes.pro
 ; PURPOSE:
 ;   convert level 1.5 SDO images into fitsfile cube
 ;   with fractional-pixel derotation, intensity rescaling, optional nocrossalign
 ;   but no temporal interpolation to other (refwav) timing
 ; CALL:
 ;   see above
 ; CALL EXAMPLE:
 ;   see end of program file
 ; INPUTS:
 ;   datadir = string path/dir containing level2 SDO files
 ;   outdir = string name subdir for results
 ; OPTIONAL KEYWORD INPUTS:
 ;   refwav: string specifying AIA or HMI reference wavelength, default '171'
 ;   notrack = 1/0: JSOC did not track rotation in im_patch cutting out
 ;   diffdesolrot = 1/0: differentially derotate target
 ;   nocrossalign=1/0: default 0 = cross-align with driftsplines
 ;   driftsdir: string path/dir of driftsplines, default '../driftscenter'
 ;   clipmin: min intensity clipping in sdo_intscale.pro
 ;   clipmax: max intensity clipping in sdo_intscale.pro
 ;   no_intscale = 1: do not intscale target AIA and HMIcont intensities
 ;   verbose=1/[0]: print and plot shifts per timestep
 ; OUTPUTS:
 ;   cube fits file
 ; METHOD:
 ;   use assoc to permit larger cubes than memory size
 ; HISTORY:
 ;   Apr  3 2020 RR: start
 ;   Apr  4 2020 RR: fix hourly 1700 skips
;-

; answer no-parameter query
if (n_params() lt 2) then begin
  sp,sdo_makenosyncfitscubes
  return
endif

; get current dir path; \ flag spaces in path if present (bloody Mac?)
; https://groups.google.com/forum/#!topic/comp.lang.idl-pvwave/Lo10H5XtN80
cd,current=thisdir
thisdir=strjoin(strsplit(thisdir,' ',/extract),'\ ') 

; defaults for keywords
if (n_elements(refwav) eq 0) then refwav='171'
if (n_elements(notrack) eq 0) then notrack=0
if (n_elements(diffdesolrot) eq 0) then diffdesolrot=0
if (n_elements(nocrossalign) eq 0) then nocrossalign=0
if (n_elements(driftsdir) eq 0) then driftsdir='driftscenter'
if (n_elements(clipmin) eq 0) then clipmin=0
if (n_elements(clipmax) eq 0) then clipmax=0
if (n_elements(no_intscale) eq 0) then no_intscale=0
if (n_elements(verbose) eq 0) then verbose=0

; define datadir subdirs
level2dir=datadir+'/level2'
cubesdir=datadir+'/cubes'
nosyncdir=datadir+'/'+outdir ;; extra here

; make dirs
if (verbose ne 0) then spawn,'mkdir -p '+cubesdir+'/plots_align/'
spawn,'mkdir -p '+nosyncdir  ;; extra here

; define permitted level2 SDO (AIA + HMI) filename strings
wavs=['94','131','171','193','211','304','335','1600','1700',$  ; AIA
      'mag','cont','dop']                                       ; HMI
nwavs=n_elements(wavs)
fitscubenames=['aia94.fits','aia131.fits','aia171.fits',$
               'aia193.fits','aia211.fits','aia304.fits',$
               'aia335.fits','aia1600.fits','aia1700.fits',$
               'hmimag.fits','hmicont.fits','hmidop.fits']

; set channel names for plots
channels=wavs
for i=0,8 do channels[i]='AIA '+wavs[i]
channels[9]='HMI magnetogram'
channels[10]='HMI continuum'
channels[11]='HMI Dopplergram'

; get iref = index of reference (and check it is there)
iref=-1
for iw=0,nwavs-1 do if (wavs[iw] eq refwav) then iref=iw
if (iref eq -1 ) then begin
  print,' ===== ERROR: refwav invalid = ',wav
  print,' ===== valid: ',wavs
  return
endif
refcube=fitscubenames[iref]

; ============== run this program for all possible SDO wavs

for iwav=0,nwavs-1 do begin
;; for iwav=8,8 do begin ;RR for tests skip repair 1700
  wav=wavs[iwav]
  print,' ===== next wav = '+wav

; remove previous 1700 insert files for skip repairs
  if (wavs[iwav] eq '1700') then $
    spawn,'rm -f '+datadir+'/level2/*b*1700.fits*'
  
; get wav file list (undo compression if fits files fpacked into .fz)
  spawn,'funpack -D '+datadir+'/level2/*'+wav+'.fits.fz'
;??  files=findfile(datadir+'/level2/*'+wav+'.fits') ;RR deprecated by IDL
  files=file_search(datadir+'/level2/*'+wav+'.fits')

  nfiles=n_elements(files)
  if (nfiles eq 1) then begin
    print,' ##### no files for wav = '+wav+'  therefore skipped'
    goto, ENDWAVLOOP
  endif

; get refcube size (it was cropped by border parameter)
  refheader=headfits_rr(cubesdir+'/'+refcube)  ;!! there must be refcube
  nxref=fxpar(refheader,'naxis1')
  nyref=fxpar(refheader,'naxis2')
  xcenpxref=nxref/2.
  ycenpxref=nyref/2.
  ntref=fxpar(refheader,'naxis3')

; get header info first wav image 
  read_sdo, files[0],index0,/nodata
  nx0=index0.naxis1
  ny0=index0.naxis2
  xcenpx0=nx0/2.
  ycenpx0=ny0/2.
  starttim=index0.t_obs  

; determine wav cadence from header last wav image
  read_sdo, files[nfiles-1],indexlast,imagelast
  tlast=indexlast.t_obs
  cadence=(anytim2tai(tlast)-anytim2tai(starttim))/float(nfiles-1)
  
; correct hourly sample skips in 1700 by adding a fake level2 file
  if (wavs[iwav] eq '1700') then begin
    for it=0,nfiles-2 do begin
      read_sdo,files[it],index1,image1
      read_sdo,files[it+1],index2,image2
      t1=index1.t_obs
      t2=index2.t_obs
      t1tai=anytim2tai(t1)
      t2tai=anytim2tai(t2)
      deltat=t2tai-t1tai
      if (abs(deltat-cadence) gt 2.0) then begin
        print,' ----- it ='+trimd(it)+'  files '+files[it]+'    '+files[it+1]
        print,' ----- 1700 gap between '+t1+' and '+t2+' ='+trimd(deltat)+$
          ' -- insert extra level2 file'
; interpolate straddle pair to intermediate time step
        ;; p=[[[image1]],[[image2]]]
        ;; image3=interpolate(p,0.5)  ;RR cubic no difference for 2)
; no, gave bad jump, guess sawtooth, instead take previous one with small dt
        image3=image2
; NB wrong because but who cares - one image/hour.
; write extra level2 file with modified t_obs in index but close
; to original t_obs to have corresponding sawtooth and crossalign sample time
        t3tai=t1tai+1.
        t3=anytim2utc(t3tai,/ccsds)
        index3=index2
        index3.t_obs=t3
        file3=repstr(files[it],'_1700.fits','_b_1700.fits')
        mwritefits,index3,image3,outfile=file3  ; mwritefits for structure
;; ; check
;;         read_sdo,file3,indextry,imagetry
      endif
    endfor

; re-obtain file list (now one more) and correct cadence    
    files=findfile(datadir+'/level2/*'+wav+'.fits')
    nfiles=n_elements(files)
    read_sdo, files[nfiles-1],indexlast,imagelast
    tlast=indexlast.t_obs
    cadence=(anytim2tai(tlast)-anytim2tai(starttim))/float(nfiles-1)

  endif ; end of 1700 skip treatment, now proper cadence, one fake duplicate

; get reference cube parameters
  nxref=fxpar(refheader,'naxis1')
  nyref=fxpar(refheader,'naxis2')
  cenxpxref=nxref/2.-0.5   ; center of FOV center px in IDL no-finger count
  cenypxref=nyref/2.-0.5
  cdelt1ref=fxpar(refheader,'cdelt1')
  cdelt2ref=fxpar(refheader,'cdelt2')
  crpix1ref=fxpar(refheader,'crpix1') ; NB funny WCS start at (1,1)
  crpix2ref=fxpar(refheader,'crpix2') ; NB funny WCS start at (1,1)
  xcenarcref=fxpar(refheader,'xcen')
  ycenarcref=fxpar(refheader,'ycen')
  crota2ref=fxpar(refheader,'crota2')
  rsun_obs=fxpar(refheader,'rsun_obs')
  timeref=fxpar(refheader,'starttim')
  tairef=anytim2tai(timeref)
  
; define outheader with timing and location parameters
  bigendian=1   ;RR my Toshiba laptop
  bitpix=16l    ;RR also for HMI cubes
  mkhdr,outheader,abs(bitpix)/8,[nxref,nyref,nfiles]
  sxaddpar,outheader,'starttim',starttim,'first image mid-exposure'
  sxaddpar,outheader,'cadence',cadence,'image cadence [s]'
  sxaddpar,outheader,'cdelt1',cdelt1ref,'x-scale [arcsec/px]'
  sxaddpar,outheader,'cdelt2',cdelt2ref,'y-scale [arcsec/px]'
;; sxaddpar,outheader,'crpix1',crpix1ref,$
;;   'disk-center offset in px from leftmost of FOV = nr 1'
;; sxaddpar,outheader,'crpix2',crpix2ref,$
;;   'disk-center offset in px from bottom of FOV = nr 1'
;RR do not copy to outheader since image size has changed per border cutting
  sxaddpar,outheader,'crota2',crota2ref,$
    'rotation angle degrees NESW = CCW'
  sxaddpar,outheader,'rsun_obs',rsun_obs,'solar radius [arcsec]'
  sxaddpar,outheader,'xcen',xcenarcref,$
    'FOV center X [arcsec from sun center]'
  sxaddpar,outheader,'ycen',ycenarcref,$
    'FOV center Y [arcsec from sun center]'
  sxaddpar,outheader,'channel',channels[iwav],$
    'SDO diagnostic name [string]'

; get size outheader
  sizeoutheader=size(outheader) 
  outheadersize=(1+fix(sizeoutheader[1]/36.))*2880
  ;RR fits header = Nx36 "card images" = Nx2880 bytes

; open output cube file for assoc, write header
  get_lun, unit_out
  if (bigendian) then openw,unit_out,nosyncdir+'/'+fitscubenames[iwav],$
    /swap_if_little_endian $
  else openw,unit_out,nosyncdir+'/'+fitscubenames[iwav]
  outassoc=assoc(unit_out,intarr(nxref,nyref),outheadersize)
  rec=assoc(unit_out, bytarr(outheadersize))
  rec[0]=byte(outheader)

; start loop over files = timesteps in level2 data
  shiftx=fltarr(nfiles)
  shifty=fltarr(nfiles)
  for it=0,nfiles-1 do begin

; read new image
    read_sdo,files[it],index,image,/uncomp_delete
    imtime=anytim2tai(index.t_obs)

; get size
    if (it eq 0) then begin
      sizeim=size(image)
      nx=sizeim[1]
      ny=sizeim[2]
    endif

; rescale image values (greyscale)
    image=sdo_intscale(index,image,wav,clipmin=clipmin,clipmax=clipmax,$
                        no_intscale=no_intscale)

; if crossalign (meaning cross-align) get AIA shift against HMI for this image
    sdoshift=[0,0]
    if (iwav lt 9 and nocrossalign ne 1) then $
      sdoshift=sdo_getsumsplineshift(wav,imtime,driftsdir) 

; reposition image for sawtooth undo 
    sdo_shiftjsoc,image,index,$
      tairef,xcenarcref,ycenarcref,cenxpxref,cenypxref,sdoshift,$
      outimage,outshift,notrack=notrack,diffdesolrot=diffdesolrot
    image=outimage
    shiftx[it]=outshift[0]
    shifty[it]=outshift[1]

; write image cropped to reference size
    if (nx lt nxref or ny lt nyref) then begin
      print,' ##### nx, ny = '+ntostr(nx)+' '+ntostr(ny)+$
        ' but nxref, nyref = '+ntostr(nxref)+' '+ntostr(nyref)
      print,' ##### redo cube making larger border sdo_makereffitscube'
      return ; Jan 22 2016 back in because IDL stops anyhow in next line
    endif
    cutim=image[0:nxref-1,0:nyref-1]
    outassoc[it]=cutim

; check
;; if (it eq 0) then print,' ##### nx,ny,nxref,nyref = ',nx,ny,nxref,nyref

  endfor

; free the output file
  free_lun,unit_out

; verbose: plot shifts 
  if (verbose ne 0) then begin
    shiftx=shiftx[0:nfiles-1]
    shifty=shifty[0:nfiles-1]
    psfilename=cubesdir+'/plots_align/shifts-'+fitscubenames[iwav]
    psfilename=str_replace(psfilename,'.fits','.eps')
    openpsplot,psfilename,thick=2,fontsize=9,xsize=8.8,ysize=8.8
    yrange=[min([min(shiftx),min(shifty)]),$
            max([max(shiftx),max(shifty)])]
    yrange=[yrange[0]-0.1*(yrange[1]-yrange[0]),$
            yrange[1]+0.1*(yrange[1]-yrange[0])]
    plot,indgen(nfiles-1),shiftx[0:nfiles-1],$
      xrange=[0,nfiles-1],yrange=yrange,xstyle=1,ystyle=1,$
      xtitle='time step',ytitle='shift in px',$
      position=[0.2,0.1,0.95,0.95]
    oplot,indgen(nfiles-1),shifty[0:nfiles-1],linestyle=2
    itlabel=(nfiles-1)/2
    labeloffset=(yrange[1]-yrange[0])*0.02
    xyouts,itlabel,shiftx[itlabel]+labeloffset,'x'
    xyouts,itlabel,shifty[itlabel]+labeloffset,'y'
    xyouts,(nfiles-1)/8,yrange[1]-0.1*(yrange[1]-yrange[0]),$
      'shifts '+fitscubenames[iwav],$
      charsize=1.2
    closepsplot,psfilename,opengv=verbose

; print success
    print,' === wrote ',nosyncdir+'/'+fitscubenames[iwav]

  endif

ENDWAVLOOP:
endfor ; end of big loop over all SDO wavs

if (nocrossalign eq 0) then spawn,'touch '+nosyncdir+'/00-NB_cubes_aligned' $
else spawn,'touch '+nosyncdir+'/00-NB_cubes_not_aligned' 

end


; =============== main for testing per IDLWAVE H-c ========================

;RR #### ?? don't run multiple via IDLWAVE but in terminal or not at all ??
;RR      apparently IDLWAVE uses some /tmp/idltemp file, then duplicate

;; cd,'/home/rutten/data/SDO/2019-11-11-transit/complete_notrack'
cd,'/home/rutten/data/SDO/2014-06-14-small'
datadir='target'

;RR ##### alternate between next groups for align / no_align
outdir='nosync_align'
nocrossalign=0
;; outdir='nosync_noalign'
;; nocrossalign=1

sdo_makenosyncfitscubes,datadir,outdir,nocrossalign=nocrossalign

;; ; inspect (cut and paste)
;; showex,/allsdo,sdodir='target/nosync_align'
;; showex,'target/nosync_align/aia1600.fits','target/nosync_noalign/aia1600.fits',/addlabels
;; showex,'target/nosync_align/aia171.fits','target/nosync_noalign/aia171.fits',/addlabels


end
