; file: makecdejtriptychs.pro = SDO assemblies CdeJ book and birthday book
; init: Apr 24 2020  Rob Rutten  Deil from IBIS livrev mosaics pro
; last: Mar 18 2022  Rob Rutten  Deil into sdolib
; note: program to make SDO triptych figures for Kees de Jager
;       Can be run for each minute since SDO 2010 start until week ago
;       run per Hyper C (IDL main)
;       "CdeJ book" = "Small-scale surface magnetism" = 2020smvc.book...29R
;       "birthday book" = "The Sun on Kees days" 
;          https://robrutten.nl/rrweb/rjr-pubs/2021LingAstRep...3R.pdf

;; pro dummy ; for emacs indentation ; ## take out to run

; ===== parameter selections

; choose figtype
;   1 = CdeJ book 9-pack magnetism photosphere
;   2 = CdeJ book 9-pack heat higher atmosphere
;   3 = one-year triples higher atmosphere
;   4 = one-year colored triples for CdeJ coffee-table book
figtype=4

; which year or years (loop)
;; years=['2014','2015','2019'] ; RR choice for CdeJ book 
years=['2011','2012','2013','2014','2015','2016','2017','2018','2019','2020']
years=['2021'] ; one only

; set date and time within year(s)
; CdeJ born April 29 1921 at 03:00 UT (4 o'clock on Texel)
; NO take 11:00 in hope full disk
;; datetimes=years+'.04.29_03:00' 
;; datetimesdash=years+'0429_03_00'
datetimes=years+'.04.29_11:10' ; yes. 11:10 is full disk 
datetimesdash=years+'0429_11:10'

usefiles=0 ; set to 1 when SDO files already downloaded into /tmp

; ========= rest should run parameter-free (except layout choices)

; figtype 1 = demo photospheric fields
if (figtype eq 1) then begin
  wavs=['continuum','magnetogram','1700']
  filewavs=['cont','mag','_1700']
  border=350 ; crop empty borders around solar disk 
endif

; figtype > 1 = demo heat above photospheric fields
if (figtype gt 1) then begin
  wavs=['magnetogram','304','171']
  filewavs=['mag','_0304','_0171']
  ;; ;; wavs=['304','171']
  ;; ;; filewavs=['_0304','_0171']
  border=150 ; wider borders to admit off-limb loops and plumes
endif

for iyear=0,n_elements(years)-1 do begin
  datetime=datetimes[iyear]
  datetimedash=datetimesdash[iyear]

  ;;;;;; start 1  ;RR fix for 1Kx1K mag 
  for iwav=1,n_elements(wavs)-1 do begin
    wav=wavs[iwav]

; get image
    raw=0  ;RR had raw=1 in CdeJ book time but since June 2020 sdo_intscale
    if (usefiles eq 0) then $
      sdo_getimage,datetime,wav,im,indexim,raw=raw $
    else begin
      file=findfile('/tmp/*'+years[iyear]+'*'+filewavs[iwav]+'.fits')
      sdo_getimage,file,wav,im,indexim,raw=raw 
    endelse

; set official SDO colors
    if (figtype eq 4) then begin
      if (wav eq 'magnetogram') then aia_lct,r,g,b,wavelnth=1,/load
      if (wav eq '304') then aia_lct,r,g,b,wavelnth=304,/load
      if (wav eq '171') then aia_lct,r,g,b,wavelnth=171,/load
    endif 
      
; crop wide borders
    image=im[border:4096-border,border:4096-border]
    sizeim=size(image)
    nx=sizeim[1]
    ny=sizeim[2]

; muck AIA intscales to undo deterioration and get nice contrasts 
    if (wav eq 'continuum') then image=histo_opt_rr(image,5E-4,/top_only)
    if (wav eq '1700') then begin
      image=histo_opt(image,1E-4)
      image=sqrt(image)>10.
    endif
    if (wav eq '304' or wav eq '171') then begin

 ; for CdeJ book I compensated the AIA deterioration by normalizing
 ; an area above the South pole outside activity belts.  In June 2020
 ; I instead improved sdo_intscale to correct the measured deterioration.

   ;;    cutsize=500
   ;;    cutcenter=[nx/2,ny/2-1000]
   ;;    cutxmin=cutcenter[0]-cutsize/2
   ;;    cutxmax=cutcenter[0]+cutsize/2
   ;;    cutymin=cutcenter[1]-cutsize/2
   ;;    cutymax=cutcenter[1]+cutsize/2
   ;;    cutim=image[cutxmin:cutxmax,cutymin:cutymax]
   ;;    cutim=histo_opt_rr(cutim,1E-2)
   ;;    meancutim=mean(cutim)
   ;;    image=image<10.*meancutim
   ;;    image=image>0.1
   ;;    image=sqrt(image)
      
 ; using sdo_intscale since Jun 24 2020  
   ;;   image=histo_opt_rr(image,1E-4)
    endif      
    
; bytscl to make 255 white (mag maintain zero)
    bytim=bytscl(image)  ; AHA: this made them nice also at low response
    if (wav eq 'magnetogram') then begin
      image=image>(-100)<(100) ; clips in "apparent" Gauss
      bytim=bytzero(image)
    endif

  ; ----- add cuts with magnified inserts in corners

    imins=bytim ; will be sucessively filled with cutouts

    if (figtype ne 4) then begin
      
; set cut dimensions
      cutsize=300
      mag=3

; loop over cutouts and magnified inserts
      for icut=0,3 do begin

; define the cut locations = center down to limb (shifts with border)
        if (icut eq 0) then cutcenter=[nx/2,ny/2]
        if (icut eq 1) then cutcenter=[nx/2,ny/2-500]
        if (icut eq 2) then cutcenter=[nx/2,ny/2-1000]
        if (icut eq 3) then cutcenter=[nx/2,+cutsize/2+130]
        cutxmin=cutcenter[0]-cutsize/2
        cutxmax=cutcenter[0]+cutsize/2
        cutymin=cutcenter[1]-cutsize/2
        cutymax=cutcenter[1]+cutsize/2

; get cut image, bytscl independently to show fine structure best
        cutim=image[cutxmin:cutxmax,cutymin:cutymax]
        cutim=scale(cutim,mag,mag) ; ancient Herbie caller to rebin
        if (wav ne 'magnetogram') then cutim=bytscl(cutim) $
        else cutim=bytzero(cutim)

; add white borders to inserts
        thick=5
        sizecut=size(cutim)
        borderim=bytarr(sizecut[1]+2*thick,sizecut[2]+2*thick)+255 ; all white
        sizeborder=size(borderim)
        nxins=sizeborder[1]
        nyins=sizeborder[2]
        borderim[thick:nxins-1-thick,thick:nyins-1-thick]=cutim[*,*]

; stick inserts in corners
        if (icut eq 0) then imins[0:nxins-1,ny-nyins:ny-1]=borderim[*,*]
        if (icut eq 1) then imins[nx-nxins:nx-1,ny-nyins:ny-1]=borderim[*,*]
        if (icut eq 2) then imins[nx-nxins:nx-1,0:nyins-1]=borderim[*,*]
        if (icut eq 3) then imins[0:nxins-1,0:nyins-1]=borderim[*,*]
        
; add white cutout frames
        thick=4 ; in px
        imins[cutxmin:cutxmax,cutymin-thick:cutymin+thick]=255
        imins[cutxmin:cutxmax,cutymax-thick:cutymax+thick]=255
        imins[cutxmin-thick:cutxmin+thick,cutymin:cutymax]=255
        imins[cutxmax-thick:cutxmax+thick,cutymin:cutymax]=255
        
      endfor ; end of loop over cutout inserts
    endif    ; not figtype eq 4 
    
;; ; inspect
;;   showex,imins
;;   STOP
    
; make ps plot
    if (figtype lt 3) then psfilename=datetimedash+'_'+wav+'.eps'
    if (figtype ge 3) then psfilename='thisyear_'+wav+'.eps'
    plotarea=[0.,0.,1.,1.] ; no white borders since no axes
    openpsplot,psfilename,thick=2,fontsize=9,xsize=8.8,ysize=8.8
    tv,imins,plotarea[0],plotarea[1],$
       xsize=plotarea[2]-plotarea[0],ysize=plotarea[3]-plotarea[1],/normal

; add datetime for CdeJ mode
;;;;    if (figtype eq 4 and wav eq 'magnetogram') then begin
    if (figtype eq 4) then begin
      xyouts,plotarea[0]+0.02,plotarea[1]+0.02,/norm,color=255,$
        datetimes[iyear]
    endif

    closepsplot,psfilename,opengv=0
    print,' ----- wrote '+psfilename

   ;;  ; check
   ;; if (wav eq 'magnetogram') then stop
    
; convert to make suited to pdflatex (tricky dicky)
    spawn,'pscropepsone '+psfilename
    spawn,'ps2pdf '+psfilename
    
  endfor ; end loop over wavs

; run latex multi-panel composer

  if (figtype ge 3) then begin
    spawn,'pdflatex joinfull'
    spawn,'pdfcrop joinfull.pdf full_'+datetimedash+'.pdf'
    spawn,'pdf2pngone  full_'+datetimedash+'.pdf'
    spawn,'rmtex; rm -f thisyear* joinfull.pdf'
    ; inspect with rrview *png
  endif
  
endfor ; end loop over years

; run latex multi-panel composer

if (figtype eq 1) then begin
  spawn,'pdflatex joinsdofigs1'
  spawn,'pdfcrop joinsdofigs1.pdf rrfig1.pdf'
  spawn,'mv rrfig1.pdf ../../version_2/rrfigures'
endif

if (figtype eq 2) then begin
  spawn,'pdflatex joinsdofigs2'
  spawn,'pdfcrop joinsdofigs2.pdf rrfig3.pdf'
  spawn,'mv rrfig3.pdf ../../version_2/rrfigures'
endif

end
