; file: sdo_latest.pro
; init: May 29 2019  Rob Rutten  Deil
; last: Jun  3 2019  Rob Rutten  Deil

;+
pro sdo_latest,wav,image,index,verbose=verbose

 ; NAME:
 ;  sdo_latest      
 ; PURPOSE:
 ;  get latest SDO image for specified wav, with index info
 ; DESCRIPTION:
 ;   uses ssw_jsoc_time2data for archived image, curl for latest
 ;     OMA serves latest AIA .fits files (lag usually within half hour)
 ;     JSOC serves only latest HMI magnetogram as .fits (longer lags)
 ; CALL:
 ;   see above
 ; INPUTS:
 ;   wav = '94','131','171','193','211','304','335','1600','1700','4500'
 ;         or 'magnetogram'
 ; OPTIONAL KEYWORD INPUTS
 ;   verbose = 1/0 (default 0)
 ; OUTPUTS:
 ;   image = array 
 ;   index = header info as tagged structure
 ; RESTRICTIONS:
 ;   requires rridl, coyotelib, lynx, curl
 ; HISTORY:
 ;   May 30 2019 RR: start for use in sdo_featurelocator.pro
;-

; answer no-parameter query 
if (n_params(0) lt 3) then begin
  print,'sdo_latest,wav,image,index,verbose=verbose'
  sp,sdo_latest
  return
endif

; defaults for keywords
if (n_elements(verbose) eq 0) then verbose=0

; check wav validity
if (wav ne '94' and wav ne '131' and wav ne '171' and wav ne '193' and $
    wav ne '211' and wav ne '304' and wav ne '335' and wav ne '1600' and $
    wav ne '1700' and wav ne '4500' and wav ne 'magnetogram') then begin
  print,' ##### sdo_latest abort: not a permitted wavelength'
  sp,sdo_latest
  return
endif

; ====== start

if (wav eq 'magnetogram') then begin

; ====== latest HMI magnetogram from JSOC

  jsocstring='http://jsoc.stanford.edu/data/hmi/images/latest/HMI_latest_Mag.fits'
  spawn,'curl -s '+jsocstring+' -o /tmp/latesthmi.fits '+$
    ' && echo "1" > /tmp/curlsuccess || echo "0"  > /tmp/curlsuccess'
  readcol,'/tmp/curlsuccess',success,format='I',/silent 
  if (success eq 0) then begin
    print,' ##### sdo_latest abort: curl failure jsocstring '+jsocstring
    return
  endif
  read_sdo,'/tmp/latesthmi.fits',index,image

endif else begin 

; ===== latest AIA image from OMA
  
; add zero(s) where needed for OMA dir string
  omawav=wav
  if (wav eq '131' or wav eq '171' or wav eq '193' $
      or wav eq '211' or wav eq '304' or wav eq '335') then omawav='0'+wav
  if (wav eq '94') then omawav='0094'

; set date into 0MA format
  spawn,"date --rfc-3339='date' > /tmp/omadate"
  readcol,'/tmp/omadate',omadatestring,format='A'
  omadatestring=repstr(omadatestring,'-','/')
  if (verbose) then print,' ---- omadatestring = '+omadatestring

; find URL of latest hour at OMA per lynx
;   following https://stackoverflow.com/questions/22510705
  spawn,"lynx -dump http://sdo.oma.be/latest/data/"+omawav+"/"+omadatestring+$
    " | awk '/H+[0-2]*/{url=$2}END{print url}' > /tmp/omalatesthour"
  readcol,'/tmp/omalatesthour',omahrstring,format='A'
  if (verbose) then print,' ---- omahrstring = '+omahrstring

; find URL of latest image at OMA per lynx
  spawn,"lynx -dump "+omahrstring+" | awk '/AIA*/{url=$2}END{print url}' > /tmp/omalatest"
  readcol,'/tmp/omalatest',omastring,format='A'
  if (verbose) then print,' ---- omastring = '+omastring
  
; get latest AIA fits file from OMA per curl (Mac OSX does not have wget)
  spawn,'curl -s '+omastring+' -o /tmp/latestaia.fits '+$
    ' && echo "1" > /tmp/curlsuccess || echo "0"  > /tmp/curlsuccess'
  readcol,'/tmp/curlsuccess',success,format='I',/silent 
  if (success eq 0) then begin
    print,' ##### sdo_latest abort: curl failure omastring '+omastring
    return
  endif

; get image (array) and index (header as structure)
  image=readfits('/tmp/latestaia.fits')
  mreadfits_header,'/tmp/latestaia.fits',index  ;RR found in read_sdo.pro

endelse ; end getting AIA image from OMA

; ===== show image
if (verbose) then showex,image  ;RR showex better than sv for zooming in 

end

; ================== test Hyper C

wav='171'
;; wav='magnetogram'
sdo_latest,wav,image,index,/verbose

end
