; file: sdo_getimagesequence.pro = get sequence one-wav full-disk SDO images
; init: Mar 15 2022  Rob Rutten  Deil from sdo_getimage.pro
; last: Mar 28 2022  Rob Rutten  Deil

;+
pro sdo_getimagesequence,taistart,taiend,wav,imarr,indexarr,$
  outdir=outdir,maxfiles=maxfiles,cadence=cadence,raw=raw

 ; Get sequence single-wavelength full-disk SDO images from JSOC archive,
 ; aia_prep them, intscale them, load into array in memory.
 ; Uses existing files instead when written in previous call.
 ; 
 ; INPUTS:
 ;   taistart, taiend: TAI values limiting the time sequence
 ;   wav: SDO diagnostic name (only one)
 ;     AIA: '94','131','171','193','211','304','335','1600','1700','4500',
 ;     HMI: 'mag','cont','dop'
 ;   NB: no 4500 with below ssw_jsoc_time2data sance ds series specifier

 ; KEYWORD INPUT OPTIONS:
 ;   outdir: file storage, default /tmp but may overflow
 ;   maxfiles: throttle number, default 100 (array 3G AIA, 6G HMI)
 ;   cadence: string, default or -1 = mimimum EUV 12s, UV 24s, HMI 45s
 ;   raw=1/0: no sdo_intscale or mucking to improve greyscale
 ; 
 ; OUTPUTS:
 ;   imarr = aia_prepped full-disk image array (may get <too> large!>
 ;   indexarr = array corresponding SDO header structures
 ;
 ; RESTRICTIONS:
 ;   needs SSW  and my https://robrutten.nl/rridl/rridl.zip
 ;   needs email registration at JSOC for archived 
 ;   works fine in my IDL 6.4 but may act differently in newer IDL (tell me?)
 ;   writing many such in /tmp may overflow / (/dev/sda2 for me)
 ;   JSOC balks when busy: do NOT run simultaneous with sdo_getdata_rr.pro
 ;
 ; HISTORY:
 ;   Mar 15 2022 RR: start for nisp_sdo.pro, from sdo_getimage.pro
 ;   Mar 17 2022 RR: repair JSOC order suggested by Sam Freeland
;-

; answer a no-parameter query
if (n_params() lt 5) then begin
  sp,sdo_getimagesequence
  return
endif

; default keywords
if (n_elements(outdir) eq 0) then outdir='/tmp/'
if (n_elements(maxfiles) eq 0) then maxfiles=100
if (n_elements(cadence) eq 0) then cadence=-1
if (n_elements(raw) eq 0) then raw=0

; SDO (AIA + HMI) level2 fitsfilename wavstrings
wavs=['94','131','171','193','211','304','335','1600','1700','4500',$  ; AIA
      'mag','cont','dop']                                              ; HMI
nwavs=n_elements(wavs)

; check wav specification 
iwav=-1
for iw=0,nwavs-1 do if (wavs[iw] eq wav) then iwav=iw
if (iwav eq -1 ) then begin
  print,' ##### sdo_getimagesequence abort: wav invalid = ',wav
  print,' ===== valid: ',wavs
  sp,sdo_getimagesequence
  return
endif
if (iwav eq 9) then begin
  print,'  ##### sdo_getimagesequence abort: 4500 does not load'
  return
endif

; check tai order
if (taiend lt taistart) then begin
  print,' ##### sdo_getimagesequence abort: taiend before taistart'
  return
endif

; define cadence default
if (cadence eq -1) then begin
  if (iwav lt 7) then cadence='12s'              ; EUV
  if (iwav eq 7 or iwav eq 8) then cadence='24s' ; UV
  if (iwav eq 9) then cadence='1h'   ; but 4500 doesn't load below 
  if (iwav gt 9) then cadence='45s'  ; HMI
endif

; check and warn re maxfiles
if (strpos(cadence,'s') gt 0) then cadencesec=fix(cadence)
if (strpos(cadence,'m') gt 0) then cadencesec=fix(cadence*60.)
if (strpos(cadence,'h') gt 0) then cadencesec=fix(cadence*3600.)
norder=fix((taiend-taistart)/cadencesec)
print,' ----- sdo_getimagesequences should order/read'+trimd(norder)+' files'
;RR AIA file size 33M (int), HMI mag 65M (float, silly)
if (norder gt maxfiles) then begin
  print,' ##### sdo_getimagesequence abort: ordered files > maxfiles'
  print,' nr files ordered ='+trimd(norder)+' or'+trimd(norder*33.)+'GB'
  return
endif

; check if previous downloads exist for this date + wav, then use those
dt=anytim2utc(taistart,/external)
dtstring='*'+trim(dt.year)+strmid(string(dt.month+1E2,format='(I3)'),1,4)+$
  strmid(string(dt.day+1E2,format='(I3)'),1,4)+'_*'+wav+'.fits' ;RR yak yak
filelist=file_search(outdir+'/'+dtstring)
nfiles=n_elements(filelist)
if (nfiles eq norder) then begin
  ;RR next gave strange errors read_sdo; also when reading whole sequence
  ;; read_sdo,filelist[0],index0,data0,/nodata
  ;; taifilestart=anytim2tai(index0.t_obs)
  ;; read_sdo,filelist[nfiles-1],index1,data1,/nodata
  ;; taifileend=anytim2tai(index1.t_obs)
    head0=headfits(filelist[0])
    tobs0=fxpar(head0,'t_obs') 
    taifilestart=anytim2tai(tobs0)
    head1=headfits(filelist[nfiles-1])
    tobs1=fxpar(head1,'t_obs')
    taifileend=anytim2tai(tobs1)
endif else begin
  taifilestart=0
  taifileend=0
endelse
; both number and start time and end time must correspond
if (nfiles eq norder and (abs(taifilestart-taistart) lt cadencesec) and $
  (abs(taifileend-taiend) lt cadencesec)) then begin 
  print,' ----- sdo_getimagesequences finds'+trimd(nfiles)+$
    ' files with this date-timespan-wav in '+outdir
  print,' ----- !!! it uses these; if wrong kill, rm -f *fits in '+outdir
  read_sdo,filelist,indexarr,imarr   ; get images, are aia_prepped
endif else begin
; no image files; get images from JSOC, aia_prep, write for reruns
;;;;      ds='aia.lev1',$ 
;RR Mar 16 2022 adding cadence balked, strange varying JSOC error messages
;RR tried full day, finally mail with demo to Sam Freeland
;RR remedy: take earlier ds specification out = some new Phil Scherrer quirk?
;RR apparently wav now defines series (but my JSOC cutouts still work OK)
;RR with ds specification out also no longer need to split AIA and HMI
;RR 4500 doesn't work but who cares, too lousy anyhow
  spawn,'rm -f '+outdir+'/'+ dtstring
  datetime1=anytim2utc(taistart,/vms)
  datetime2=anytim2utc(taiend,/vms)
  indexl1=0 ; avoid error strlastchar (Sam Freeland) in ssw_jsoc_time2data
  datal1=0 ; idem
  ssw_jsoc_time2data,datetime1,datetime2,indexl1,datal1,$
    cadence=cadence,waves=wavs[iwav],max_files=maxfiles,$
    /comp_delete,/uncomp_delete  ;RR no level-1 files left

; gives long comment sugggesting taking off output cube but no datal1 above
; stops at missing input
  
  ; aia_prep all images to level 2 ("1.5") and write in outdir for next time
  ; Nov 17 2020 add Gregal fix /no_mpo_update; Feb  3 2021 out again
  aia_prep,indexl1,datal1,indexarr,imarr,/do_write_fits,outdir=outdir
endelse

; ======= process images for better-contrast greyscale

if (raw ne 1) then begin

; loop per image
  nimages=n_elements(indexarr)
  for iim=0,nimages-1 do begin

; get image 
    indexim=indexarr[iim]
    im=imarr[*,*,iim]  ; non-intscaled = "raw" image
    
; below copied from sdo_getimage.pro 

; rescale image intensity with default clips

; get image size
    sizeim=size(im)
    nx=sizeim[1]
    ny=sizeim[2]

; AIA channel: sdo_intscale with standard clips
    if (iwav lt 9) then begin
      im=sdo_intscale(indexim,im,wav,clipmin=-1,clipmax=-1)
    endif

; AIA UV needs histop clipping for some reason, set values first for all
    if (iwav eq 7 or iwav eq 8) then begin
      if (iim eq 0) then begin
        imhist=histo_opt_rr(im,1E-5,indices,/top_only)
        imcent=imhist[nx/2-nx/4:nx/2+nx/4,ny/2-ny/4:ny/2+ny/4]
        mima=minmax(imhist)
        clipmin=mima[0]
        clipmax=mima[1]
        print,' ----- '+trimd(n_elements(indices))+' bright pixels clipped'
        print,' -----  sqrt likely improves UV scenes'
      endif
      im=im>(clipmin)<(clipmax)
    endif
    
; AIA 4500 is fairly  hopeless ##### doesn't load in this version anyhow
    if (iwav eq 9) then begin
      reformimage,im,ima,flatten=50
      subim=ima[nx/4:nx-nx/4,ny/4:ny-ny/4]
      subim=histo_opt_rr(subim)
      momim=moment(subim)
      meanim=momim[0]
      rms=sqrt(momim[1]) ;; /meanim
      cutrms=1
      imclip=ima>(meanim-cutrms*rms)<(meanim+cutrms*rms)
      im=imclip
    endif

; HMI magnetogram: clip for weaker fields, maintain zero
    if (iwav eq 10) then begin
; rescale to minmax center part first image
      if (iim eq 0) then begin
        subim=im[nx/2-nx/5:nx/2+nx/5,ny/2-ny/5:ny/2+ny/5] ; check
        momim=moment(subim)
        rms=sqrt(momim[1])
        cutrms=3
      endif
      imclip=im>(-cutrms*rms)<(+cutrms*rms)
      im=imclip
    endif

; HMI magnetogram:  remove JSOC-added extra speckled edge, blacken outside
    if (iwav eq 10) then begin
      sizeim=size(im)
      nx=sizeim[1]
      ny=sizeim[2]
      minim=min(im)
      rsunarcsec=indexim.rsun_obs
      rsunpx=rsunarcsec/indexim.cdelt1
      for ix=0,nx-1 do begin
        for iy=0,ny-1 do begin
          radpx=sqrt((ix-nx/2.)^2+(iy-ny/2.)^2)
          if (radpx gt rsunpx) then im[ix,iy]=minim 
        endfor
      endfor
    endif

; replace into result array
    imarr[*,*,iim]=im
    
  endfor ; end loop over images
endif    ; end of if raw eq 0

end ; end of program


; =================== main for tests per IDLWAVE H-c ======================

cd,'/home/rutten/tmp'  ;RR my /tmp, not system /tmp

datetime1='2019.11.11_15:17:00'   ; Mercury transit, sits near disk center 
datetime2='2019.11.11_15:18:00'
taistart=anytim2tai(datetime1)
taiend=anytim2tai(datetime2)
cadence=-1  ; default = fastest
wav='304'
outdir='/home/rutten/rr/tmp'

;; wav='mag'
;; wav='1600'
;; wav='4500'
;; wav='171'
;; wav='1700'

sdo_getimagesequence,taistart,taiend,wav,imarr,indexarr,$
  outdir=outdir,maxfiles=maxfiles,cadence=cadence,raw=raw

; inspect
showex,imarr

end
