; file: sdo_getimage.pro = get full-disk SDO image
; init: Apr 24 2020  Rob Rutten  Deil from sdo_featurelocator.pro
; last: Jul 31 2022  Rob Rutten  Deil

;+
pro sdo_getimage,specifier,wav,im,indexim,$
  outdir=outdir,raw=raw,latest=latest,intpolpair=intpolpair

 ; NAME:
 ;   sdo_getimage.pro
 ;
 ; DESCRIPTION:
 ;   Get full-disk image from JSOC archive or latest or existing file
 ;
 ; EXAMPLE CALL:
 ;   see end of program file
 ;
 ; INPUTS:
 ;   specifier: 3 options
 ;     archive mode: specifier='YYYY.MM.DD_HH:MM'
 ;       get full-disk AIA or HMI image (4Kx4K) from JSOC archive 
 ;       any time from Apr 15 2010 until a week ago
 ;       mandatory format, eg: '2013.12.21_12:00'
 ;     latest mode: specifier='latest'
 ;       get latest full-disk (1Kx1K ; any AIA; HMI 'magnetogram' only)
 ;       latest means half hour ago for AIA, more for HMI magnetograms
 ;     file mode: specifier = string path/filename of SDO image fits file
 ;       written by a previous call or by another program
 ;       (keywords wav and intpolpair inactive)
 ;       may also be outdir+'/latestaia.fits',outdir+'/latesthmi.fits'
 ;   wav: SDO diagnostic name
 ;     AIA: '94','131','171','193','211','304','335','1600','1700','4500' 
 ;     HMI: 'magnetogram','continuum','Dopplergram'
 ;          (for mode "latest" only 'magnetogram')
 ;
 ; KEYWORD INPUT OPTIONS:
 ;   outdir: default /tmp but may overflow
 ;   raw=1/0: if set then do not muck the intscale for better contrast
 ;   intpolpair=1/0: get and interpolate straddle pair instead of close time
 ;     NB: no correction for solar rotation, at disk center 10 arcsec/hr
 ;         so up to 0.25 arcsec X smearing for HMI pair, less towards limb
 ;         can't correct since (X,Y)-dependent
 ; 
 ; OUTPUTS:
 ;   im = aia_prepped full-disk image array
 ;   indexim = SDO header structure
 ;
 ; KEYWORD OUPUTS
 ;   latest: set when specifier = 'latest'
 ;
 ; RESTRICTIONS:
 ;   needs my rridl IDL library
 ;     https://robrutten.nl/rridl/rridl.zip
 ;   needs unix/linux curl and lynx for latest
 ;   needs email registration at JSOC for archived 
 ;   works fine in my IDL 6.4 but may act differently in newer IDL (tell me?)
 ;   writing many such in /tmp may overflow / (/dev/sda2 for me)
 ;   JSOC balks when busy: do NOT run simultaneous with sdo_getdata_rr.pro

 ; HISTORY:
 ;   Apr 24 2020 RR: excerpt from sdo_featurelocator.pro
 ;   May 11 2020 RR: 304 scaling, remove magnetogram edge
 ;   May 18 2020 RR: intpolpair
 ;   Jun 18 2020 RR: sensitivity degradation correction in sdo_intscale
 ;   Jul 31 2022 RR: MPO update issue
;-

; answer a no-parameter query
if (n_params() lt 4) then begin
  sp,sdo_getimage
  return
endif

; default keywords
if (n_elements(outdir) eq 0) then outdir='/tmp' 
if (n_elements(raw) eq 0) then raw=0
if (n_elements(intpolpair) eq 0) then intpolpair=0

; interprete specifier
latest=(specifier eq 'latest')
archive=(strmatch(specifier,'20*.*.*_*:*'))
file=(latest eq 0 and archive eq 0)

; check mode
if (latest+archive+file ne 1) then begin
  print, " ##### sdo_getimage abort:"+$
    " specifier not YYYY.MM.DD_HH:MM or 'latest' or SDO file"
  sp,sdo_getimage
  return
endif

; HMI no continuum or Dopplergram for latest
if (latest and (wav eq 'continuum' or wav eq 'Dopplergram')) then begin
  print, " ##### sdo_getimage abort:"+$
    " HMI mode 'latest' only 'magnetogram'"
  sp,sdo_getimage
  return
endif

; define permitted SDO (AIA + HMI) level1 fitsfilename strings
wavs=['94','131','171','193','211','304','335','1600','1700','4500', $  ; AIA
      'magnetogram','continuum','Dopplergram']                          ; HMI
nwavs=n_elements(wavs)
hmiseries=['hmi.M_45s','hmi.Ic_45s','hmi.V_45s']

; define bottom pedestals
defmin=[0, 0, 0, 1.5, 7, 10, 120, 30, 15, 3.5,$ ; AIA
        0., -8000., -5000.]                     ; HMI

; check wav specification 
iwav=-1
for iw=0,nwavs-1 do if (wavs[iw] eq wav) then iwav=iw
if (iwav eq -1 ) then begin
  print,' ##### sdo_getimage abort: wav invalid = ',wav
  print,' ===== valid: ',wavs
  sp,sdo_getimage
  return
endif

; ------ latest mode

if (latest) then sdo_latest,wav,im,indexim

; ----- archive mode = meaning get from archive at JSOC

if (archive) then begin
; convert specifier datetime into TAI
  t0=anytim2tai(specifier)

; get image close in time
  if (intpolpair eq 0) then begin
    if (iwav lt 10) then begin   ; AIA 12s or 24s cadence 
      datetime1=anytim2utc(t0-20,/vms)
      datetime2=anytim2utc(t0+20,/vms)
      ssw_jsoc_time2data,datetime1,datetime2,ind1,im1,ds='aia.lev1',$
        waves=[fix(wavs[iwav])],max_files=1,/uncomp_delete,/comp_delete
    end else begin                     ; HMI 45s cadence
      datetime1=anytim2utc(t0+36,/vms) ; UTC lags TAI by Delta T approx 36 s
      datetime2=anytim2utc(t0+82,/vms) ; 36+45=81 maybe next already
      ssw_jsoc_time2data,datetime1,datetime2,ind1,im1,ds=hmiseries[iwav-10],$
        max_files=1,/uncomp_delete,/comp_delete
    endelse

  ; aia_prep to level 1.5
  ; ## Nov 17 2020 add Gregal fix /no_mpo_update, out again Feb  3 2021 
 ;;    aia_prep,ind1,im1,indexim,im   ;; ,/no_mpo_update
  ; ## Jul 31 2022 back in again for bad stx_findlocation results
    aia_prep,ind1,im1,indexim,im,/no_mpo_update   ;; ,/no_mpo_update
    
;;  STOP ;RR May 17 2020 trials whether there is B pixel-size correction: no
  endif

  ; get straddle pair images and interpolate to desired time between
  if (intpolpair eq 1) then begin
; split between the three cadences
;   showex: near disk center rotation noticeable in HMI 45s pair
;   but same rotation affects co-alignment so straddle interpolation better
;   but so near disk center HMI pair gets smeared in X over 0.25 arcsec
    if (iwav lt 7) then begin ; EUV 12s cadence
      datetime1=anytim2utc(t0-14,/vms)
      datetime2=anytim2utc(t0+14,/vms)
      ssw_jsoc_time2data,datetime1,datetime2,inds,ims,ds='aia.lev1',$
;;;    notify='R.J.Rutten@uu.nl',$  ;RR put into SSW startup
        waves=[fix(wavs[iwav])],max_files=2,/uncomp_delete,/comp_delete
    endif
    if (iwav eq 7 or iwav eq 8) then begin ; UV 24s cadence
      datetime1=anytim2utc(t0-26,/vms)
      datetime2=anytim2utc(t0+26,/vms)
      ssw_jsoc_time2data,datetime1,datetime2,inds,ims,ds='aia.lev1',$
        waves=[fix(wavs[iwav])],max_files=2,/uncomp_delete,/comp_delete
    endif
    if (iwav gt 8) then begin ; HMI 45s cadence and Delta T timing difference
      datetime1=anytim2utc(t0+36-70,/vms) ; UTC lags TAI Delta T approx 36 s
      datetime2=anytim2utc(t0+36+70,/vms)
      ssw_jsoc_time2data,datetime1,datetime2,inds,ims,ds=hmiseries[iwav-10],$
        max_files=2,/uncomp_delete,/comp_delete
    endif

; check at least two images gotten
    if (n_elements(ims) lt 33554432l) then begin
      print,' ###### stx_getimage abort: failure obtaining two SDO images'
      print,'        inspect inds, ims'
    endif

; aia_prep first two images to level 1.5 and AIA ps size
; ## Nov 17 2020 add Gregal fix; Mar 15 2022 out again
; ## Jul 31 2022 back in again
    aia_prep,inds[0],ims[*,*,0],indexim1,im1,/no_mpo_update
    aia_prep,inds[1],ims[*,*,1],indexim2,im2,/no_mpo_update 
    
; interpolate straddle pair to t0 STX image datetime
    time1=anytim2tai(indexim1.t_obs)
    time2=anytim2tai(indexim2.t_obs)
    p=[[[im1]],[[im2]]]
    timefrac=(t0-time1)/(time2-time1)
    im=interpolate(p,timefrac)   ;RR bilinear, cubic no difference for 2)

; correct keyword t_obs (not sxaddpar since /str)
    indexim=indexim1
    indexim.t_obs=anytim2utc(t0,/ccsds)

;; ; check
;;     STOP
;;     showex,im1,im2,im
  endif ; end of intpolpair straddle pair

; write the SDO image in outdir for possible future use
  mwritefits,indexim,im,$
    outfile=outdir+'/sdo_'+specifier+'_'+wavs[iwav]+'.fits'
  
endif ; end get JSOC-archived image

; file mode: use already written aia_prepped image
if (file) then begin
; if rice-compressed (.fz) then uncompress
  dummy=cgrootname(specifier,extension=ext)  ;RR also needs coyotelib
  if (ext eq 'fz') then begin
    fileonly=file_basename(specifier+'_'+wavs[iwav],'.fz')
    tmpfile=outdir+'/'+fileonly
    spawn,'cp '+specifier+' '+tmpfile+'.fz'
    spawn,'funpack -D '+tmpfile +''
  endif else tmpfile=specifier
  mreadfits,tmpfile,indexim,im
endif

; ======= process image

; rescale image intensity with default clips
lev2wav=wav
if (iwav eq 10) then lev2wav='mag'
if (iwav eq 11) then lev2wav='cont'
if (iwav eq 12) then lev2wav='dop'

; set standard clips
clipmin=-1
clipmax=-1

; get image size
sizeim=size(im)
nx=sizeim[1]
ny=sizeim[2]

; AIA channel: sdo_intscale
if (iwav lt 9 and raw ne 1) then begin
  im=sdo_intscale(indexim,im,lev2wav,clipmin=clipmin,clipmax=clipmax)
;;  im=histo_opt_rr(im)
endif

; AIA 4500 is fairly  hopeless
if (iwav eq 9 and raw ne 1) then begin
  reformimage,im,ima,flatten=50
  subim=ima[nx/4:nx-nx/4,ny/4:ny-ny/4]
  subim=histo_opt_rr(subim)
  momim=moment(subim)
  meanim=momim[0]
  rms=sqrt(momim[1]) ;; /meanim
  cutrms=1
  imclip=ima>(meanim-cutrms*rms)<(meanim+cutrms*rms)
  im=imclip
endif

; HMI magnetogram: clip for weaker fields, maintain zero
if (iwav eq 10 and raw ne 1) then begin
; rescale to minmax center part
  subim=im[nx/2-nx/5:nx/2+nx/5,ny/2-ny/5:ny/2+ny/5] ; check
  momim=moment(subim)
  rms=sqrt(momim[1])
  cutrms=3
  imclip=im>(-cutrms*rms)<(+cutrms*rms)
  im=imclip
endif

; HMI magnetogram:  remove JSOC-added extra speckled edge, blacken outside
if (iwav eq 10) then begin
  sizeim=size(im)
  nx=sizeim[1]
  ny=sizeim[2]
  minim=min(im)
  rsunarcsec=indexim.rsun_obs
  rsunpx=rsunarcsec/indexim.cdelt1
  for ix=0,nx-1 do begin
    for iy=0,ny-1 do begin
      radpx=sqrt((ix-nx/2.)^2+(iy-ny/2.)^2)
      if (radpx gt rsunpx) then im[ix,iy]=minim 
    endfor
  endfor
endif

end


; =================== main for tests per IDLWAVE H-c ======================


;; wav='171'
;; wav='94'
;; wav='continuum'
;; wav='1700'
;; wav='1600'
wav='304'
;; wav='magnetogram'

;; sdo_getimage,'2018.01.01_00:00',wav,im,indexim ;; ,/intpolpair
sdo_getimage,'2019.11.11_15:17',wav,im,indexim,$
  outdir='/tmp'  ;; ,/intpolpair

;; sdo_getimage,'latest','magnetogram',im,indexim 
;; sdo_getimage,'/tmp/sdo_2019.12.10_20:12:00_continuum.fits','continuum',im,indexim

; inspect
showex,im

end
