; file: sdo_getdata_rr.pro = caller program SDO target cutout production
; init: Apr  6 2016  Rob Rutten  Deil
; last: Jan  2 2021  Rob Rutten  Deil
; note: if you are not Rob Rutten adapt items marked @?? below

;+
pro sdo_getdata_rr,datetimestart,duration,xsol,ysol,$
  email=email,name=name,$
  xsize=xsize,ysize=ysize,wavshmi=wavshmi,wavsaia=wavsaia,$
  centercadence=centercadence,targetcadence=targetcadence,$
  refwav=refwav,trimbox=trimbox,diffdesolrot=diffdesolrot,$
  notrackcenter=notrackcenter,notracktarget=notracktarget,$
  euvanchor=euvanchor,aiassw=aiassw,aligncenter=aligncenter,$
  clipmin=clipmin,clipmax=clipmax,no_intscale=no_intscale,$
  addfires=addfires,mpegs=mpegs,targetmovie=targetmovie,$
  centeronly=centeronly,targetonly=targetonly,$
  verbose=verbose

 ; NAME:
 ;   sdo_getdata_rr with _rr added because default email and name for me
 ;   
 ; PURPOSE:
 ;   order, download, process SDO image sequences using cutouts, first
 ;   large "center" fields at lower cadence to determine
 ;   cross-alignments including time-dependent drifts, then "target"
 ;   fields of the (likely small) target field and cross-align these.
 ;   
 ; INPUTS:
 ;   datetimestart (UTC): string, mandatory format '2015.09.27_07:00'
 ;   duration: minutes, e.g., 60 (no quotes, integer)
 ;   xsol, ysol: solar (X,Y) in arcsec of desired cutout center at start
 ;
 ; OPTIONAL KEYWORD INPUTS:
 ;   email: requester's (must be registered at JSOC), default mine  @??
 ;   name: requester's name, default me @?? 
 ;   xsize, ysize: cutout field of view in arcsec (default 150x150 for SST)
 ;   wavshmi: desired HMI products in 1D string array
 ;     (default: ['magnetogram', 'continuum'])
 ;   wavsaia: desired AIA channels in 1D string array
 ;     (example: ['171','195','1600'], default: all 7 EUV and 2 UV)
 ;   centercadence: string, default '2m' 
 ;   targetcadence: string, default 'fast'
 ;   refwav = string with reference wavelength name, default '171'
 ;     because the EUV wavs have fastest cadence = 12s, the UV wavs 24s
 ;     NB: do not use 1700 as reference since it skips 1 image/hour 
 ;         for the 4500AA one/hour continuum gathering
 ;   trimbox = align only over selected center subfield [xmin,ymin,xmax,ymax]
 ;   diffdesolrot > 0: differentially de-solar-rotate target
 ;   notrackcenter = 1/0: do not track center rotation.
 ;     Default: 0 (track) for duration < 600 min, 1 (notrack) for
 ;     longer, an arbitrary delimiter (center rotates 9 arcsec/hr).
 ;     With tracking the center field looses disk-center azimuthal
 ;     symmetry in subfield tile averaging from shifting across disk
 ;     center.  Without tracking subsequent center image straddle pairs
 ;     are smeared over the rotation shift during the centercadence
 ;     interval in their temporal interpolation to the reference
 ;     timing, so then better keep the center cadence interval short.
 ;   notracktarget = 1/0: do not track target (default: track)
 ;   euvanchor =  anchor EUV, default pair ['304','mag']
 ;   aiassw = 1/0 get AIA from SSW instead of from JSOC (slower, less robust)
 ;   aligncenter = 1/0 add center/cubesxal crossaligned cubes, default 0
 ;   clipmin = mimimum target intensity clip sdo_intscale.pro, default -1/-1
 ;   clipmax = maximum target intensity clip sdo_intscale.pro, default -1/0
 ;   no_intscale = 1: do not rescale target intensities (only HMIcont: /2.5)
 ;   addfires = 1/0: add 304x131 "fires" fitscubes (see LAR-1), default 0 
 ;   mpegs = 1/0: add mpegs of target/cubes, default 0 
 ;   targetmovie = 1/0:  add 4-panel target overview movie, default 0
 ;   centeronly = 1/0: no target download and processing (alignment tests)
 ;   targetonly = 1/0: no center stuff, no target aligns (rough quickies)
 ;   verbose=1/0: print and plot image shifts etc, default 0
 ;   
 ; OUTPUTS:
 ;   synchronized, crossaligned, dedrifted SDO fitscubes in target/cubes
 ;
 ; EXAMPLE:
 ;   below program end below
 ;
 ; RESTRICTIONS:
 ;   target (X,Y) must be within the limb
 ;   needs SSW and many routines in my rridl
 ;   other users than RR: add name and email specifications
 ;   
 ; HISTORY:
 ;   Apr  6 2016 RR: start 
 ;   Apr 20 2016 RR: streamlined to work as black box for others
 ;   Aug 18 2016 RR: optional ordering AIA @ SSW 
 ;   Aug 28 2016 RR: useheadpnt
 ;   Jan  3 2017 RR: useheadpnt but MPO values rotate and scale 
 ;   Mar 31 2017 RR: redid intensity clipping
 ;   Jan 10 2018 RR: no_intscale
 ;   Jul 14 2018 RR: segmentation for JSOC file limits
 ;   Jan 20 2020 RR: check presence alignment wavelengths
 ;   Feb  8 2020 RR: notrack, euvanchor; noheadpnt out 
 ;   Apr  2 2020 RR: refwav from 304 tot 171 for notrack problems
 ;   May 28 2020 RR: improved start-stop timings
 ;   Jun  4 2020 RR: notrack split center, target
 ;   Jul  6 2020 RR: diffdesolrot
 ;   Jul 20 2020 RR: targetonly (for quickie roughs; never used since?)
;-

; answer no-parameter query 
if (n_params(0) lt 4) then begin
  sp,sdo_getdata_rr  ; shows header in separate editor window
  return
endif

; defaults for keywords
if (n_elements(name) eq 0) then name='Rob Rutten'         ; @??
if (n_elements(email) eq 0) then email='robenrietjerutten@gmail.com' ; @?? 
if (n_elements(xsize) eq 0) then xsize=150.
if (n_elements(ysize) eq 0) then ysize=150.
if (n_elements(wavshmi) eq 0) then wavshmi=['magnetogram','continuum']
if (n_elements(wavsaia) eq 0) then $
  wavsaia=['94','131','171','193','211','304','335','1600','1700']
if (n_elements(centercadence) eq 0) then centercadence='2m'
if (n_elements(targetcadence) eq 0) then targetcadence='fast'
if (n_elements(refwav) eq 0) then refwav='171'
if (n_elements(trimbox) eq 0) then trimbox=[-1,-1,-1,-1]
if (n_elements(notrackcenter) eq 0) then notrackcenter=0
if (n_elements(notracktarget) eq 0) then notracktarget=0
if (n_elements(diffdesolrot) eq 0) then diffdesolrot=0
if (n_elements(euvanchor) eq 0) then euvanchor=['304','mag']
if (n_elements(aiassw) eq 0) then aiassw=0
if (n_elements(aligncenter) eq 0) then aligncenter=0
if (n_elements(clipmin) eq 0) then clipmin=-1   ; CAREFUL!
if (n_elements(clipmax) eq 0) then clipmax=0
if (n_elements(no_intscale) eq 0) then no_intscale=0
if (n_elements(addfires) eq 0) then addfires=0 
if (n_elements(mpegs) eq 0) then mpegs=0
if (n_elements(targetmovie) eq 0) then targetmovie=0
if (n_elements(centeronly) eq 0) then centeronly=0
if (n_elements(targetonly) eq 0) then targetonly=0
if (n_elements(verbose) eq 0) then verbose=0

; set wall-clock timer (seconds)
jobstartgetdata=systime(1) 

; print pacifier
if (verbose ne 0) then print,' ===== sdo_getdata_rr started at UT = '+$
  anytim2utc(timestartgetdata,/vms)

; check there is something to do
if (centeronly eq 1 and targetonly eq 1) then begin
  print,' ##### abort: setting centeronly and targetonly = do nothing'
  return
endif

; check mandatory format of datetimestart
if (strlen(datetimestart) ne 16 or $
    strmid(datetimestart,4,1) ne '.' or $
    strmid(datetimestart,7,1) ne '.' or $
    strmid(datetimestart,10,1) ne '_' or $
    strmid(datetimestart,13,1) ne ':') then begin
  print, ' ##### sdo_getdata_rr abort: datetimestart not YYYY.MM.DD_HH:MM'
  return
endif

; check target center within limb
sundata=get_sun(datetimestart) ; geocentric physical solar ephemeris
if (sqrt(xsol^2+ysol^2) gt sundata[1]) then begin
  print,' #####  sdo_getdata_rr abort: (xsol,ysol) location beyond limb'
  return
endif

; check notrack vs diffdesolrot
if (notracktarget and diffdesolrot) then begin
  print,' ------ setting notracktarget implies no derot, diffdesolrot set to 0'
  diffdesolrot=0
endif

; check euvanchor
if (not(euvanchor[1] eq '1600' or euvanchor[1] eq '1700' or $
        euvanchor[1] eq 'mag') and $
    not(euvanchor[0] eq '304' or euvanchor[0] eq '131')) then begin
  print, ' ##### WARNING: non-included euvanchor = '+euvanchor
  wait,30
endif

; check presence of alignment wavelengths
if (targetonly eq 0) then begin 
  dummy=where(wavshmi eq 'magnetogram',ordermag)
  dummy=where(wavsaia eq '1700',order1700)
  dummy=where(wavsaia eq '304',order304)
  dummy=where(wavsaia eq '171',order171)
  dummy=where(wavsaia eq '211',order211)
  dummy=where(wavsaia eq '94',order94)
  if (ordermag*order304 eq 0) then begin
    print,' ##### sdo_getdata_rr ABORT: need magnetogram + 304 for EUV align'
    return
  endif
  if (order304 eq 0 or order171 eq 0 or order211 eq 0 or order94 eq 0) then $
    print,' ##### WARNING: no 171 or 211 or 94 may fail further alignments'
endif

; reset notrackcenter for long duration
if (targetonly eq 0 and duration gt 600) then begin
  notrackcenter=1
  print,' ===== default notrackcenter put to 1 for long duration'
  print,'       Otherwise cross-alignment hampered by center field drift'
  print,'       Consider instead splitting your order into parts'
endif 

; get current dir path; \ flag spaces in path if present (bloody Mac?)
; https://groups.google.com/forum/#!topic/comp.lang.idl-pvwave/Lo10H5XtN80
cd,current=thisdir
thisdir=strjoin(strsplit(thisdir,' ',/extract),'\ ') 

; define subdirs
if (targetonly eq 0) then begin
  centerdir=thisdir+'/center'
  driftsdir=thisdir+'/driftscenter'
  centerlevel1dir=centerdir+'/level1'
endif
if (centeronly eq 0) then begin
  targetdir=thisdir+'/target'
  targetlevel1dir=targetdir+'/level1'
endif

; delete previous stuff 
if (targetonly ne 1) then begin
  spawn,'rm -rf '+centerdir
  spawn,'rm -rf '+driftsdir
endif
if (centeronly ne 1) then spawn,'rm -rf '+targetdir

; make subdirs
if (targetonly eq 0) then begin
  spawn,'mkdir -p '+centerdir
  spawn,'mkdir -p '+centerlevel1dir
  spawn,'mkdir -p '+driftsdir
endif
if (centeronly eq 0) then begin
  spawn,'mkdir -p '+targetdir
  spawn,'mkdir -p '+targetlevel1dir
endif

; check/adapt centercadence to get minimum 12 samples 
if (targetonly eq 0) then begin
  cadcen=float(centercadence)  ; minutes
  if (strpos(centercadence,'s') gt 0) then cadcen=float(centercadence)/60. 
  if (strpos(centercadence,'h') gt 0) then cadcen=float(centercadence)*60. 
  if (duration/cadcen lt 12) then begin
    for i=0,20 do begin 
      cadcen=cadcen/2. 
      if (float(duration)/cadcen gt 12) then break 
    endfor
    if (cadcen lt 0.8) then cadcen=0.8  ;RR minimum 48s = 4 EUV, 2 UV, 1 HMI
    centercadence=trim(cadcen)+'m'
    print,' ===== centercadence changed to '+centercadence
  endif
endif

; set up time segmentation if needed to stay within JSOC "record" limits
nsegments=1
if (centeronly eq 0) then begin
; "Limit for AIA to about 15,000 and for HMI about 30,000 in each request"
; (a JSOC "record" is a level1 fits file I think) 
  if (targetcadence eq 'fast') then begin 
    cadeuv=0.2       ; 12s in minutes
    caduv=0.4        ; 24s in minutes
    cadhmi=0.75      ; 45s in minutes
  endif else begin   ;RR assume specified as sec or min or hours
    if (strpos(targetcadence,'s') gt 0) then targetcad=targetcadence/60. 
    if (strpos(targetcadence,'m') gt 0) then targetcad=targetcadence
    if (strpos(targetcadence,'h') gt 0) then targetcad=targetcadence*60. 
    if (targetcad ge 0.2) then cadeuv=targetcad else cadeuv=0.2
    if (targetcad ge 0.4) then caduv=targetcad else caduv=0.4 
    if (targetcad ge 0.75) then cadhmi=targetcad else cadhmi=0.75
  endelse
  neuv=fix(total(fix(wavsaia lt 1500)))
  nuv=fix(total(fix(wavsaia gt 1500)))
  nhmi=n_elements(wavshmi)
  maxrecord=max([neuv*duration/cadeuv,nuv*duration/caduv,duration/cadhmi])
  nsegments=fix(maxrecord/14500.)+1   ; JSOC limit for AIA = 15000
;; nsegments=fix(maxrecord/200.)+1     ; !!! fake for segmentation tests
endif

; timings
segduration=fix(float(duration)/nsegments)
tstarttai=anytim2tai(datetimestart)

; print warning at segmentation
if (nsegments gt 1) then begin
  print,' ##### WARNING: request ='+trimd(long(maxrecord))+$
    ' files exceeds JSOC limit; split into'+trimd(nsegments)+' segments'
endif

; print rough output size estimate
if (centeronly eq 0) then begin
  sizetarget=neuv*duration/cadeuv+nuv*duration/caduv+nhmi*duration/cadhmi
  sizetarget=sizetarget*(xsize/0.6)*(ysize/0.6)
  sizetarget=sizetarget*3*2./1.E9 ; GB level1, level2, cubes for target
endif else sizetarget=0
if (targetonly eq 0) then begin
  sizecenter=(neuv+nuv+nhmi)*(duration/cadcen)*(750./0.6)*(750./0.6)
  sizecenter=sizecenter*5*2./1.E9 ;  GB level1, level2, cubes for center
endif else sizecenter=0
print,' ===== warning: may take long; '+$
  ' generates'+trimd(nsegments*8-(centeronly+targetonly)*4)+' JSOC emails,'+$
  ' roughly '+trimd(fix(sizecenter+sizetarget))+' GB files'

; =============== start big data-getting loop over time segments

for isegment=0,nsegments-1 do begin

; increase last segment to get samples beyond requested end 
;RR May 24 2020: cadcen extra for center only; suggested by Krishna Prasad
;RR Jun 12 2020: center add 5 min to accommodate euvanchor delaymin 
  if (targetonly eq 0) then begin
    centersegduration=segduration
    if (isegment eq nsegments-1) then begin
      centersegduration=fix(fix(segduration/cadcen)*cadcen+max([5,cadcen]))
      segduration=segduration+1 ; target add 1 minute to last segment
    endif
  endif

; define start (= datetimestart or previous end) and end for this segment
  if (isegment eq 0) then taisegstart=tstarttai else taisegstart=taisegstop
  tsegstart=anytim2utc(taisegstart,/ccsds) 
  strput,tsegstart,'_',10            ;RR change into my datetime input format
  tsegstart=strmid(tsegstart,0,16)   ; take off :xx seconds; is still UT
  taisegstop=taisegstart+60.*segduration   ; will be start of next segment
  tsegstop=anytim2utc(taisegstop,/ccsds)   ; tsegstop is only for printout
  strput,tsegstop,'_',10  
  tsegstop=strmid(tsegstop,0,16)

; define center (X,Y) location for ordering
  if (targetonly eq 0) then begin
; track center: set center (X,Y) East to rotate through (0,0) midway 
    if (notrackcenter eq 0) then begin
      xcenstart=fix(-duration*9./60./2.+0.5)  ; center rotation X = 9"/h
      ycenstart=0                             ; Y nonzero but negligibe  
      xcen=xcenstart
      ycen=ycenstart
      if (xcenstart lt -100) then $ ;RR leftside field at X=-100-375 arcsec
        print,' ##### long order has center field starting at xcenstart ='+$
        trimd(xcenstart)+' which may upset drift determinations; split order?'
; differentially rotate next center segment 
      if (isegment gt 0) then begin
        xycen=rot_xy(xcenstart,ycenstart,taisegstart-tstarttai,$
                     date=datetimestart)
        xcen=xycen[0]
        ycen=xycen[1]
      endif
    endif else begin
; notrackcenter=1: stay at disk center
      xcen=0
      ycen=0
    endelse
  endif
  
; define target (X,Y) location for ordering
  if (centeronly eq 0) then begin
    xtar=xsol
    ytar=ysol
; track target: rotate segment start over previous segment rotation
; serves to adjust JSOC cutouts; images get aligned over jumps
    if (notracktarget eq 0) then begin
      if (isegment eq 0) then xytar=[xsol,ysol] else begin
        xytar=rot_xy(xsol,ysol,taisegstart-tstarttai,date=tsegstart)
        xtar=xytar[0]
        ytar=xytar[1]
      endelse
    endif
  endif
  
; print segment info 
  if (nsegments gt 1) then begin
    print,' ===== start on segment'+trimd(isegment+1)+$
      '   segduration ='+trimd(segduration)
    print,'       tsegstart = '+tsegstart+'   tsegstop  = '+tsegstop
    print,'       xycenter = '+trimd(xcen)+', '+trimd(ycen)+$
      '   xytarget = '+trimd(xtar)+', '+trimd(ytar)
  endif

; set skipjsoc=1 for tests on segment timing only
  skipjsoc=0  ; !!! set 1 for fake stop for testing segment timing
  if (skipjsoc) then goto, SKIPJSOC

; -------- order data, either HMI+AIA at JSOC or only AIA via SSW

  if (aiassw ne 0) then wavsjsoc=wavshmi else wavsjsoc=[wavsaia,wavshmi]
  
; order disk-center data at JSOC (for offset and drift determinations)
  if (targetonly eq 0) then begin
;RR notrack=0: JSOC enables tracking for its cutout specification
    print,' ======== sdo_getdata_rr starts ordering center data at JSOC'
    sdo_orderjsoc,tsegstart,centersegduration,xcen,ycen,email,name,$
      wavs=wavsjsoc,xsize=750,ysize=750,cadence=centercadence,$
      notrack=notrackcenter,centeridents,centersizes 

; optionally order disk-center AIA data from SSW instead of JSOC
    if (aiassw ne 0) then begin
      print,' ======== sdo_getdata_rr starts ordering center AIA data via SSW'
      sdo_orderssw,tsegstart,centersegduration,xcen,ycen,email,$
        wavs=wavsaia,xsize=750,ysize=750,cadence=centercadence,$
        notrack=notrackcenter,centersswident 
    endif
  endif
  
; order target data at JSOC unless /centeronly; now notrack is an option
  if (centeronly ne 1) then begin
    print,' ======== sdo_getdata_rr starts ordering target data at JSOC'
    sdo_orderjsoc,tsegstart,segduration,xtar,ytar,email,name,$
      wavs=wavsjsoc,xsize=xsize,ysize=ysize,cadence=targetcadence,$
      notrack=notracktarget,targetidents,targetsizes 

; optionally order target AIA data from SSW
    if (aiassw ne 0) then begin
      print,' ======== sdo_getdata_rr starts ordering target AIA data via SSW'
      sdo_orderssw,tsegstart,segduration,xtar,ytar,email,$
        wavs=wavsaia,xsize=xsize,ysize=ysize,cadence=targetcadence,$
        notrack=notracktarget,targetsswident
    endif
  endif
  
; ---------- get data

; get disk-center data from JSOC  (sdo_getjsoc will wait if not yet there)
  if (targetonly eq 0) then begin
    print,' ======== sdo_getdata_rr starts getting JSOC center data'
    sdo_getjsoc,centeridents,centerlevel1dir 
  endif
  
; get target data from JSOC (sdo_getjsoc will wait if not yet there)
  if (centeronly ne 1) then begin
    print,' ======== sdo_getdata_rr starts getting JSOC target data'
    sdo_getjsoc,targetidents,targetlevel1dir

; optionally get AIA data via SSW  (sdo_getssw will wait if not yet there)
    if (aiassw ne 0) then begin
      print,' ======== sdo_getdata_rr starts getting center SSW AIA data'
      sdo_getssw,centersswident,centerlevel1dir
      print,' ======== sdo_getdata_rr starts getting target SSW AIA data'
      sdo_getssw,targetsswident,targetlevel1dir
    endif
  endif

  SKIPJSOC:
endfor   ; ===== end big data-getting loop over time segments

if (skipjsoc eq 1) then STOP

; ----------- find and apply offsets and drifts

; determine offsets and drifts from /center and apply to /target
if (targetonly eq 0) then sdo_alignaia_rr,sdoprep=1,refwav=refwav,$
  trimbox=trimbox,diffdesolrot=diffdesolrot,$
  notrackcenter=notrackcenter,notracktarget=notracktarget,$
  euvanchor=euvanchor,aligncenter=aligncenter,$
  border=border,centeronly=centeronly,$
  clipmin=clipmin,clipmax=clipmax,no_intscale=no_intscale,$
  addfires=addfires,mpegs=mpegs,targetmovie=targetmovie,$
  verbose=verbose

if (targetonly eq 1) then begin
  sdo_maketargetfitscubes,targetdir=targetdir,driftsdir=driftsdir,$
    sdoprep=1,refwav=refwav,diffdesolrot=diffdesolrot,$
    clipmin=clipmin,clipmax=clipmax,no_intscale=no_intscale,$
    nocrossalign=1,border=border,addfires=addfires,$
    mpegs=mpegs,targetmovie=targetmovie,moviedir=moviedir
  spawn,'touch '+targetdir+'/cubes/00-NB_cubes_NOT_aligned'
endif

; ----------  done; print elapsed time    
timedone=systime(1)
timelaps=ntostr((timedone-jobstartgetdata)/60.,format='(F11.1)')
print,' ===== sdo_getdata_rr done; took '+timelaps+' min'

end

; =============== main for testing per IDLWAVE H-c ======================

;; ----- standard small test run (25 min on my laptop)
cd,'/home/rutten/data/SDO/2014-06-14-small'
sdo_getdata_rr,'2014.06.14_09:00',10,-432,-312,xsize=90,ysize=90,$
  /addfires

;; ; inspect
;; showex,/allsdo,sdodirs=['center/cubes','center/cubesxal']

;; showex,/allsdo,sdodirs='target/cubes'
;RR wow: at end multiple patches wide apart flare together

end
