; file: sdo_fitscubes2crispex.pro
; init: Sep 13 2014  Rob Rutten  Dublin
; last: Dec 17 2018  Rob Rutten  Deil

pro sdo_fitscubes2crispex,cubesdir,instring,outfile,$
  diagname=diagname,bytscale=bytscale,$
  verbose=verbose

;+
 ; NAME:
 ;   sdo_fitscubes2crispex.pro  
 ;
 ; PURPOSE:
 ;   convert multiple fitscubes into 1 La Palma (nonfits) crispex input cube
 ;
 ; CALL:
 ;   sdo_fitscubes2crispex,cubesdir,instring,outfilestart,$
 ;     bytscale=bytscale,verbose=verbose
 ;
 ; INPUTS:
 ;   cubesdir = string with dirname where SDO cubes are, end with /
 ;   instring = string with filename part for selection, eg _gbo.fits'
 ;   outfilestart = string with output dirname/filename without extension    
 ;
 ; OPTIONAL KEYWORD INPUTS:
 ;   diagname=1/0: label byte output images with SDO channel identifier
 ;   bytscale=0: some output format as input (float, integer, byte)
 ;   bytscale=1: bytscale with min and max of first image as cutoffs
 ;   bytscale=2: bytscale with min and max of whole cubefile as cutoffs
 ;   verbose=1/0: print progress 
 ;
 ; OUTPUTS:
 ;   file outfile+extension
 ;    - dimensions: nx,ny,nt*nw
 ;        nw = number SDO channels (instead of CRISP line positions nlp)
 ;        nw and nt specified in outfilename: crispex needs mw, showex nt
 ;    - added extension: .bcube, .icube, or .fcube depending on data type
 ;
 ; USAGE EXAMPLE:
 ;  CRISPEX call together with a CRISP line-sampling data set:
 ;     crispex,'sdo_crispex_nw8_nt222.bcube',single_cube=8,$
 ;     refcube=['crispex.6563.08:02:31.time_corrected.aligned.icube',$
 ;              'crispex.6563.08:02:31.time_corrected.aligned_sp.icube']
 ;
 ; RESTRICTIONS:
 ;   - output format is old SST "La Palma", not fits (as for IRIS)
 ;   - only frugal header size 512 (not containing nt, nw alas, silly)
 ;   - file name.icube, dimensions [nx,nt,nt*nw]
 ;
 ; HISTORY:
 ;   Oct 19 2014 RR: start
 ;   Feb 11 2015 RR: removed option spectrumfile
 ;   Sep 12 2015 RR: use headfits_rr to accommodate Mac dir with space
 ;   Feb 29 2016 Rob rutten: corrected typo to also insert AIA 211
 ;   Dec 17 2018 RR: nt into filename for showex calls
;-

; answer no-parameter query
if (n_params() lt 3) then begin
  print,'  sdo_fitscubes2crispex,cubesdir,instring,outfile,'
  print,'   [diagname=diagname,bytscale=bytscale,verbose=verbose]'
  return
endif

; defaults for keywords
if (n_elements(diagname) eq 0) then diagname=0
if (n_elements(bytscale) eq 0) then bytscale=0
if (n_elements(verbose) eq 0) then verbose=0

; print opener
if (verbose) then print,' ===== sdo_fitscubes2crispex.pro starts'

; define permitted SDO channel strings in desired crispex display order
wavs=['cont','mag','dop','1700','1600','304','171','193',$
      '335','211','131','94']
nwavs=n_elements(wavs)
channel=strarr(nwavs)

; find pertinent (instring-selected) SDO fitscube files
ifile=-1
infiles=strarr(1)
for iwav=0,nwavs-1 do begin
  infile=findfile(cubesdir+'/*'+wavs[iwav]+'*'+instring+'*')
  if (infile ne '') then begin
    ifile=ifile+1
    channel[ifile]=wavs[iwav]
    if (ifile eq 0) then infiles[0]=infile else infiles=[infiles,infile]
  endif
endfor
nchannels=1l*n_elements(infiles)  ;RR long because iwit may get > 32000

; verbose: print channels of files found
if (verbose) then begin
  print,' ===== '+ntostr(nchannels)+' SDO channels found: '
  print,'   '+ntostr(channel[0:nchannels-1])
endif

; start big loop over input files
for ichannel=0,nchannels-1 do begin

  if (verbose) then print,' ===== file nr '+ntostr(ichannel)+' = '+infiles[ichannel]

; first file: get cube geometry and file datatype from the fits header
  if (ichannel eq 0) then begin
    header_in=headfits_rr(infiles[ichannel])
    header_insize=(1+fix(n_elements(header_in)/36.))*2880
    inbitpix=fxpar(header_in,'bitpix')
    nx_in=fxpar(header_in,'naxis1') 
    ny_in=fxpar(header_in,'naxis2') 
    nt_in=fxpar(header_in,'naxis3') 
    if (nx_in eq 0 or ny_in eq 0 or nt_in eq 0) then begin
      print,' ###### no proper fitscube since no nx, ny, nt in header '+infile
      return
    endif
  endif

; open current input file for assoc 
;RR fits files are big_endian but intel (my Toshiba) is litle_endian?
;RR little_endian=(BYTE(1,0,1))[0]  
;RR shows indeed my intel is little (what's in a name - littintel)
  get_lun, unit_in
  openr,unit_in,infiles[ichannel],/swap_if_little_endian
  if (inbitpix eq -32) then $
    inassoc=assoc(unit_in,fltarr(nx_in,ny_in),header_insize)
  if (inbitpix eq 16) then $
    inassoc=assoc(unit_in,intarr(nx_in,ny_in),header_insize)
  if (inbitpix eq 8) then $
    inassoc=assoc(unit_in,bytarr(nx_in,ny_in),header_insize)

; check input image size
  im0=inassoc[0]
  sizeim0=size(im0)
  nx_im0=sizeim0[1]
  ny_im0=sizeim0[2]
  if (nx_im0 ne nx_in or ny_im0 ne ny_in) then begin
    print, ' ##### first image [nx,ny] unequal to nx or ny in first header'
    return
  endif

; if bytscale set get min and max of the whole cube
  if (bytscale ne 0) then begin
    chanmin=1.e10
    chanmax=-1.e10
    if (bytscale eq 1) then ntminmax=1
    if (bytscale eq 2) then ntminmax=nt_in
    for it=0,ntminmax-1 do begin
      image=inassoc[it]
      if (min(image) lt chanmin) then chanmin=min(image)
      if (max(image) gt chanmax) then chanmax=max(image)
    endfor
; maintain zero value for magnetograms and Doppler maps
    if (channel[ichannel] eq 'mag' or channel[ichannel] eq 'dop') then begin
      minzero=min([chanmin,-chanmax])
      maxzero=max([chanmax,-chanmin])
      limit=max([abs(minzero),maxzero])
      chanmax=limit
      chanmin=-limit
    endif
  endif

; first channel: start output file
  if (ichannel eq 0) then begin

; set output parameters
    nx_out=nx_in
    ny_out=ny_in
    nt_out=nt_in
    outbitpix=inbitpix
    if (bytscale ne 0) then begin
      outbitpix=8
      im0=bytscl(im0)
    endif 
    blank=bytarr(nx_out,ny_out)

; define outfile extension including nr of wavelengths and nt_out
    strnwnt='_nw'+ntostr(nchannels)+'_nt'+ntostr(nt_out)
    if (outbitpix eq -32) then fileext=strnwnt+'.fcube'
    if (outbitpix eq 16) then fileext=strnwnt+'.icube'
    if (outbitpix eq 8) then fileext=strnwnt+'.bcube'
    
; set header_out
    header_out=make_lp_header(im0,nt=nt_out*nchannels)
    header_out=str_replace(header_out,' datatype',$
                           'stokes=[I], ns=1 :  datatype')
    header_outsize=512

; open output file for assoc (little now; swap for others than my intel) 
    get_lun, unit_out
    openw,unit_out,outfile+fileext,/swap_if_big_endian 
    if (outbitpix eq -32) then $
      outassoc=assoc(unit_out,fltarr(nx_out,ny_out),header_outsize)
    if (outbitpix eq 16) then $
      outassoc=assoc(unit_out,intarr(nx_out,ny_out),header_outsize)
    if (outbitpix eq 8) then $
      outassoc=assoc(unit_out,bytarr(nx_out,ny_out),header_outsize)

; write header_out
    if (header_outsize ne 0) then begin
      rec=assoc(unit_out, bytarr(header_outsize))
      rec[0]=byte(header_out)
    endif

; end of startup output for channel=0
  endif  

; set diagname overlay image for this channel
  if (diagname ne 0 and bytscale ne 0) then  begin
    while !d.window ne -1 do wdelete,!d.window 
    window,xsize=nx_out,ysize=ny_out,/pixmap,retain=2
    !p.font=-1
    tv,blank
    xyouts,ny_out/30,ny_out/30,channel[ichannel],color=255,$
      /device,charsize=sqrt(ny_out)/10.,charthick=sqrt(ny_out)/10.
    imdiagname=tvrd()
  endif

; write all images of this file iw-distributedly into the output file
  for it=0,nt_out-1 do begin
    image=inassoc[it]
    if (bytscale ne 0) then image=bytscl(image,min=chanmin,max=chanmax)
    if (diagname ne 0 and bytscale ne 0) then $
      image[where(imdiagname gt 1)]=255
    iwit=it*nchannels+ichannel
    outassoc[iwit]=image
  endfor

; free the input file 
  free_lun,unit_in

endfor

; free the output file
free_lun,unit_out

if (verbose ne 0) then print,$
  ' ===== sdo_fitscubes2crispex wrote file '+outfile+fileext

end

; ==================== test per IDLWAVE super C ======================

cubesdir='/home/rutten/data/SST/2016-09-05-demo/sdo2sst'
instring='_for_ip.fits'
outfilestart='/tmp/sdo2crispex'
spawn,'rm -f '+outfile+'*cube'
sdo_fitscubes2crispex,cubesdir,instring,outfilestart,$
  /diagname,bytscale=0,/verbose

crispfile=findfile(outfilestart+'*cube')
crispex,crispfile,single_cube=11
showex,crispfile,nt_mw=35



end

