; file: sdo_firelevel.pro = determine SDO fire detection levels
; init: Sep  7 2020  Rob Rutten  Deil
; last: Dec 31 2020  Rob Rutten  Deil

;+
pro sdo_firelevel,cube304,cube131,aia3013,aia3013cut,aia3013color,$
  heightdiff=heightdiff,header=header,$
  cutactive=cutactive,cutlimit=cutlimit,colorlimit=colorlimit,niter=niter

 ; set SDO fire detector levels (see Lingezicht Astrophysics Reports 1)
 ;   
 ; INPUTS (mandatory):
 ;  cube304, cube131: 3D image sequence cubes (AIA 304 and 131) 
 ;   
 ; OUTPUTS (mandatory):
 ;  aia3013: product cube
 ;  aia3013cut: aia3013 intensity clip value for overexposing ARs
 ;  aia3013color: aia3013 intensity threshold value for marking fires
 ;
 ; OPTIONAL KEYWORD INPUTS:
 ;  heightdiff: "formation height" im1 above im2 in km (default 0)
 ;  header: RR cubefile header with SDO keywords, mandatory for heightdiff
 ;  cutactive: times sigma cut in activity removal iteration (default 2)
 ;  cutlimit: times sigma cut after iteration for byte scale clip (10)
 ;  colorlimit: times sigma level for coloring above, after iteration (5)
 ;  niter: number of activity-removal iterations (10)
 ;
 ; WARNINGS:
 ;   limit values are arbitrary choices requiring experimentation
 ;   heightdiff applies to whole field, better for smaller limbward fields
 ; 
 ; HISTORY:
 ;   Sep  8 2020 RR: start for "SolO campfires" in LAR-1
;-

; answer no-parameter query 
if (n_params(0) lt 5) then begin
  sp,sdo_firelevel
  return
endif

; default keyword values
if (n_elements(heightdiff) eq 0) then heightdiff=0.
if (n_elements(header) eq 0) then header=''
if (n_elements(cutactive) eq 0) then cutactive=2
if (n_elements(cutlimit) eq 0) then cutlimit=10
if (n_elements(colorlimit) eq 0) then colorlimit=5
if (n_elements(niter) eq 0) then niter=10

; usually sdo_firelevel is used on center cutouts, neglect heightdiff
; and anyhow I believe that these fires are footpoints at commom height
; in 304 and 131 brightened by particle-beam heating
if (heightdiff eq 0) then cube131cor=cube131

; apply heightdiff = shift cube B, as in findimshift_tiled.pro
if (heightdiff ne 0) then begin

; check header presence 
  if (header[0] eq '') then begin
    print,' ##### sdo_firelevel abort: heightdiff requires header'
    return
  endif

; astrometry
  arcsecpx=0.6
  rsun_obs=fxpar(header,'rsun_obs')
  rsunpx=rsun_obs/arcsecpx
  rsunkm=696340.
  solx=fxpar(header,'xcen')
  soly=fxpar(header,'ycen')
  solvec=sqrt(float(solx)^2+soly^2)
  sintheta=solvec/rsun_obs                  ; r/R value
  delradkm=heightdiff*sintheta              ; projected height difference
  delradpx=delradkm*(rsunpx/rsunkm)         ; in px units

; position angle of sol (X,Y) vector from disk center
  gamma=atan(abs(soly)/(abs(solx)>0.01))   ; 0.01 avoids y/0=NaN
  delradx=delradpx*cos(gamma)
  delrady=delradpx*sin(gamma)
; correct quadrant signs
  if (solx lt 0) then delradx=-delradx
  if (soly lt 0) then delrady=-delrady

; shift cube131 back over radial height-difference component
  if (max(abs([delradx,delrady])) gt 0.01) then begin
    avgB=avg(cube131)
    reformcube,cube131,cube131cor,shift=-[delradx,delrady],$
      missingvalue=avgB,splinip=1
  endif 
endif ; end of heightdiff correction

; make product cube
; NB: no normalization by mean as in LAR-1 v1-2; it served to just fill
;     integer range, but the pipeline needs center/target scale equality
aia3013=float(cube304)*cube131cor ; each cube is integer below 32000 
aia3013=fix(aia3013/30000.+0.5)  ; integer, should have reasonable values

; iteratively determine cut and color levels in product units
thiskeep=aia3013
for iter=0,niter-1 do begin
  mom=moment(thiskeep,sdev=sdev)
;;  print,mom[0],sdev,5.*sdev
  keep=thiskeep[where(thiskeep lt mom[0]+cutactive*sdev)]  ; delivers 1D index
  thiskeep=keep ; is now also 1D index, not image
endfor
aia3013cut=mom[0]+cutlimit*sdev
aia3013color=mom[0]+colorlimit*sdev

end

; ======================= IDLWAVE test per Hyper C

; as in LAR-1
cd,'/media/rutten/RRHOME/alldata/SolO/2020-05-30-first/sdo'
cube304=readfits('target/cubes/aia304.fits',headerA)
cube131=readfits('target/cubes/aia131.fits',headerB)
heightdiff=1000.
heightdiff=0
sdo_firelevel,cube304,cube131,aia3013,aia3013cut,aia3013color,$
  heightdiff=heightdiff,header=headerA
aia3013clip=aia3013<aia3013cut
aia3013clip[where(aia3013 gt aia3013color)]=max(aia3013clip)
; not cyan but white

; inspect
showex,histo_opt_rr(aia3013),aia3013clip,cube304,cube131,/blink

end
