; file: sdo_checkcadence.pro
; init: Jul 11 2018  Rob Rutten  Deil back from LMSAL
; last: Jun 22 2020  Rob Rutten  Deil
; note: slow by getting all the header time values
;       speeded up by taking out level1 business

;+
pro sdo_checkcadence,diag=diag,cadence=cadence

;-
 ; PURPOSE:
 ;   check JSOC level2 file lists on regularity of time sampling    
 ;   
 ; DESCRIPTION:
 ;   to check temporal segmentation of long-duration JSOC requests
 ;   
 ; CALL:
 ;   see above
 ;   
 ; OPTIONAL KEYWORD INPUTS:
 ;   diag = SDO diagnostic (AKA channel) selector, e.g., 
 ;     '304', '94', 'cont', 'mag', '1600' (default '304')
 ;   cadence = speficy (desired) cadence, default = take from first files
 ;
 ; OUTPUTS:
 ;   printout and plot on screen  
 ;   
 ; RESTRICTIONS:
 ;   run from SDO pull-over subdirectory target or center 
 ;   
 ; HISTORY:
 ;   Jul 13 2018 RR: start
;-

; defaults for keywords
if (n_elements(diag) eq 0) then diag='304'
if (n_elements(cadence) eq 0) then cadence=0

; set search selector for files (oops, awkward variety in naming habits)
; level1
;   hmi.M_45s.20140614_080900_TAI.2.magnetogram.fits
;   hmi.Ic_45s.20140614_080900_TAI.2.continuum.fits
;   aia.lev1_uv_24s.2014-06-14T080925Z.1700.image.fits  ; idem 1600
;   aia.lev1_euv_12s.2014-06-14T080949Z.304.image.fits  ; idem EUV (e.g., .94)
; level2
;   HMI20140614_070940_mag.fits
;   HMI20140614_070940_cont.fits
;   AIA20140614_080942_1700.fits
;   AIA20140614_040119_0304.fits
;   AIA20140614_080949_0094.fits
if (diag eq 'cont') then begin
  lev1sel='continuum'
  lev2sel='cont.fits'
endif
if (diag eq 'mag') then begin
  lev1sel='magnetogram'
  lev2sel='mag.fits'
endif
if (float(diag) gt 0) then begin
  lev1sel='.'+diag+'.'
  lev2sel=diag+'.fits'
endif 

; uncompress level2 diag files if necessary
print,' ----- uncompress level2/diag files if needed (may take long)'
spawn,'funpack -D  level2/*'+lev2sel+'.fz'

; find level1 and level2 files for this diag
lev1names=file_search('level1/*'+lev1sel+'*.fits',count=nlev1)
if (lev1names[0] eq '') then begin
  print,' ===== warning: no level1 files found'
endif
lev2names=file_search('level2/*'+lev2sel,count=nlev2)
if (lev2names[0] eq '') then begin
  print,' ##### ABORT: no level2 *.fits files found (compressed?)'
  RETURN
endif
if (nlev2 ne nlev1) then print,' ##### WARNING: '+$
  'nlev1 ='+trimd(nlev1)+' BUT nlev2 ='+trimd(nlev2) 
nfiles=nlev2
print, ' ----- '+diag+' file count = '+trimd(nfiles)

; declare timing arrays
namesutc=strarr(nfiles)
namestai=dblarr(nfiles)
lev1tobsutc=strarr(nfiles)
lev1trecutc=strarr(nfiles)
lev2tobsutc=strarr(nfiles)
lev2trecutc=strarr(nfiles)
lev1tobstai=dblarr(nfiles)
lev1trectai=dblarr(nfiles)
lev2tobstai=dblarr(nfiles)
lev2trectai=dblarr(nfiles)

; distill observing times from filenames
for it=0,nfiles-1 do begin
; nastily judicious since path and level and diag dependent 
  ;; if (level eq 'level1') then begin
  ;;   datetime=strmid(lev1names[it],24,17)   ; EUV
  ;;   if (diag eq '1600' or diag eq '1700') then $
  ;;     datetime=strmid(lev1names[it],23,17)   ; UV
  ;;   if  (diag eq 'cont') then $
  ;;     datetime=strmid(lev1names[it],18,15)   ; continuum
  ;;   if  (diag eq 'mag') then $
  ;;     datetime=strmid(lev1names[it],17,15)   ; magnetogram
  ;;   if  (diag eq 'cont' or diag eq 'mag') then begin
  ;;     yr=strmid(datetime,0,4)
  ;;     mo=strmid(datetime,4,2)
  ;;     dd=strmid(datetime,6,2)
  ;;     hr=strmid(datetime,9,2)
  ;;     mi=strmid(datetime,11,2)
  ;;     ss=strmid(datetime,13,2)
  ;;   endif else if (diag eq '1600' or diag eq '1700') then begin
  ;;     yr=strmid(datetime,0,4)
  ;;     mo=strmid(datetime,5,2)
  ;;     dd=strmid(datetime,8,2)
  ;;     hr=strmid(datetime,11,2)
  ;;     mi=strmid(datetime,13,2)
  ;;     ss=strmid(datetime,15,2)
  ;;   endif else if (float(diag) lt 1500) then begin  ; all EUVs 
  ;;     yr=strmid(datetime,0,4)
  ;;     mo=strmid(datetime,5,2)
  ;;     dd=strmid(datetime,8,2)
  ;;     hr=strmid(datetime,11,2)
  ;;     mi=strmid(datetime,13,2)
  ;;     ss=strmid(datetime,15,2)
  ;;   endif
  ;; endif

;;  if (level eq 'level2') then begin $
  datetime=strmid(lev2names[it],10,15)  
  if  (diag eq 'cont' or diag eq 'mag') then begin
    yr=strmid(datetime,0,4)
    mo=strmid(datetime,4,2)
    dd=strmid(datetime,6,2)
    hr=strmid(datetime,9,2)
    mi=strmid(datetime,11,2)
    ss=strmid(datetime,13,2)
  endif else if (diag eq '1600' or diag eq '1700') then begin
    yr=strmid(datetime,0,4)
    mo=strmid(datetime,4,2)
    dd=strmid(datetime,6,2)
    hr=strmid(datetime,9,2)
    mi=strmid(datetime,11,2)
    ss=strmid(datetime,13,2)
  endif else  if (float(diag) lt 1500) then begin  ; all EUVs 
    yr=strmid(datetime,0,4)
    mo=strmid(datetime,4,2)
    dd=strmid(datetime,6,2)
    hr=strmid(datetime,9,2)
    mi=strmid(datetime,11,2)
    ss=strmid(datetime,13,2)
  endif
;  endif

; construct date-times in my standard format
  namesutc[it]=yr+'.'+mo+'.'+dd+'_'+hr+':'+mi+':'+ss

; convert to TAI
  namestai[it]=anytim2tai(namesutc[it])

; now get the header times: t_obs and t_rec
; NB: level1 aren't classical fits files but have a structure header

; level1 only for it=0 to speed up (these commands are slow)
  if (it eq 0 and nlev1 gt 0) then begin
    read_sdo,lev1names[it],index  ; no data means no data
    lev1tobsutc[it]=index.t_obs
    lev1tobstai[it]=anytim2tai(lev1tobsutc[it])
    lev1trecutc[it]=index.t_rec
    lev1trectai[it]=anytim2tai(lev1trecutc[it])
  endif

; level2
  head2=headfits(lev2names[it])
  lev2tobsutc[it]=fxpar(head2,'t_obs')
  lev2tobstai[it]=anytim2tai(lev2tobsutc[it])
  lev2trecutc[it]=fxpar(head2,'t_rec')
  lev2trectai[it]=anytim2tai(lev2trecutc[it])

; show date-times distill for first file
  if (it eq 0) then begin
    print,' ----- '+diag+' distill:  '+datetime+'    '+ namesutc[it]
;;    print,'   level1 t_obs t_rec: '+lev1tobsutc[it]+'  '+lev1trecutc[it]
    print,'   level2 t_obs t_rec: '+lev2tobsutc[it]+'  '+lev2trecutc[it]
  endif else if (fix(it/100.) eq it/100.) then $
    print,'      it ='+trimd(it)+' of total ='+trimd(nfiles)

endfor

; if not specified find cadence as in sdo_makereffitscube.pro
if (cadence eq 0) then begin
  itcad=min([100,nfiles-1])  ; segmentation occurs only for more than 100
  cadence=(lev2tobstai[itcad]-lev2tobstai[0])/float(itcad) ; mean 
endif
print,' ----- cadence (sec) = '+trimd(cadence)

; plot with constant-cadence trend subtraction
timesteps=indgen(nfiles)
;; if (level eq 'level1') then times=lev1tobstai-lev1tobstai[0]
;; if (level eq 'level2') then 
times=lev2tobstai-lev2tobstai[0]
tdif=times-timesteps*cadence
mima=minmax(tdif)
plot,timesteps,tdif,$
  xtitle='timestep',ytitle='time-timestep*cadence  [s]',$
  xrange=[0,nfiles],/xstyle,$
  yrange=1.2*mima+[0,5],/ystyle,psym=-6,symsize=2,linestyle=1,charsize=2.

print,' ===== minmax deviations from cadence ='+trimd(mima)+' seconds'

; plot difference t_rec - t_obs
window,1
plot,timesteps,lev2tobstai-lev2trectai,$
  xtitle='timestep',ytitle='t_obs - t_rec  [s]',/ynozero,$
  psym=-6,linestyle=1,charsize=2,symsize=2

;; ; plot difference t_obs level1 and level2  # no point, identical
;; window,2
;; plot,timesteps,lev1tobstai-lev2tobstai,/ynozero ; none present, always identical

; check on deviators
ndev=0
for it=2,nfiles-1 do begin
;;  if (level eq 'level1') then dt=lev1tobstai[it]-lev1tobstai[it-1]
;;  if (level eq 'level2') then 
  dt=lev2tobstai[it]-lev2tobstai[it-1]

; printout for bad ones with straddle pairs
  if (abs(dt-cadence) gt 1.1 and it lt nfiles-2) then begin
    ndev=ndev+1
    print,'  it = '+trimd(it)+'   dt = '+trimd(dt)+'      files:'
    for showfile=-2,+2 do begin
      ;; print,'  it = '+trimd(it+showfile)+$
      ;;   '  level1: '+lev1names[it+showfile]+' '+lev1tobsutc[it+showfile]
      print,'     '+lev2names[it+showfile]+' '+lev2tobsutc[it+showfile]
    endfor
  endif 
endfor
print,' ##### RESULT: '+trimd(ndev)+' cadence deviators'

end

; ======== run per Hyper-C


;; cd,'/media/rutten/RRDATA/alldata/SDO/2014-06-21-sainzdalda/center'
;; ;;sdo_checkcadence,level='level1'  ; OK!
;; sdo_checkcadence     ; 4 devs: 108, 109, 395, 396 @ 4s  

;; cd,'/media/rutten/RRDATA/alldata/SDO/2014-06-21-sainzdalda/target'
;; ;; sdo_checkcadence,level='level1' ; OK!  (nt=11100)
;; sdo_checkcadence    ; also 4 devs  @ 16s and 8s instead of 12s

;; cd,'/media/rutten/RRDATA/alldata/SDO/2014-06-14-testlong/center'
;; sdo_checkcadence  ; OK only +- 0.01s deviations from 300s
cd,'/home/rutten/data/SDO/2020-05-25-pipetests/testseg/center'
sdo_checkcadence,cadence=120.

end

