; file: sdo_allimages2fitscubes.pro = process SDO cutouts into aligned cubes
; init: Jan  6 2014  Rob Rutten  Deil
; last: Jul  6 2020  Rob Rutten  Deil

;+

pro sdo_allimages2fitscubes,datadir,refwav=refwav,sdoprep=sdoprep,$
  notrack=notrack,diffdesolrot=diffdesolrot,$
  nocrossalign=nocrossalign,driftsdir=driftsdir,$
  clipmin=clipmin,clipmax=clipmax,no_intscale=no_intscale,$
  border=border,mpegs=mpegs,itend=itend,$
  nopack=nopack,moviedir=moviedir,targetmovie=targetmovie,$
  maildone=maildone,verbose=verbose

 ; PURPOSE:
 ;    wrapper to process SDO images into aligned fitscubes and optional mpegs
 ;
 ; DESCRIPTION:
 ;    optionally calls sdo_prep = aia_prep with time-sliding reference,
 ;    then makes the reference cube, than slaves all other wavelengths
 ;
 ; EXAMPLE CALL:
 ;   at the bottom of this program file
 ;
 ; INPUTS:
 ;   datadir = string path/topdir with subdirs level1, level2, cubes
 ;
 ; OPTIONAL KEYWORD INPUTS:
 ;   refwav = string with reference wavelength name, default '171'
 ;     (because the EUV wavs have fastest cadence = 12s, the UV wavs 24s) 
 ;   sdoprep = 1/0: inputs are level1 files, aia_prep them to level2 files
 ;   notrack = 1/0: JSOC did not track rotation in im_patch cutting out
 ;   diffdesolrot = 1/0: differentially derotate target
 ;   nocrossalign = 1/0: do not cross-align using drift splines (0)
 ;   clipmin: lower intensity clipping sdo_intscale.pro
 ;   clipmax: upper intensity clipping sdo_intscale.pro
 ;   no_intscale = 1/0: do not intscale target AIA and HMIcont intensities 
 ;   mpegs = 1/0: add mpg movie files with clock etc
 ;   itend: stop after itend time steps for smaller-volume test;
 ;   nopack = 1/0: do not fpack level2 files when done (default 0 = do fpack)
 ;   moviedir: path for movies, default datadir+'/mpegs'
 ;   driftsdir: path driftsplines, default 'datadir+/../driftscenter'
 ;   border: value to clip borders, default 15
 ;   targetmovie = 1/0: make 4-panel movie (takes long)
 ;   maildone = 'xx@xx.xx': string with email address (default '')
 ;   verbose = 1/0: print progress, display shifts (set to 0 for remote run)
 ;
 ; OUTPUTS:
 ;   integer cubes in datadir/cubes
 ;   optional mpegs in moviedir
 ;   optional email upon completion
 ;
 ; METHOD:
 ;   all programs use assoc per image to handle cubes exceeding memory
 ;
 ; HISTORY:
 ;   Jan 10 2014 RR: start
 ;   Jan 16 2014 RR: aia_prep_tracked_cutouts => sdo_prep_cutouts
 ;   Feb  5 2014 RR: add mpegs, maildone
 ;   Aug 15 2014 RR: clipmin, clipmax
 ;   Sep  2 2015 RR: permit spaces (single) in dir path (Mac...)
 ;   Feb 25 2016 RR: crossalign: not only UV but also EUV > HMI
 ;   Jan 10 2018 RR: no_intscale
 ;   Jul 18 2018 RR: Rice-compress level2 files using fpack
 ;   Apr 18 2020 RR: nocrossalign
 ;   Jul  5 2020 RR: diffdesolrot
;-

; set wall-clock timer (seconds)
timestartimfits=systime(1)

; defaults for keywords
if (n_elements(refwav) eq 0) then refwav='171'
if (n_elements(sdoprep) eq 0) then sdoprep=0
if (n_elements(notrack) eq 0) then notrack=0
if (n_elements(diffdesolrot) eq 0) then diffdesolrot=0
if (n_elements(nocrossalign) eq 0) then nocrossalign=0
if (n_elements(clipmin) eq 0) then clipmin=0
if (n_elements(clipmax) eq 0) then clipmax=0
if (n_elements(no_intscale) eq 0) then no_intscale=0
if (n_elements(mpegs) eq 0) then mpegs=0
if (n_elements(itend) eq 0) then itend=0      ; 0 = do all files
if (n_elements(nopack) eq 0) then nopack=0
if (n_elements(moviedir) eq 0) then moviedir=datadir+'/mpegs/'
if (n_elements(driftsdir) eq 0) then driftsdir=datadir+'/../driftscenter/'
if (n_elements(border) eq 0) then border=15
if (n_elements(targetmovie) eq 0) then targetmovie=0
if (n_elements(maildone) eq 0) then maildone=0
if (n_elements(verbose) eq 0) then verbose=0

; print opener
if (verbose eq 1) then print,' ===== sdo_allimages2fitscubes starts'

; define SDO level2 wavelength identifiers and cubefilenames
wavs=['94','131','171','193','211','304','335','1600','1700',$
      'mag','cont','dop']
fitscubenames=['aia94.fits','aia131.fits','aia171.fits',$
               'aia193.fits','aia211.fits','aia304.fits',$
               'aia335.fits','aia1600.fits','aia1700.fits',$
               'hmimag.fits','hmicont.fits','hmidop.fits']
nwavs=n_elements(wavs)

; define datadir subdirs
level1dir=datadir+'/level1'
level2dir=datadir+'/level2'
cubesdir=datadir+'/cubes' 

; make dirs if not yet existing 
if (sdoprep ne 0) then spawn,'mkdir -p '+level1dir
spawn,'mkdir -p '+level2dir
spawn,'mkdir -p '+cubesdir
if (verbose ne 0) then spawn,'mkdir -p '+cubesdir+'/plots_align/'
if (mpegs ne 0) then spawn,'mkdir -p '+moviedir

; get iref = index of reference (and check it is there)
iref=-1
for iwav=0,nwavs-1 do if (wavs[iwav] eq refwav) then iref=iwav
if (iref eq -1 ) then begin
  print,' ===== ERROR: refwav invalid = ',refwav
  print,' ===== valid: ',wavs
  return
endif

; get icont = index of HMI cont (and check it is there)
icont=-1
for iw=0,nwavs-1 do if (wavs[iw] eq 'cont') then icont=iw
if (icont eq -1 ) then begin
  print,' ##### NB: no hmicont data so no good SDO fits keyword rewriting'
endif
contcube=fitscubenames[icont]

; optional aia_prep all level1 images with time-sliding reference
; (calling it here avoids the need for noticing when it <finally> ends)
; Apr  2 2020 notrack option out since level1 wrong already
if (sdoprep ne 0) then $
  sdo_prep,datadir,refwav=refwav,verbose=verbose

; if sdoprep eq 0 undo compression if fits files are fpacked into .fz 
; (per wav to avoid funpack limit)
if (sdoprep eq 0) then begin
  print,' ===== uncompress all level2 files if necessary (slow)'
  for iwav=0,nwavs-1 do $
    spawn,'funpack -D '+level2dir+'/*'+wavs[iwav]+'.fits.fz'
endif

; make initial reference cube (border-crop, no sawtooth undo or crossalign)
print,' ===== starting on initial ref cube = ',$
  datadir+'/cubes/'+fitscubenames[iref]
sdo_makereffitscube,datadir,border=border,refwav=refwav,$
  clipmin=clipmin,clipmax=clipmax,no_intscale=no_intscale,$
  itend=itend,verbose=verbose

; make initial HMI cont cube just for the keywords (overkill but let be)
sdo_makereffitscube,datadir,border=border,refwav=refcont,$
  clipmin=clipmin,clipmax=clipmax,no_intscale=no_intscale,$
  itend=itend,verbose=verbose

; make slave cubes: JSOC track sawtooth undo, crop, crossalign, diffdesolrot
for iwav=0,nwavs-1 do begin
  if (iwav ne iref) then begin
    fileswav=findfile(level2dir+'/*'+wavs[iwav]+'.fits')
    if (n_elements(fileswav) gt 1) then begin
      print,' ===== starting on slave cube = ',$
        datadir+'/cubes/'+fitscubenames[iwav]
      sdo_makeslavefitscube,datadir,wavs[iwav],refwav=refwav,$
        notrack=notrack,diffdesolrot=diffdesolrot,border=border,$
        nocrossalign=nocrossalign,driftsdir=driftsdir,$
        clipmin=clipmin,clipmax=clipmax,no_intscale=no_intscale,$
        itend=itend,verbose=verbose
    endif
  endif 
endfor

; now redo also refcube for sawtooth undo and optional crossalign
iwav=iref
fileswav=findfile(level2dir+'/*'+wavs[iwav]+'.fits')
if (n_elements(fileswav) gt 1) then begin
  print,' ===== starting on ref cube = ',$
    datadir+'/cubes/'+fitscubenames[iwav]
  sdo_makeslavefitscube,datadir,wavs[iwav],refwav=refwav,$
    notrack=notrack,diffdesolrot=diffdesolrot,border=border,$
    nocrossalign=nocrossalign,driftsdir=driftsdir,$
    clipmin=clipmin,clipmax=clipmax,no_intscale=no_intscale,$
    itend=itend,verbose=verbose
endif

; fpack Rice-compress all level2 fits files with fits deletion
; (per wav to avoid fpack limit)
if (nopack eq 0) then begin
  print,' ----- fpack = Rice-compress all level2 files (slow)'
  for iwav=0,nwavs-1 do spawn,'fpack -D -Y '+level2dir+'/*'+wavs[iwav]+'.fits'
endif

; optional mpegs (with default sharpening)
if (mpegs ne 0) then sdo_allfitscubes2mpegs,datadir+'/cubes/',moviedir,$
  /cutxga,/clock,/datebanner,/sharpen,verbose=verbose

; optional 4-panel movie
if (targetmovie ne 0) then sdo_maketargetmovie,$
  cubesdir=cubesdir,moviedir=moviedir

; print elapsed time
timedone=systime(1)
timelaps=ntostr((timedone-timestartimfits)/60.,format='(F10.1)')
print,' ===== sdo_allimages2fitscubes took '+timelaps+' minutes'

; optional mail confirmation with elapsed time
if (maildone ne 0) then begin
  spawn,'echo "sdo_allimages2fitscubes done in $HOSTNAME in "'$
    +timelaps+'" hours" > ~/tmp/maildone.txt'
  spawn,'mutt -s "RR DONE: sdo_allimages2fitscubes" '+maildone+$
    ' < ~/tmp/maildone.txt'
endif

end

; =============== main for testing per IDLWAVE H-c =========================

datadir='/home/rutten/data/SDO/2014-06-14-small/target'
;; datadir='/home/rutten/data/SDO/2019-11-11-transit/midpoint-notrack'

sdo_allimages2fitscubes,datadir ;; ,/diffdesolrot,/notrack,/nopack
;;   /sdoprep,$
;;   /mpegs,/verbose
;;   maildone='robenrietjerutten@gmail.com' 
;;   itend=5

showex,/allsdo,sdodir=datadir+'/cubes'
;; showex,datadir+'cubes/aia1700.fits',datadir+'cubes/hmimag.fits'

end

