; file: sdo_alignaia_rr.pro
; init: May  7 2016  Rob Rutten  Deil
; last: Mar 14 2022  Rob Rutten  Deil

;+
pro sdo_alignaia_rr,$
  refwav=refwav,sdoprep=sdoprep,preponlytarget=preponlytarget,$
  aligncenter=aligncenter,trimbox=trimbox,nocrossalign=nocrossalign,$
  notrackcenter=notrackcenter,notracktarget=notracktarget,$
  diffdesolrot=diffdesolrot,border=border,centeronly=centeronly,$
  euvanchor=euvanchor,$
  clipmin=clipmin,clipmax=clipmax,no_intscale=no_intscale,$
  addfires=addfires,mpegs=mpegs,targetmovie=targetmovie,$
  name=name,verbose=verbose

 ; NAME:
 ;   sdo_alignaia_rr
 ;   
 ; PURPOSE:
 ;   find SDO offsets for /center, optionally cross-align AIAs to HMI 
 ;   
 ; OPTIONAL KEYWORD INPUTS:
 ;   refwav = string with reference wavelength name, default '171'
 ;     (because the EUV wavs have fastest cadence = 12s, the UV wavs 24s) 
 ;   sdoprep = 1/0 aia_prep level1 > level2 ("level 1.5", default 0)
 ;   preponlytarget = 1/0: aia_prep only dir target, do not redo dir center
 ;     (set this for repeat after abort in sdo_writepairspline.pro)
 ;   aligncenter = 1/0: add crossaligned cubes in center/cubesxal (default 0)
 ;   trimbox = align only over selected center subfield [xmin,ymin,xmax,ymax]
 ;   nocrossalign = 1/0: co-align target cubes (default 0 = do)
 ;   notrackcenter = 1/0: no center rotation tracking
 ;   notracktarget = 1/0: no target rotation tracking
 ;   diffdesolrot = 1/0: differentially de-solar-rotate target
 ;   border: value to clip borders, default 15
 ;   centeronly = 1/0: only center processing (for align tests)
 ;   euvanchor = anchor EUV in reversed order, default ['304','mag']
 ;   clipmin = minimum target intensity clip sdo_intscale.pro, default -1
 ;   clipmax = maximum target intensity clip sdo_intscale.pro, default -2
 ;   no_intscale = 1: do not intscale target AIA and HMIcont intensities
 ;   addfires = 1/0: add 304x131 "fires" fitscubes (see LAR-1), default 0 
 ;   mpegs = 1/0: add mpegs of target/cubes, default 0 
 ;   targetmovie = 1/0 make 4-panel overview movie (default 0)
 ;   name = yourname (default 'Rob Rutten') 
 ;   verbose=1/0: print and plot image shifts etc, default 0
 ;   
 ; OUTPUTS:
 ;   nonaligned cubes in /center/cubes, splined drifts in /driftscenter
 ;   optionally cross-aligned center cubes in /center/cubesxal for checking
 ;   synchronized, cross-aligned SDO fitscubes in /target/cubes
 ;   
 ; EXAMPLE:
 ;   below program end below
 ;
 ; RESTRICTIONS:
 ;   needs SSW and many routines in my rridl
 ;   qualifier _rr means "do this my way": specific dirnames etc
 ;   call from within sdo_getdata_rr.pro or similar
 ;   
 ; HISTORY:
 ;   May  7 2016 RR: start = cut out of sdo_getdata_rr.pro
 ;   Aug 20 2016 RR: /aligncenter
 ;   Jan 10 2018 RR: /no_intscale
 ;   Jun 27 2020 RR: /preponlytarget
 ;   Jul  5 2020 RR: diffdesolrot for target
 ;   Nov 16 2020 RR: addfires
 ;   Mar 14 2022 RR: correction notrack in sdo_maketargetfitscubes
;-

; defaults for keywords (some repeats for standalone application)
if (n_elements(refwav) eq 0) then refwav='171'
if (n_elements(sdoprep) eq 0) then sdoprep=0
if (n_elements(preponlytarget) eq 0) then preponlytarget=0
if (sdoprep eq 1) then preponlytarget=1
if (n_elements(aligncenter) eq 0) then aligncenter=0
if (n_elements(trimbox) eq 0) then trimbox=[-1,-1,-1,-1]
if (n_elements(nocrossalign) eq 0) then nocrossalign=0
if (n_elements(notrackcenter) eq 0) then notrackcenter=0
if (n_elements(notracktarget) eq 0) then notracktarget=0
if (n_elements(diffdesolrot) eq 0) then diffdesolrot=0
if (n_elements(border) eq 0) then border=15
if (n_elements(centeronly) eq 0) then centeronly=0
if (n_elements(euvanchor) eq 0) then euvanchor=['304','mag']
if (n_elements(clipmin) eq 0) then clipmin=-1 ; NB special
if (n_elements(clipmax) eq 0) then clipmax=-2 ; NB special 
if (n_elements(no_intscale) eq 0) then no_intscale=0
if (n_elements(addfires) eq 0) then addfires=0 
if (n_elements(mpegs) eq 0) then mpegs=0
if (n_elements(targetmovie) eq 0) then targetmovie=0
if (n_elements(name) eq 0) then name='Rob Rutten' ;@ please change...
if (n_elements(verbose) eq 0) then verbose=0

; set wall-clock timer (seconds)
timestartalign=systime(1) 

; check notrack vs diffdesolrot
if (notracktarget and diffdesolrot) then begin
  print,' ------ setting notracktarget implies no derot, diffdesolrot set to 0'
  diffdesolrot=0
endif

; set aligncenter when addfires
if (addfires eq 1 and aligncenter eq 0) then begin
  print,' ----- aligncenter set active for addfires'
  aligncenter=1
endif

; get current dir path; \ flag spaces in path if present (bloody Mac?)
; https://groups.google.com/forum/#!topic/comp.lang.idl-pvwave/Lo10H5XtN80
cd,current=thisdir
thisdir=strjoin(strsplit(thisdir,' ',/extract),'\ ') 

; define subdirs made by sdo_getdata_rr.pro
targetdir=thisdir+'/target'
centerdir=thisdir+'/center'
driftsdir=thisdir+'/driftscenter'

; set moviedir
if (mpegs ne 0 or targetmovie ne 0) then begin
  moviedir=targetdir+'/mpegs/'
  spawn,'mkdir -p '+moviedir
endif else moviedir=''

; determine offsets and drifts from /center data
sdo_getdriftsplines,centerdir,refwav=refwav,$
  sdoprep=sdoprep,trimbox=trimbox,notrack=notrackcenter,$
  euvanchor=euvanchor,xalpair=xalpair,verbose=verbose
spawn,'touch '+centerdir+'/cubes/00-NB_cubes_not_aligned'

; if \aligncenter add cross-aligned center/cubesxal for checking shifts
; (no diffdesolrot since fast pairs comparisons per tile)
if (aligncenter ne 0) then begin
  tempxaldir=thisdir+'/tempxal'
  spawn,'mkdir -p '+tempxaldir
  spawn,'rm -f '+tempxaldir+'/*'
  spawn,' ln -s '+centerdir+'/level1 '+tempxaldir+'/level1' 
  spawn,' ln -s '+centerdir+'/level2 '+tempxaldir+'/level2' 
  sdo_allimages2fitscubes,tempxaldir,refwav=refwav,sdoprep=sdoprep,$
    notrack=notrackcenter,diffdesolrot=0,$
    clipmin=clipmin,clipmax=clipmax,no_intscale=0,$ ; NB! clips as for target
    nocrossalign=0,driftsdir=driftsdir,border=border
  spawn,'touch '+tempxaldir+'/cubes/00-NB_cubes_aligned'
  spawn,'rm -f '+tempxaldir+'/level1'
  spawn,'rm -f '+tempxaldir+'/level2'
  spawn,'rm -rf '+centerdir+'/cubesxal'
  spawn,'mv '+tempxaldir+'/cubes '+centerdir+'/cubesxal'
  spawn,'rm -rf '+tempxaldir
  
; add check-xal plot for alignment checking
  print,' ===== sdo_alignaia_rr makes xal driftplot'
  sdo_writepairspline,centerdir,xalpair,/aligncenter,xalpair=xalpair
  spawn,'gv '+driftsdir+'/check'+'*.ps'

; optionally write "fire detector" clip levels (LAR-1 "campfires")
; NB: needs same sdo_intscale for center/cubesxal and target/cubes
  if (addfires eq 1) then begin
    head304=headfits(centerdir+'/cubesxal/aia304.fits')
    nt=fxpar(head304,'naxis3')
    ntcut=min([nt-2,20])
    cutrange=[nt/2-ntcut/2,nt/2+ntcut/2] 
    cubeA=readfitscube(centerdir+'/cubesxal/aia304.fits',trange=cutrange)
    cubeB=readfitscube(centerdir+'/cubesxal/aia131.fits',trange=cutrange)
    sdo_firelevel,cubeA,cubeB,aia3013,aia3013cut,aia3013color,$
      heightdiff=0,header=head304
    
    ; write cut and color thresholds into driftscenter for target application
    firefile=driftsdir+'/fireclips.dat'
    writecol,firefile,aia3013cut,aia3013color,fmt='(2F10.1)'
    
    ; write center fire cubefiles (whyse not, vanish when deleting center)
    sdo_addfires,aia3013cut,aia3013color,cubesdir=centerdir+'/cubesxal'
  endif ; end addfires

endif   ; end aligncenter

; process /target data 
; eg choices: clipmin, clipmax, no_intscale, nocrossalign, diffdesolrot
if (centeronly ne 1) then $
 sdo_maketargetfitscubes,targetdir=targetdir,refwav=refwav,$
  sdoprep=preponlytarget,notrack=notracktarget,$
  diffdesolrot=diffdesolrot,$
  nocrossalign=nocrossalign,driftsdir=driftsdir,border=border,$
  clipmin=clipmin,clipmax=clipmax,no_intscale=no_intscale,$
  addfires=addfires,mpegs=mpegs,targetmovie=targetmovie,moviedir=moviedir,$
  nopack=nopack $
else print,' ##### /centeronly => no target processing'

; back to calling dir
cd,thisdir

; done; print elapsed time
timedone=systime(1)
timelaps=ntostr((timedone-timestartalign)/60.,format='(F11.1)')
print,' ===== sdo_alignaia_rr done; took '+timelaps+' min'

; for RR only: copy movie to corresponding wrk dir (in laptop for talks)
if (targetmovie ne 0 and name eq 'Rob Rutten') then begin
  spawn,'mkdir -p `echo $cwd | sed -e "s|.*data/|/home/rutten/rr/wrk/|g"`/mpegs'
  spawn,'cp '+moviedir+'/4psdotarget.mpg `echo $cwd | sed -e "s|.*data/|/home/rutten/rr/wrk/|g"`/mpegs/.'
  spawn,'mplayer -really-quiet '+moviedir+'/4psdotarget.mpg -loop 0'
endif

end


; =============== main for testing per IDLWAVE H-c ======================

;; cd,'/home/rutten/data/SDO/2019-11-11-transit/midpoint'
;; cd,'/home/rutten/data/SDO/2019-11-11-transit/ingress'
;; cd,'/home/rutten/data/SDO/2019-11-11-transit/egress'

cd,'/home/rutten/data/SDO/2014-06-14-small' ; fire everywhere
;; cd,'/media/rutten/RRHOME/alldata/SST/2014-06-24-solnet/sdo' ; 1 small fire
;; cd,'/media/rutten/RRHOME/alldata/SolO/2020-05-30-first/sdo' ; LAR-1 redo
;; cd,'/media/rutten/RRHOME/alldata/SDO/2019-11-11-solnet/midpoint-nov' ; OK

sdo_alignaia_rr,/addfires
; inspect
;; showex,/allsdo,sdodirs=['center/cubes','center/cubesxal']
showex,/allsdo,sdodirs='target/cubes'

end
