; file: sdo_addebfafs.pro
; init: Dec 31 2020  Rob Rutten  Deil
; last: Jan  2 2021  Rob Rutten  Deil

;+
pro sdo_addebfafs,cubesdir,mode=mode,heightdiff=heightdiff

 ; write aia1617xxx.fits in cubesdir with xxx = rat or mul
 ;
 ; INPUTS:
 ;   cubesdir: having aia1600 and aia1700 files, output goes there
 ;
 ; OPTIONAL KEYWORD INPUTS:
 ;   mode:
 ;      1 = ratio 1600/1700 (default)
 ;      2 = multiplication 1600x1700
 ;      3 = both
 ;   heightdiff = radial offset correction (MCs = 220 km, not for EBs or FAFs)
 ;      NB: classic EBFAF movies in 2014-2017 talks had heightdiff=0
 ; 
 ; OUTPUTS:
 ;   aia1617xxx.fits file(s) in cubesdir
 ;
 ; HISTORY:
 ;   Dec 31 2020 RR: start from sdo_addfires.pro and sdo_makeeblocmovie.pro
;-

; answer wrong-parameter query 
if (n_params(0) ne 1) then begin
  sp,sdo_addebfafs
  return 
endif

; keyword defaults
if (n_elements(mode) eq 0) then mode=1
if (n_elements(heightdiff) eq 0) then heightdiff=0

; get input UV files
file1600=file_search(cubesdir+'/aia1600*.fits')
file1700=file_search(cubesdir+'/aia1700*.fits')

; check presence and uniqueness UV files
if (file1600 eq '' or file1700 eq '') then begin
  print,' ##### sdo_addfires abort: no aia1600*fits or aia1700*fits in dir ' +cubesdir
  return
endif
if (n_elements(file1600) gt 1 or n_elements(file1700) gt 1) then begin
  print,' ##### sdo_addfires abort: multiple aia1600*fits or aia1700*fits in dir '+cubesdir
  return
endif

; get 1600 and 1700 average intensities
avg1600=avg(getmeanfitscube(file1600))
avg1700=avg(getmeanfitscube(file1700))

; ----- input files assoc business 

; set endian
bigendian=1

; get file1600 dimensions and datatype from fits header
headin=headfits_rr(file1600)
headinsize=(1+fix(n_elements(headin)/36.))*2880
nx=fxpar(headin,'naxis1') 
ny=fxpar(headin,'naxis2') 
nt=fxpar(headin,'naxis3') 
bitpix=fxpar(headin,'bitpix')

; check integer 
if (bitpix ne 16) then begin
  print,' ##### sdo_addebfafs abort: aia input fitscubes not integer'
  return
endif

; open file1600 for assoc = inassoc1
get_lun,unit_in1
if (bigendian) then openr,unit_in1,file1600,/swap_if_little_endian $
else openr,unit_in1,file1600
inassoc1=assoc(unit_in1,intarr(nx,ny),headinsize)

; open file1700 for assoc = inassoc2
get_lun,unit_in2
if (bigendian) then openr,unit_in2,file1700,/swap_if_little_endian $
else openr,unit_in2,file1700
inassoc2=assoc(unit_in2,intarr(nx,ny),headinsize)

; goto loop over thismode when mode=4
thismode=mode
if (mode eq 3) then thismode=1

NEXTMODE:

; define outfile
if (thismode eq 1) then outfile=cubesdir+'/aia1617rat.fits'
if (thismode eq 2) then outfile=cubesdir+'/aia1617mul.fits'

; open outfile for assoc = outassoc
get_lun,unit_out
headout=headin
if (thismode eq 1) then sxaddpar,headout,'channel','AIA 1600/1700'
if (thismode eq 2) then sxaddpar,headout,'channel','AIA 1600x1700'
headoutsize=(1+fix(n_elements(headout)/36.))*2880
if (bigendian) then openw,unit_out,outfile,/swap_if_little_endian $
else openw,unit_out,outfile
outassoc=assoc(unit_out,intarr(nx,ny),headoutsize)
rec=assoc(unit_out, bytarr(headoutsize))
rec[0]=byte(headout)

; now the act
for it=0,nt-1 do begin

; get images
  a1600=inassoc1[it]
  a1700=inassoc2[it]

; apply heightdiff correction except for thismode=3 or center sampling
;RR NB: 1600-1700 limbshift may have helped in classic by reducing MC contrast
;RR     whereas EBs & FAFs share locations limbward in 1600 and 1700
  a1600cor=a1600
  if (heightdiff ne 0) then begin
    
; exclude center sampling    
    solx=fxpar(headin,'xcen')
    soly=fxpar(headin,'ycen')
    if (abs(solx) gt 100 or abs(soly) gt 100) then begin
      
; astrometry
      arcsecpx=0.6
      rsun_obs=fxpar(headin,'rsun_obs')
      rsunpx=rsun_obs/arcsecpx
      rsunkm=696340.
      solvec=sqrt(float(solx)^2+soly^2)
      sintheta=solvec/rsun_obs              ; r/R value
      delradkm=heightdiff*sintheta          ; projected height difference
      delradpx=delradkm*(rsunpx/rsunkm)     ; in px units

; position angle of (X,Y) vector from disk center
      gamma=atan(abs(soly)/(abs(solx)>0.01))   ; 0.01 avoids y/0=NaN
      delradx=delradpx*cos(gamma)
      delrady=delradpx*sin(gamma)
      
; correct quadrant signs
      if (solx lt 0) then delradx=-delradx
      if (soly lt 0) then delrady=-delrady

; shift image 1600 back over radial height-difference component
      if (max(abs([delradx,delrady])) gt 0.01) then begin
        reformimage,a1600,a1600cor,$
          shift=-[delradx,delrady],missingvalue=avg1600,splinip=1
      endif
    endif   ; end check on center sampling
  endif     ; end check on thismode 
  
; normalize to get good output number range
  a1600=float(a1600cor>0) 
  a1700=float(a1700>0)
  if (thismode eq 1) then a1700=histo_opt_rr(a1700,1E-4,/bottom_only)+1 
  norm1600=a1600/avg1600
  norm1700=a1700/avg1700
  
; three modes
  if (thismode eq 1) then outim=fix(norm1600/norm1700*500) 
  if (thismode eq 2) then outim=fix(norm1600*norm1700*200) 
  outassoc[it]=outim
  
endfor ; of loop over images

; repeat if mode=4
thismode=thismode+1
if (mode eq 3 and thismode eq 2) then goto,NEXTMODE

end 

; =============== main for testing per IDLWAVE H-c ======================

;; cd,'/media/rutten/RRHOME/alldata/SDO/2011-08-18-bello/sdo'
cd,'/media/rutten/RRHOME/alldata/SST/2017-09-22-peb/sdo'
cubesdir='target/cubes'
;; cd,'/media/rutten/RRHOME/alldata/SDO/2010-11-18-nelson'
;; cubesdir='cubes'
mode=3

sdo_addebfafs,cubesdir,mode=mode,heightdiff=heightdiff

; check
showex,/allsdo,sdodirs=cubesdir,wavindA=4,wavindB=5

end
