; file: solisimages2fitscubes.pro = write cube.fits from loose fits images 
; init: Feb 22 2022  Rob Rutten  Deil = copy gongimages2fitscube.pro
; last: Feb 23 2022  Rob Rutten  Deil

;+
pro solisimages2fitscubes,inpath,filestring,cube1file,cube2file,$
  timefile=timefile,cut1=cut1,cut2=cut2,$
  xmin=xmin,xmax=xmax,ymin=ymin,ymax=ymax,$
  itstart=itstart,itend=itend,verbose=verbose

 ; convert a sequence of SOLIS files to two fitscubes
 ;
 ; INPUTS:
 ;   inpath = string with path to input SOLIS files (end with /)
 ;   filestring = string with selective part of file names
 ;
 ; OPTIONAL KEYWORD INPUTS:
 ;   timefile: string filename, include 'tai' (default in /tmp)
 ;   cut1, cut2: fractions for greying outside, default cut=3, cut2=5
 ;   xmin,xmax,ymin,ymax,itstart,itend: select partial cube
 ;   verbose = 1/0 printout
 ;
 ; OUTPUTS:
 ;   cube1file = string with output path/filename ending with .fits
 ;   cube2file = string with output path/filename ending with .fits
 ;
 ; METHOD:
 ;   assoc so can treat large file lists
 ;
 ; HISTORY:
 ;   Feb 22 2022: start = copy of gongimages2fitscubes.pro
;-

; answer a no-parameter query
if (n_params() lt 4) then begin
  sp,solisimages2fitscubes
  return
endif

; default keyword values
if (n_elements(timefile) eq 0) then timefile='/tmp/solisimagestimestai.dat'
if (n_elements(cut1) eq 0) then cut1=3.
if (n_elements(cut2) eq 0) then cut2=5.
if (n_elements(xmin) eq 0) then xmin=0
if (n_elements(xmax) eq 0) then xmax=0
if (n_elements(ymin) eq 0) then ymin=0
if (n_elements(ymax) eq 0) then ymax=0
if (n_elements(itstart) eq 0) then itstart=0
if (n_elements(itend) eq 0) then itend=0
if (n_elements(verbose) eq 0) then verbose=0

; check timefile on including tai
if (strmatch(timefile,'*tai*') ne 1) then begin
  print,' ##### timefile name should include "tai"'
  return
endif

; remove earlier files
spawn,'rm -f '+cube1file+' '+cube2file+' '+timefile

; uncompress SLIS files when rice-compressed (.fz) with the funny -E 
spawn,'funpack -E 1 '+inpath+'/*'+filestring+'*.fts.fz'
spawn,'rm -f '+inpath+'/*'+filestring+'*.fts.fz'  ; -D not allowed with -E

; get filelist
files=findfile(inpath+'/*'+filestring+'*.fts') 
ntfull=n_elements(files)
if (itend eq 0) then itend=ntfull-1
nt=itend-itstart+1

; get file size 
inheader=headfits_rr(files[0])
nxfull=fxpar(inheader,'naxis1') 
nyfull=fxpar(inheader,'naxis2') 
if (xmax eq 0) then xmax=nxfull-1
if (ymax eq 0) then ymax=nyfull-1
nx=xmax-xmin+1
ny=ymax-ymin+1

; set endian, get data type, define outheader
bigendian=1
bitpix=fxpar(inheader,'bitpix')
mkhdr,outheader,abs(bitpix)/8,[nx,ny,itend-itstart+1]
sizeoutheader=size(outheader)  
; fits header = Nx36 "card images" = Nx2880 bytes
outheadersize=(1+fix(sizeoutheader[1]/36.))*2880

; open cube1file for assoc, write header
get_lun, unit_1out
if (bigendian) then openw,unit_1out,cube1file,/swap_if_little_endian $
else openw,unit_1out,cube1file
if (bitpix eq -32) then out1assoc=assoc(unit_1out,fltarr(nx,ny),outheadersize)
if (bitpix eq 16) then out1assoc=assoc(unit_1out,intarr(nx,ny),outheadersize)
if (bitpix eq 8) then out1assoc=assoc(unit_1out,bytarr(nx,ny),outheadersize)
if (outheadersize ne 0) then begin
  rec=assoc(unit_1out, bytarr(outheadersize))
  rec[0]=byte(outheader)
endif

; open cube2file for assoc, write header
get_lun, unit_2out
if (bigendian) then openw,unit_2out,cube2file,/swap_if_little_endian $
else openw,unit_2out,cube2file
if (bitpix eq -32) then out2assoc=assoc(unit_2out,fltarr(nx,ny),outheadersize)
if (bitpix eq 16) then out2assoc=assoc(unit_2out,intarr(nx,ny),outheadersize)
if (bitpix eq 8) then out2assoc=assoc(unit_2out,bytarr(nx,ny),outheadersize)
if (outheadersize ne 0) then begin
  rec=assoc(unit_2out, bytarr(outheadersize))
  rec[0]=byte(outheader)
endif

; set up timing array
if (timefile ne '') then timarr=dblarr(nt)

; loop over timestep it 
for it=itstart,itend do begin

; verbose comforter
  if (verbose) then print,'==== time step ',ntostr(it)
  
; read next fitsfile  @@@@ two images > two cubes
  twoim=readfits(files[it],head)
  im1=twoim[*,*,0]
  im2=twoim[*,*,1]

; make outside grey (same value for all time steps)
  if (it eq 0) then begin
    meanim1=avg(im1[nx/2-nx/4:nx/2+nx/4,ny/2-ny/4:ny/2+ny/4])
    meanim2=avg(im2[nx/2-nx/4:nx/2+nx/4,ny/2-ny/4:ny/2+ny/4])
    cutim1=meanim1/cut1
    cutim2=meanim2/cut2
  endif
  reformimage,im1,image1,black2grey=[cutim1,meanim1]
  reformimage,im2,image2,black2grey=[cutim2,meanim2]

;; ; check
;;   showex,image1,image2 ; blink: hacw pair NOT co-spatial size difference?
;;   STOP
  
; assoc image1 into cube1file
  out1assoc[it-itstart]=image1[xmin:xmax,ymin:ymax]

 ; assoc image1 into cube2file
  out2assoc[it-itstart]=image2[xmin:xmax,ymin:ymax]

; add time into timing array
  if (timefile ne '') then begin
    datetime=fxpar(head,'date_obs')
    ; SOLIS uses date-obs
    if (datatype(datetime) ne 'STR') then datetime=fxpar(head,'date-obs')
    if (datatype(datetime) ne 'STR') then begin
      print,' ##### no DATE_OBS or DATE-OBS keyword in header'
      return
    endif
    timarr[it-itstart]=anytim2tai(datetime)
  endif

endfor ; end loop over files

; write timefile
writecol,timefile,timarr,fmt='(E21.12)'

; free output files
free_lun,unit_1out
free_lun,unit_2out

print,' ===== solisimages2fitscubes wrote ',cube1file, cube2file

end

; =============== test hyper-C ===============

cd,'/home/rutten/data/ALMA/2015-12-16-rb/solis/hacw'
inpath='.'
selstring='151216'
cube1file='allcore.fits'
cube2file='allsumwing.fits'
timefile='alltimestai.dat'

solisimages2fitscubes,inpath,selstring,cube1file,cube2file,$
  cut1=cut1,cut2=cut2,timefile=timefile,/verbose

; inspect cubefiles
showex,cube1file,cube2file

; show startoff sample of time file
readcol,timefile,times,format='D',/silent
print,anytim2utc(times[0:10],/ccsds) 

end
