; file: nisp_sdo.pro
; init: Feb 27 2022  Rob Rutten  Deil from solis_sdo.pro
; last: Mar 30 2022  Rob Rutten  Deil
; note: UNDER CONSTRUCTION 
; todo: see emwrknispjot

;+
pro nisp_sdo,whichnisp,selstring,rmsmin,$
  nispdir=nispdir,sdodir=sdodir,aiafile=aiafile,outfile=outfile,$
  gongstation=gongstation,itbestnisp=itbestnisp,$
  fixscale=fixscale,muckrandom=muckrandom,$
  averspan=averspan,timedelay=timedelay,$
  notrackcenter=notrackcenter,notracktarget=notracktarget,$
  pxnisp=pxnisp,anglenisp=anglenisp,$
  maxselfmetcalf=maxselfmetcalf,maxcometcalf=maxcometcalf,$
  cutselfmetqual=cutselfmetqual,$
  trimboxcoal=trimboxcoal,heightdiff=heightdiff,inshift12=inshift12,$
  skipselbest=skipselbest,$
  skipshiftselfal=skipshiftselfal,skipmetselfal=skipmetselfal,$
  skipcoal=skipcoal,checkmetcalf=checkmetcalf,$
  checksdocenter=checksdocenter,checksdotarget=checksdotarget,$
  show=show,blink=blink,verbose=verbose

 ; NISP = either GONG Halpha or SOLIS/FDP full-disk Halpha images.

 ; Puts co-aligned NISP target cube in dir <sdodir>/target/cubes.
 ; Uses pattern matching aia304 to greyscale-reversed NISP sampling.
 ; Started for SOLIS/FDP reversed Halpha-summed-wings, merged with GONG.
 ; Background: ALMA endnotes Lingezicht Astrophysics Report 2 on my website.
 ; Difficult.  Follows steps and tricks of former gong_sdo.pro.

 ; Not robust.  However, if just a few full-disk NISP images get well
 ; aligned, probaby needing a sufficiently large sufficiently quiet
 ; field at disk center, these may serve to co-align other data to
 ; NISP and via that to SDO.  I started this effort to put ALMA images
 ; on SDO via GONG Halpha, later SOLIS Halpha wingsum.  GONG is better
 ; re scale.
 ;
 ; It seems that GONG matches 304 well without heightdiff correction,
 ; i.e. that the Halpha core and 304 canopies and limb extension are
 ; about the same (fits with LAR-2 idea of 304 spicule-II bright-tip
 ; emission and sheath 304 emission around return fibrils).  However,
 ; SOLIS hacw summed-wings images clearly have large heightdiff offset
 ; since formed much deeper (starts of spicules-II and ends of return
 ; fibrils).  My disk-tile analysis indicated 3000 km for 304 above
 ; wingsums.  The ad-hoc heightdiff-mucking solution is to use for
 ; SOLIS a very large disk-center trimbox that gives reliable angle
 ; and shift but too large platescale (SOLIS px too large) and then
 ; use fixscale=-2 and heightdiff to put SOLIS hacw wingsums also
 ; limbward on GONG Halpha and 304.  Value heightdiff=3000 did well in
 ; a near-limb trial.  Such a match is fake.  The alternative is to
 ; use a small near-center trimbox giving the actual SOLIS scale but
 ; this likely misfires from being too small hence Metcalving
 ; unreliably because only the large-scale canopy patterns match.  It
 ; will be better to get the NISP px scale from limbfits, Metcalf only
 ; for angle, and use findimshift_tiled. In any case, when using SOLIS
 ; one should check with simultaneous GONG.  Then 304-aligned and
 ; heightdiff-mucked target cutouts must correspond precisely.

 ; Thus, for Halpha-304 there is better pattern match for the wingsums
 ; than for GONG core but the latter has smaller or no heightdiff
 ; problem.  The question is which Halpha ALMA corresponds best with.
 ; If ALMA has significant heightdiff from 304 and GONG as well as
 ; from SOLIS than precise NISP scales must be determined from
 ; limbfits, with Metcalving only for angle and shift.  
 ;
 ; REQUIRED INPUT
 ; <nispdir>
 ;   for SOLIS: SOLIS Halpha_cw images downloaded with this recipe
 ;   into nispdir (default solis/hacw):
 ;   NB: in strings 3 times f02 defines level 2 core/wingsum = "hacw"
 ;     > lftp https://solis.nso.edu/pubkeep/f02/201512/k4f02151216/
 ;         # replace by your date               YYYYMM      YYMMDD/
 ;     > mget k4f02151216t18*.fts.fz   # get all images hour 18:++
 ;         #       YYMMDD HH
 ;     > exit
 ;  
 ;   for GONG: GONG Halpha images downloaded with this recipe
 ;   into nispdir (default gong): 
 ;     > lftp ftp://gong2.nso.edu/HA/haf/201804/20180412  
 ;         # replace by your date        YYMMDD YYYYMMDD
 ;     > mget 20180412154*  # get all images 15:40-15:49
 ;     > mget 20180412155*  # get all images 15:50-15:59, etcetera
 ;     > exit
 ;   which I keep out of this program to avoid slow download repeats
 ;
 ; <sdodir> with SDO pipeline results in driftscenter/ and target/
 ;   get these in <sdodir> (default sdo) with e.g.:
 ;   > sdo_getdata_rr,'2018.04.12_15:40',30,-293,-298,xsize=175,ysize=175,$
 ;       /aligncenter,youridents (/notrackcenter, /notracktraget as above)
 ;
 ; USAGE:
 ;   run first with rmsmin=0 for on-screen plot NISP rms > threshold
 ;   run then with trimboxcoal absent or -1 to get full-disk blink 304-NISP
 ;     (304 is gotten and shifted over its driftscenter cross-align value) 
 ;     select large quietest area (no activity, no filaments) in NISP
 ;     GONG: best around disk center; SOLIS: best near target 
 ;   tmode 1: uses single full-disk images (best rms NISP, corresponding 304)
 ;   tmode 2: use time-segment-averaged full-disk sequences
 ;
 ; INPUTS:
 ;   whichnisp = 'GONG' or 'SOLIS'
 ;   selstring: string selector SOLIS/GONG filename part, eg date 
 ;   rmsmin: rms value above which to pass, choose from graph when rmsmin=0
 ;
 ; OPTIONAL KEYWORD INPUTS (def = default):
 ;   nispdir (default 'gong' or 'solis/hacw')
 ;   sdodir  (default 'sdo/')
 ;   aiafile: center/cubesxal to use = aia304.fits (default), aia3013.fits
 ;   outfile: filename center/cubesxal, target/cubes
 ;     default 'gong.fits', 'solis.fits', adapt for eg diffheight
 ;   gongstation = use only 1 of 'B','C','L','M','T','U', def 'all'
 ;   itbestnisp = fixed NISP it selection, def -1 = find best rms
 ;   fixscale = 0: default, Metcalf => scale+angle+shift (scale dangerous)
 ;            = value: use given pxnisp
 ;            = -1: Metcalf but then replace found pxnisp with nominal value
 ;            = -2: apply px_nisp rescale set by heightdiff
 ;            = -3: use nominal, Metcalf only for angle, use findimshift_tiled
 ;            = -4: run limbfit.pro to measure pxnisp and use that
 ;      NB: Mar 23 2022: options -3, -4 not <yet> implemented
 ;   muckrandom = 1/0: randomize brightest/darkest features in align pair
 ;   averspan: 
 ;     default 0 = tmode 1: use single full-disk images for coalign
 ;     > 0 = tmode 2: use mean full-disk images over averspan minutes
 ;         around itbestnisp, with derotation center trimboxcoal
 ;   timedelay: sample AIA 304 timedelay minutes before NISP
 ;     works in both tmode-1 and tmode-2 but
 ;     NB: solar rotation shift in tracked tmode 1
 ;         differential rotation away from trimboxcoal center in tmode 2
 ;   notrackcenter, notracktarget: as in call sdo_getdata_rr (def 0)
 ;   pxnisp: improved value for Metcalf repeat, default from header 
 ;   anglenisp: improved value for Metcalf repeat, default 0
 ;   trimboxcoal: selected part Metcalf images for coalign,
 ;     -1 (default) stop after selecting best NISP
 ;         in showex-blink selected NISP to AIA 304 on NISP scales
 ;         set trimboxcoal by clicking on lowerleft-upperright corners
 ;         selecting a large truly quiet area (no activity, no filaments).
 ;         Enter the printed corner values in trimboxcoal and run again.
 ;     <-2 use central trimbox with sides abs(trimboxcoal) arcsec 
 ;   heightdiff: mean canopy formation height 304 above this Halpha (> 0)  
 ;   inshift12: use value from earlier coalign run to improve Metcalf
 ;   maxselfmetcalf: self-align max number Metcalf iterations (def 1)
 ;   maxcometcalf:  self-align max number Metcalf iterations (def 4)
 ;   cutselfmetqual: don't self-align images with higher metqual (def 1E-2)
 ;   skipselbest = 1/0: skip selection better NISP images 
 ;   skipshiftselfal = 1/0: skip shift selfalign (eg done previously) 
 ;   skipmetselfal = 1/0: skip Metcalf selfalign (only multistation GONG)
 ;   skipcoal = 1/0: skip co-alignment if already done (in temp files)
 ;   checkmetcalf = 1/0: showex Metcalf result trimboxcoal best pair
 ;   checksdocenter = 1/0: showex sdodir/center/cubesxal
 ;   checksdotarget = 1/0: showex sdodir/target/cubes
 ;RR ('check" means showex-blink but parameter starts show and blink taken)
 ;   show: showex results (def 0)
 ;     show=0: no intermediate showex so runs left alone (overnight?)
 ;     show=1: showex self-align and co-align results (quit to continue)
 ;     show=2: add two coalign findalignimage result blinks (quit to continue)
 ;   blink: duration in s for NISP-SDO Metcalf blinkers (def 0) better use!
 ;   verbose=1/0: more or less output (def 0)
 ;
 ; OUTPUTS:
 ;   SDO-fitting NISP-Ha fitscube files:
 ;     <sdodir>/center/cubesxal/outfile
 ;     <sdodir>/target/cubes/outfile
 ;   alignment stuff in <nispdir>:
 ;     graph rms values, threshold and misfits:  nisp_rms_selection.ps
 ;     graph NISP self-align tileshifts: nisp_tileshifts.ps
 ;   temporary files in <nispdir>/temp needed for sequential skipping 
 ;
 ; WARNINGS: Not robust, no good for blind black-box usage.  Better
 ; inspect the co-align Metcalf blinks because the iteration may
 ; derail terribly.  Then select better quiet trimboxcoal in the
 ; Halpha-304 blink (for -1).  Then co-align metqual results in
 ; nisp_sdo_align_metqual.ps and the final center-field blink and the
 ; target field blink may show how well it fared.  Perhaps play with
 ; the flatten and smear parameters in the align calls within the
 ; program, and/or adjust cutmetcalf cutoffs and heightdiff.  I found
 ; double Metcalf iteration (first self-align, then co-align)
 ; necessary for multi-station GONG but slow.  Skipped for SOLIS and
 ; single-station GONG.
 ; 
 ; HISTORY:
 ;   Feb 27 2022 RR: merge solis_sdo.gong and gong_sdo.pro
;-

; answer no-parameter query 
if (n_params(0) ne 3) then begin
  sp,nisp_sdo
  return    
endif

; set wall-clock timer (seconds)
timestartnispsdo=systime(1) 

; check and set nisp
if (whichnisp ne 'GONG' and whichnisp ne 'SOLIS') then begin
  print," ##### whichnisp either 'GONG' or 'SOLIS'; abort"
  return
endif
if (whichnisp eq 'GONG') then nisp=1 else nisp=2 

; defaults for keywords
if (n_elements(nispdir) eq 0) then begin
  if (nisp eq 1) then nispdir='gong'
  if (nisp eq 2) then nispdir='solis/hacw'
endif
if (n_elements(sdodir) eq 0) then sdodir='sdo/'
if (n_elements(aiafile) eq 0) then aiafile='aia304.fits'
if (n_elements(outfile) eq 0) then begin
  if (nisp eq 1) then outfile='gong.fits'
  if (nisp eq 2) then outfile='solis.fits'
endif
if (n_elements(gongstation) eq 0) then gongstation='all' 
if (n_elements(itbestnisp) eq 0) then itbestnisp=-1
if (n_elements(fixscale) eq 0) then fixscale=0
if (n_elements(muckrandom) eq 0) then muckrandom=0
if (n_elements(averspan) eq 0) then averspan=0
if (n_elements(timedelay) eq 0) then timedelay=0
if (n_elements(notrackcenter) eq 0) then notrackcenter=0
if (n_elements(notracktarget) eq 0) then notracktarget=0
if (n_elements(pxnisp) eq 0) then pxnisp=0
if (n_elements(anglenisp) eq 0) then anglenisp=0
if (n_elements(maxselfmetcalf) eq 0) then maxselfmetcalf=1    ;NB default
if (n_elements(maxcometcalf) eq 0) then maxcometcalf=4        ;NB default
if (n_elements(cutselfmetqual) eq 0) then cutselfmetqual=1E-2 ;NB default
if (n_elements(trimboxcoal) eq 0) then trimboxcoal=-1
if (n_elements(heightdiff) eq 0) then heightdiff=0  
if (n_elements(inshift12) eq 0) then inshift12=[0,0]
if (n_elements(skipselbest) eq 0) then skipselbest=0
if (n_elements(skipshiftselfal) eq 0) then skipshiftselfal=0
if (n_elements(skipmetselfal) eq 0) then skipmetselfal=0
if (n_elements(skipcoal) eq 0) then skipcoal=0
if (n_elements(checkmetcalf) eq 0) then checkmetcalf=0
if (n_elements(checksdocenter) eq 0) then checksdocenter=0
if (n_elements(checksdotarget) eq 0) then checksdotarget=0
if (n_elements(show) eq 0) then show=0
if (n_elements(blink) eq 0) then blink=0
if (n_elements(verbose) eq 0) then verbose=0

; error checks
if (nisp eq 1 and nispdir ne 'gong') then $
  print," ##### warning: nispdir # 'gong'; !!! maybe restart IDL" 
if (nisp eq 2 and nispdir ne 'solis/hacw') then $
  print," ##### warning: nispdir # 'solis/hacw'; !!! maybe restart IDL" 

; ========= preamble: initial setups

; tmode
if (averspan lt 1) then tmode=1 else tmode=2
if (trimboxcoal[0] eq -1) then tmode=1  ; get the showex blink

; result fitscubes
nispcenterfile=sdodir+'/center/cubesxal/'+outfile
nisptargetfile=sdodir+'/target/cubes/'+outfile

; temporary files (to enable partial redo skips; rm -rf temp when all done)
spawn,'mkdir -p '+nispdir+'/temp'
spawn,'mkdir -p '+nispdir+'/temp/images304'
nispselectfile=nispdir+'/temp/nisp_select.fits'
nispseltaifile=nispdir+'/temp/nisp_selnisptai.dat'
nispfullfile=nispdir+'/temp/nisp_fulldisk.fits'
corefile=nispdir+'/temp/nisp_allcore.fits'
allnispfile=nispdir+'/temp/nisp_all.fits'
alltaifile=nispdir+'/temp/nisp_alldisktai.dat'
nispitselbestfile=nispdir+'/temp/nisp_itselbest.dat'
norotselfshiftfile=nispdir+'/temp/nisp_norotselfshift.fits'
rotsalfile=nispdir+'/temp/nisp_rotsal.fits'
norotsalfile=nispdir+'/temp/nisp_norotsal.fits'
coalignfile=nispdir+'/temp/nisp_coalign.dat'
nispaiacenterfile=nispdir+'/temp/nisp_aia_center.fits' 
nispaiatargetfile=nispdir+'/temp/nisp_aia_target.fits' 
nisptarcutfile=nispdir+'/temp/nisp_aia_cut.fits' 
nisptarselffile=nispdir+'/temp/nisp_aia_self.fits' 
aiabestim=nispdir+'/temp/aiabest.fits'
nispbestim=nispdir+'/temp/nispbest.fits'

; AIA 304 center parameters
px_aia=0.6 ; arcsec after aia_prep
if (file_test(sdodir+'/center/cubesxal') eq 1) then begin
  sdofile=sdodir+'/center/cubesxal/'+aiafile
  headcen=headfits(sdofile)
  nx_cen=fxpar(headcen,'naxis1')
  ny_cen=fxpar(headcen,'naxis2')
  nt_cen=fxpar(headcen,'naxis3')
  start_cen=fxpar(headcen,'starttim')
  cadence_cen=fxpar(headcen,'cadence')
  aiacentai=anytim2tai(start_cen)+indgen(nt_cen)*cadence_cen
endif else begin
  print,' ##### nisp_sdo abort: no sdodir/center/cubesxal/'+aiafile
  return
endelse

; AIA 304 target parameters
sdotargetfile=sdodir+'/target/cubes/aia304.fits'
if (file_test(sdotargetfile) ne 1) then begin
  print," ##### abort: no file sdodir+'/target/cubes/aia304.fits'"
  return
endif
headtar=headfits(sdotargetfile)
datetimestart=fxpar(headtar,'starttim') ; time first SDO image mid-exposure
taistart=anytim2tai(datetimestart)
nx_tar=fxpar(headtar,'naxis1')
ny_tar=fxpar(headtar,'naxis2')
nt_tar=fxpar(headtar,'naxis3')
xcen_tar=fxpar(headtar,'xcen')
ycen_tar=fxpar(headtar,'ycen')
cadence_tar=fxpar(headtar,'cadence')
aiatargettai=taistart+indgen(nt_tar)*cadence_tar

; NISP start: collect into a temp file unless present, write timings file 
if (nisp eq 1) then begin
  print,' ----- start on gongimages2fitscubes'
  if not(file_test(allnispfile) eq 1) then $
    gongimages2fitscube,nispdir,selstring,allnispfile,timefile=alltaifile
  readcol,alltaifile,allnisptai,format='D'
endif
if (nisp eq 2) then begin
  print,' ----- start on solisimages2fitscubes'
  if not(file_test(allnispfile) eq 1) then $
    solisimages2fitscubes,nispdir,selstring,corefile,allnispfile,$
    timefile=alltaifile
  readcol,alltaifile,allnisptai,format='D'
endif

; judicious flatten and smear for NISP selfaligns in NISP px
flattennisp=10
smearnisp=5

; dimensions NISP full-disk file
allhead=headfits(allnispfile)
nx_nisp=fxpar(allhead,'NAXIS1') 
ny_nisp=fxpar(allhead,'NAXIS2') 
nt_nisp=fxpar(allhead,'NAXIS3')
if (nx_nisp ne 2048 or ny_nisp ne 2048) then begin
  print,' ##### nisp_sdo.pro OOPS: nx_nisp or ny_nisp not 2048'
  return
endif
if (nt_nisp ne n_elements(allnisptai)) then begin
  print,' ##### nisp_sdo.pro OOPS: nt_nisp does not correspond to timefile'
  return
endif

; get list selstring-defined NISP files (SOLIS .fts instead of .fits)
if (nisp eq 1) then begin
;;;  if (gongstation eq 'all') then $
  filelist=file_search(nispdir+'/*'+selstring+'*h.fits') ;;; else $
;;; filelist=file_search(nispdir+'/*'+selstring+'*'+gongstation+'h.fits')
endif
if (nisp eq 2) then filelist=file_search(nispdir+'/*'+selstring+'*.fts')
if (filelist[0] eq '') then begin
  print,' ##### nisp_sdo.pro abort: no NISP files found'
  return 
endif else print,' ----- nisp_sdo.pro retains '+trimd(n_elements(filelist))

; get or set of find pxnisprad and pxnispnominal
nisphead=headfits(filelist[0])
nispsolrad=fxpar(nisphead,'SOLAR-R') ; arcsec; no underscores is NISP way
if (nispsolrad lt 1) then nispsolrad=nispsolrad*60*60./2. ; earlier diam deg
if (nisp eq 1) then begin
; get default pxnisp per radius from first NISP file header
; to be rescaled by matching best NISP to AIA center 304
  pxnisprad=fxpar(nisphead,'RADIUS')  ; only GONG, plm 928
  pxnispnominal=nispsolrad/pxnisprad  ;RR data 2012-08-15-quiet: 1.05297
endif
if (nisp eq 2) then begin
  pxnisprad=928       ; ad-hoc
  pxnispnominal=1.055 ; ad-hoc
endif
if (pxnisp eq 0 or fixscale eq -1) then pxnisp=pxnispnominal

; get nisptrimcenter = inside border-cropped AIA center field size in pxnisp
boxsize=500 ; in arcsec    (AIA pipeline center = 750x750 arcsec)
boxpx=fix(boxsize/pxnisp)
nisptrimcenter=[nx_nisp/2-boxpx/2,ny_nisp/2-boxpx/2,$
                nx_nisp/2+boxpx/2,ny_nisp/2+boxpx/2]

; ========== NISP quality weeding

; warning when NISP exceeds SDO

; limit NISP time range to overlap with AIA center time range
; (because value driftscenter needed for AIA full-disk 304 at ha_best time)
dummy=min((min(aiacentai)-allnisptai)>0,itnispfirst)
dummy=min((max(aiacentai)-allnisptai)>0,itnisplast)
nt_overlap=itnisplast-itnispfirst+1
nisptaioverlap=allnisptai[itnispfirst:itnisplast]

; skip NISP weeding if done before
if (skipselbest eq 1) then begin
  print,' ===== skip selection best NISPs'
  goto,SKIPSELBEST
endif

; get NISP center cutout cube into memory 
allits=indgen(nt_nisp)
itstation=strarr(nt_nisp)
for it=itnispfirst,itnisplast do begin
  im=readfitscube(allnispfile,trange=[it,it])
  if (nisp eq 1) then begin
; B: Big Bear; C: CTIO; L: Learmonth; M: Mauna Loa, T: El Teide; U: Udaipur.
    if (strmatch(filelist[it],'*B*')) then itstation[it]='B'
    if (strmatch(filelist[it],'*C*')) then itstation[it]='C'
    if (strmatch(filelist[it],'*L*')) then itstation[it]='L'
    if (strmatch(filelist[it],'*M*')) then itstation[it]='M'
    if (strmatch(filelist[it],'*T*')) then itstation[it]='T'
    if (strmatch(filelist[it],'*U*')) then itstation[it]='U'
  endif else itstation='X'
  cutim=im[nisptrimcenter[0]:nisptrimcenter[2],$
           nisptrimcenter[1]:nisptrimcenter[3]]
  if (it eq itnispfirst) then begin
    szcut=size(cutim)
    nxcut=szcut[1]
    nycut=szcut[2]
    allcut=fltarr(nxcut,nycut,nt_overlap)
  endif
  allcut[*,*,it-itnispfirst]=cutim
endfor

; set missing_nisp for below
missing_nisp=avg(allcut[*,*,0])

; get rms
;? split normalization between stations? 
;? switch to FFT power in some spatial frequency band as diagnostic?
szcut=size(allcut)
nt_cut=szcut[3]
rms=fltarr(nt_cut)
if (nisp eq 1) then begin
  stations=['B','C','L','M','T','U']
  nstat=6
  for istat=0,nstat-1 do begin
    statits=where(itstation eq stations[istat],statcount)
    if (statcount gt 0) then begin  
      statcube=allcut[*,*,statits]
      mimacube=minmax(histo_opt_rr(statcube,1E-2))
      normcube=(float(statcube)-mimacube[0])/(mimacube[1]-mimacube[0])
      normcube=normcube/avg(normcube)
      reformcube,normcube,muckcube,flatten=flattennisp,smear=smearnisp,$
        /splinip
    ;; ; check for flatten and smear experiments
    ;; showex,normcube,muckcube
    ;; STOP 
      muckcube=fix(muckcube*5000)    ; I like integers
      for it=0,statcount-1 do begin
        mom=moment(muckcube[*,*,it],sdev=sdev)
        rms[statits[it]]=sdev
; set rms to zero if single GONG station specified and not this one
; so that setting rmsmin value takes then out
        if (gongstation ne 'all' and stations[istat] ne gongstation) then $
          rms[statits[it]]=0.
      endfor
    endif
  endfor
endif

if (nisp eq 2) then begin
  mimacube=minmax(histo_opt_rr(allcut,1E-2))
  normcube=(float(allcut)-mimacube[0])/(mimacube[1]-mimacube[0])
  normcube=normcube/avg(normcube)
  reformcube,normcube,muckcube,flatten=flattennisp,smear=smearnisp,/splinip
  ;; ; check for flatten and smear experiments
  ;; showex,normcube,muckcube
  ;; STOP 
  muckcube=fix(muckcube*5000)  ; I like integers
  for it=0,nt_cut-1 do begin
    mom=moment(muckcube[*,*,it],sdev=sdev)
    rms[it]=sdev
  endfor
endif

; get best-rms it unless itbestnisp specified
if (itbestnisp eq -1) then dummy=max(rms,itnispbest) $
else itnispbest=itbestnisp

; rmsmin=0: plot rms on screen for selecting lower threshold rmsmin
if (rmsmin eq 0) then begin
  mima=minmax(rms)
  rmsrange=mima[1]-mima[0]
  offset=rmsrange*0.07
  yrange=[mima[0]-0.2*rmsrange,mima[1]+0.1*rmsrange]
  timesmin=(nisptaioverlap-nisptaioverlap[0])/60.
  xrange=[min(timesmin)-2,max(timesmin)+2]
  its=indgen(nt_nisp)
  plot,timesmin,rms,psym=6,symsize=2,charsize=2,$
    xrange=xrange,xstyle=1,yrange=yrange,ystyle=1,$
    xtitle='time from start overlap [min]',ytitle='rms around mucked average'
  xyouts,timesmin,rms-1.5*offset,charsize=1.2,alignment=0.5,trim(its)
  if (nisp eq 1) then xyouts,timesmin,rms-offset,$
    charsize=1.5,alignment=0.5,itstation
  print,' ##### check rms along (mucked) images in showex blinker'
  print,' ##### select rmsmin value from plot and run again'
  showex,allcut
  retall
endif

; regain allcut memory
undefine,allcut

; get NISP selections passing rms > rmsmin threshold
pass=where(rms gt rmsmin,nsel)
if (nsel eq 0) then begin
  print,' ##### nisp_sdo.pro OOPS: no values left above rmsmin'
  return
endif else print,' ----- rmsmin leaves'+trimd(nsel)+' images'
selnisptai=allnisptai[pass]
selhaits=allits[pass]
selrms=rms[pass]
reformcubefile,allnispfile,nispselectfile,tselect=pass 
selstations=itstation[pass] 

; get selection it best NISP Ha if not itbestnisp specified
if (itbestnisp eq -1) then begin
  dummy=max(selrms,itselbest)
  print,' ----- itnispbest ='+trimd(itnispbest)+$
    '   itselbest ='+trimd(itselbest)
endif else begin
  itselbest=where(selnisptai eq allnisptai[itnispbest])
  if (itselbest[0] eq -1) then begin
    print,' ##### nisp_sdo abort: itbestnisp not in rms selection'
    return
  endif
endelse

; writecol (arrays overdoing it) for future skips
writecol,nispitselbestfile,itselbest,fix(missing_nisp),fmt='(2I10)'

; writecol selstations and selnisptai for future skips
writecol,nispseltaifile,selstations,selnisptai,fmt='(A1,D20.2)'  ; array

; ------ plot rms with threshold and selected its as postscript for keeping
psfilename=nispdir+'/nisp_rms_selection.ps'
sznt=50/nt_nisp ; symbol and character size
axrat=1.62      ; golden ratio
openpsplot,psfilename,thick=2,fontsize=4,xsize=8.8,ysize=8.8/axrat
mima=minmax(rms)
rmsrange=mima[1]-mima[0]
offset=rmsrange*0.08
yrange=[mima[0]-0.2*rmsrange,mima[1]+0.1*rmsrange]
allminutes=(allnisptai-allnisptai[0])/60.
selminutes=(selnisptai-allnisptai[0])/60.
xrange=[min(allminutes)-2,max(allminutes)+2]
allits=indgen(nt_nisp)
selits=indgen(nsel)
plot,allminutes,rms,psym=6,symsize=1.5*sznt,charsize=2*sznt,$
  xrange=xrange,xstyle=1,yrange=yrange,ystyle=1,$
  xtitle='time from start NISPs [min]',ytitle='rms around mucked average'

; add station indentifier for GONG
if (nisp eq 1) then begin
  xyouts,allminutes,rms-offset,charsize=1.2*sznt,alignment=0.5,itstation
  xyouts,allminutes,rms-1.7*offset,charsize=1.2*sznt,alignment=0.5,trim(allits)
endif

; add all-NISP it number undereath 
xyouts,allminutes,rms-1.7*offset,charsize=1.2*sznt,alignment=0.5,trim(allits)

; add selection it number above
xyouts,selminutes,selrms+0.4*offset,charsize=1.2*sznt,$
  alignment=0.5,trim(selits)

; overplot the rms cutoff line
plots,[xrange[0],xrange[1]],[rmsmin,rmsmin],linestyle=3

; overplot SDO center range overlap markers when within NISP time range
overminutes=(allnisptai[itnispfirst:itnisplast]-allnisptai[0])/60.
if (overminutes[0] gt allminutes[0]) then $
  plots,[overminutes[0],overminutes[0]],[yrange[0],yrange[1]],linestyle=1
if (overminutes[nt_overlap-1] lt allminutes[nt_nisp-1]) then $
  plots,[overminutes[nt_overlap-1],overminutes[nt_overlap-1]],$
  [yrange[0],yrange[1]],linestyle=1

; close ps file
closepsplot,psfilename,opengv=0

SKIPSELBEST:   ; ========================================= SKIP

; read selstations and selnisptai and it values when skipping selbest
if (skipselbest eq 1) then begin
  readcol,nispseltaifile,selstations,selnisptai,format='A,D'
  readcol,nispitselbestfile,itselbest,missing_nisp
  itselbest=fix(itselbest[0])
  nsel=n_elements(selnisptai)
  missing_nisp=missing_nisp[0]
endif

; get SSW disk-center rotation away from taistart (maybe use after skip)
rotarcsec=fltarr(2,nsel)
for it=0,nsel-1 do $
  rotarcsec[*,it]=rot_xy(0,0,tstart=taistart,tend=selnisptai[it])
shiftrotcen=transpose(rotarcsec)/pxnisp

; SKIP shiftselfalign part (likely when done already)
if (skipshiftselfal eq 1) then begin
  print,' ===== skip shift self-align'
  goto,SKIPSHIFTSELFALIGN
endif

; ----- first wash = tileshift self-align for nisptrimcenter = center-disk 

print,' ----- shift-only self-align per findimshift_tiled'

; get shifts averaged over tiles applying diffderot per tile per timediff
; so input must be rotating-scene original file
; this puts all on imbest, needs derotation at end to taistart
selffour=fltarr(nsel,4) ; shiftx,shifty,confx,confy
selfshift=fltarr(nsel,2) 
imbest=readfitscube(nispselectfile,trange=[itselbest,itselbest])
reformimage,imbest,imbestmuck,trimbox=nisptrimcenter,$
  flatten=flattennisp,smear=smearnisp
taibest=selnisptai[itselbest]
for it=0,nsel-1 do begin
  imit=readfitscube(nispselectfile,trange=[it,it])
  reformimage,imit,imitmuck,trimbox=nisptrimcenter,$
    flatten=flattennisp,smear=smearnisp
  taiit=selnisptai[it]
  selffour[it,*]=$  ;RR four because [gives shiftx,shifty,cofx,confy]
    findimshift_tiled(imbestmuck,imitmuck,$
                      outdir='/tmp',timediff=taibest-taiit,heightdiff=0,$
                      arcsecpx=pxnisp,$
                      pxtile=75,tilecut=2,precision=0.01,$
                      writetileshifts=0,$
                      wavpair=['itselbest','selit'],datetime=datetimestart,$
                      shiftchart=0,blink=0,verbose=0)
  selfshift[it,0:1]=selffour[it,0:1]
  print,' ----- it ='+trimd(it)+'/'+trim(nsel-1)+': '+$
    ' selfshiftx,y ='+trimd(selfshift[it,0])+', '+trimd(selfshift[it,1])
endfor

; make plot of tileshifts
psfilename=nispdir+'/nisp_tileshifts.ps'
sznt=50/nt_nisp ; symbol and character size
axrat=1.62      ; golden ratio
openpsplot,psfilename,thick=2,fontsize=4,xsize=8.8,ysize=8.8/axrat
mima=minmax(selfshift)
shiftrange=mima[1]-mima[0]
offset=shiftrange*0.08
yrange=[mima[0]-0.2*shiftrange,mima[1]+0.1*shiftrange]
selminutes=(selnisptai-allnisptai[0])/60.
xrange=[min(selminutes)-2,max(selminutes)+2]
selits=indgen(nsel)
plot,selminutes,selfshift[*,0],psym=6,symsize=1.5*sznt,charsize=2*sznt,$
  xrange=xrange,xstyle=1,yrange=yrange,ystyle=1,$
  xtitle='time from start  [min]',ytitle='tiled shifts  [x=O,  y=*]'
oplot,selminutes,selfshift[*,1],psym=2,symsize=1.5*sznt
xyouts,selminutes,selfshift+0.4*offset,charsize=1.2*sznt,$
  alignment=0.5,trim(selits)
xyouts,selminutes[itselbest],selfshift[itselbest,0]+0.8*offset,$
  charsize=2*sznt,alignment=0.5,'best'
if (nisp eq 1) then xyouts,selminutes,selfshift-offset,$
  charsize=1.2*sznt,alignment=0.5,selstations
closepsplot,psfilename,opengv=0

; apply above-found tile shifts together with disk-center derotation
sumshift=selfshift-shiftrotcen
reformcubefile,nispselectfile,norotselfshiftfile,shift=sumshift,/splinip
;; ; check
;; showex,norotselfshiftfile,nispselectfile
;; STOP

SKIPSHIFTSELFALIGN: ; ============================================== SKIP

if (not(file_test(norotselfshiftfile))) then begin
  print,' ##### nisp_sdo.pro abort: no file temp/nisp_norotselfshift.fits'
  return
endif

; Metcalf selfalign for multiple stations GONG sampling
spawn,'cp '+norotselfshiftfile+' '+norotsalfile ; for no Metcalf selfalign
if (nisp eq 1 and skipmetselfal eq 0) then begin
  stationbest=selstations[itselbest]
  statselect=where(selstations ne stationbest)
  if (statselect[0] ne -1) then begin
    if (verbose) then $
      print,' ----- nisp_sdo.pro starts SLOW other-GONG-station Metcalf align'
    selfalignfitscube,norotselfshiftfile,norotsalfile,$
      naverref=1,itbest=itselbest,tselect=statselect,$
      flatten=flattennisp,smear=-smearnisp,applypx2asym=0,$
      alignshift=0,alignmetcalf=1,$
      maxmetcalf=maxselfmetcalf,cutmetqual=cutselfmetqual,$
      trimbox=nisptrimcenter,$
      transdatafile=nispdir+'/temp/gong_selfalign.dat',$
      plotpath=nispdir+'/gong_selfalign',show=0  ; (=1 showexs each it)
    if (show eq 2) then begin
      print,' =====  check Metcalf self-align improvement (no rotation)'
      showex,norotsalfile,norotselfshiftfile
    endif
  endif else skipmetselfal=1
endif

; check NISP metselfalign
if (show eq 2 and skipmetselfal eq 0) then begin
  print,' ===== check metcalf selfalign: A = no rotation, less erratic?'
  showex,norotsalfile,nispselectfile
endif

; put disk-center rotation back in for use below rotsalfile =
; selfaligned selected fulldisk cubefile as original images
reformcubefile,norotsalfile,rotsalfile,shift=+shiftrotcen,/splinip

; check
if (show ge 2 and not(skipshiftselfal eq 1 and skipmetselfal eq 1)) then begin
  print,' ===== check eventual selfalign versus original selection'
  print,' ===== zoom-in, shift time slider, A better than B?'
  showex,rotsalfile,nispselectfile
endif

; skip co-align
if (skipcoal eq 1) then begin
  print,' ===== skip co-align NISP best to AIA 304'
  ; option trimboxcoal = central field; need to know that
  if (trimboxcoal[0] lt -2) then begin
    boxsize=abs(trimboxcoal) ; in arcsec
    boxpx=fix(boxsize/pxnisp)
    trimboxcoal=[nx_nisp/2-boxpx/2,ny_nisp/2-boxpx/2,$
                 nx_nisp/2+boxpx/2,ny_nisp/2+boxpx/2]
  endif
  goto,SKIPCOALIGN
endif

; tai's of selected it
taibestnisp=selnisptai[itselbest]
taibest304=taibestnisp-timedelay*60. ; seconds; NB single-image rotation problem

; ====== tmode 1: align single pair of images NISP and 304

if (tmode eq 1) then begin

; =========  get best NISP and closest full-disk 304

; get best fulldisk NISP Halpha
  headrotsal=headfits(rotsalfile)
  ntsel=fxpar(headrotsal,'naxis3')
  trange=[itselbest,itselbest]
  fullnisp=readfitscube(rotsalfile,trange=trange)

; black2grey for bad dark outside disk and write as file
  cenbest=fullnisp[nisptrimcenter[0]:nisptrimcenter[2],$
                   nisptrimcenter[1]:nisptrimcenter[3]]
  meanha=avg(cenbest)
  cut=meanha/5.
  reformimage,fullnisp,bestnisp,black2grey=[meanha-2*cut,meanha]

; write file 
  writefits,nispfullfile,bestnisp

; get corresponding 304 fulldisk image (check on earlier download)
  datetime=anytim2utc(taibest304,/ccsds)  ; taibest304 = TAI best with delay
  strput,datetime,'_',10 
  datetime_rr=str_replace(datetime,'-','.')
; check on earlier download
  if (file_test(nispdir+'/temp/images304/sdo_'+datetime_rr+'_304.fits')) $
  then sdo_getimage,nispdir+'/temp/images304/sdo_'+datetime_rr+'_304.fits',$
    '304',sdoim304,indexfull304,outdir=nispdir+'/temp/images304/' else $
      sdo_getimage,datetime_rr,'304',sdoim304,indexfull304,$
    outdir=nispdir+'/temp/images304/'

; shift fulldisk 304 over center splineshift 304 to mag sampling
  if (file_test(sdodir+'/driftscenter') ne 1) then begin
    print," ##### abort: no sdodir+'/driftscenter'"
    return
  endif
  shift304mag=sdo_getonesplineshift(taibest304,611,sdodir+'/driftscenter')
  aia304=shift_img(sdoim304,-shift304mag,missing=0)

; tmode=1 image inputs for further processing and Metcalf coalign
  im304=aia304
  imnisp=bestnisp

end ; end tmode 1 = get pair of single full-disk images for aligning

; ===== trimbox=-1 initial blink for defining trimboxcoal
; transform AIA to NISP format
if (trimboxcoal[0] eq -1) then begin
  reformimage,im304,aianisp304,$
    congridfactor=[0.6/pxnisp,0.6/pxnisp],$
    nxlarge=nx_nisp,nylarge=ny_nisp,$
    cutcentralx=nx_nisp,cutcentraly=ny_nisp,/splinip
  print,' ##### select trimboxcoal, rerun with trimboxcoal'
  showex,imnisp,aianisp304,/blink
  return
endif

; next here because trimboxcoal=-1 => tmode=1 startoff ended

; option trimboxcoal = central field 
if (trimboxcoal[0] lt -2) then begin
  boxsize=abs(trimboxcoal) ; in arcsec (AIA pipeline center = 750x750 arcsec)
  boxpx=fix(boxsize/pxnisp)
  trimboxcoal=[nx_nisp/2-boxpx/2,ny_nisp/2-boxpx/2,$
               nx_nisp/2+boxpx/2,ny_nisp/2+boxpx/2]
endif

; (x,y) center trimboxcoal from disk center in NISP px
xcentrim=(trimboxcoal[2]+trimboxcoal[0])/2-nx_nisp/2 
ycentrim=(trimboxcoal[3]+trimboxcoal[1])/2-ny_nisp/2

; solar (X,Y) of center trimboxcoal in arcsec (stable for /notrack)
solx=xcentrim/pxnisp
soly=ycentrim/pxnisp

; ====== tmode 2: get time-averaged NISP and 304 for aligning 

if (tmode eq 2) then begin

; AIA: get averspan sequence, shift to HMI mag, derotate, average 

; get 304 full-disk aversequence into arrray
  taistart=taibest304-averspan*30.
  taiend=taibest304+averspan*30.
  sdo_getimagesequence,taistart,taiend,'304',im304arr,index304arr,$
    maxfiles=120,cadence=-1,raw=0,outdir=nispdir+'/temp/images304/'
  ncoal304=n_elements(index304arr)

; shift fulldisk 304 over cross-align splineshift 304 to mag 
; and derotate around taibest for (X,Y) of center trimboxcoal
  if (file_test(sdodir+'/driftscenter') ne 1) then begin
    print," ##### abort: no sdodir+'/driftscenter'"
    return
  endif
  aia304arr=im304arr
  for it=0,ncoal304-1 do begin
    
; header t_obs > tai
    tobs=index304arr[it].t_obs
    taiim=anytim2tai(tobs)

; SDO crossalign shift
    shift304mag=sdo_getonesplineshift(taiim,611,sdodir+'/driftscenter')
;;     aia304arr[*,*,it]=shift_img(im304arr[*,*,it],-shift304mag,missing=0)>0
;RR no >0: aia304arr darkest gets much lighter, missing makes no difference
;RR so splinip=1 adds negative overshoots in black offlimb, cut to 0 helps
;RR OOPS splinip=0 in _rr version OK greyscale but shows fixed shift
;RR added splinip=2 bilinear interpolation (without affecting old choices)
;RR OK: greyscale remains intact and shift304mag applies properly

; derotation for center of trimboxcoal
    rot304arcsec=rot_xy(solx,soly,tstart=taibest304,tend=taiim)-[solx,soly]
    shiftrot=reform(rot304arcsec/px_aia)
    aia304arr[*,*,it]=shift_img_rr(im304arr[*,*,it],-shiftrot-shift304mag,$
                                   missing=0,splinip=2)
  endfor
  ;; ; check
  ;; showex,aia304arr,im304arr
  ;; STOP
  
; time-average 304 sequence 
  im304=total(aia304arr,3)/ncoal304
  
; NISP: get averspan sequence, derotate, average 

; get NISP full-disk aversequence
  taibestnisp=taibestnisp[0]  ;RR needs [0], took an hour - F##K IDL
  taistart=taibestnisp-fix(averspan*30.)
  taiend=taibestnisp+fix(averspan*30.)
  dummy=min(abs(selnisptai-taistart),itnispstart)
  dummy=min(abs(selnisptai-taiend),itnispend)
  imnisparr=readfitscube(rotsalfile,trange=[itnispstart,itnispend],$
                         nxout=nxnisp,nyout=nynisp,ntout=ncoalnisp)

; derotate around taibest for (X,Y) of center of trimboxcoal
  nisparr=imnisparr
  for it=0,ncoalnisp-1 do begin
    taiim=selnisptai[itnispstart+it]
    rotnisparcsec=rot_xy(solx,soly,tstart=taibestnisp,tend=taiim)-[solx,soly]
    shiftrot=reform(rotnisparcsec/pxnisp)
    nisparr[*,*,it]=shift_img_rr(imnisparr[*,*,it],-shiftrot,$
                                 missing=0,splinip=2)
  endfor
  ;; ; check
  ;; showex,nisparr,imnisparr
  ;; STOP

; time-average NISP sequence 
  imnisp=total(nisparr,3)/ncoalnisp

endif ; end tmode=2 multi-image temporal mean construction

; ===== further processing for either tmode

; ===== muckrandom = randomize bright extremes (CBPs) in both (TRICKY)
;RR ;;; = trial clipping dark but 304 filaments are not darker than quiet
;RR       and cutting Halpha dark cuts limb darkening

if (muckrandom ne 0) then begin
  print,' ----- apply muckrandom cuts to both best images'
  for ibest=1,2 do begin
    if (ibest eq 1) then imbest=im304
    if (ibest eq 2) then imbest=imnisp

; cut out large central box for clip determination
    szbest=size(imbest)
    nxbest=szbest[1]
    nybest=szbest[2]
    bestcut=imbest[nxbest/2-nxbest/4:nxbest/2+nxbest/4,$
                   nybest/2-nybest/4:nybest/2+nybest/4]

; iterate clipping bright outliers as sdo_firelevel.pro
    thiskeep=bestcut
    for iter=0,4 do begin
      mom=moment(thiskeep,sdev=sdev)
      keep=thiskeep[where(thiskeep lt mom[0]+2*sdev)]
      ;;;  and thiskeep gt mom[0]-2*sdev)] 
      thiskeep=keep ; now an 1D index of non-brightest pixels
    endfor
    imbestmax=mom[0]+3*sdev  ; judicious, from inspect blinks below
    ;;; imbestmin=mom[0]-2*sdev  

; clip at these values
    imclip=imbest
    if (max(imclip) gt imbestmax) then imclip=imbest<imbestmax $
    else imclip[0,0]=imbestmax  ; force same greyscale when no clip
;;; if (min(imclip) lt imbestmin) then imclip=imclip>imbestmin $
;;; else imclip[0,1]=imbestmin 
    ;; ; inspect
    ;; showex,imbest,imclip,/blink

; set random greyscale range over midrange clipped cutout
    clipcut=imclip[nxbest/2-nxbest/4:nxbest/2+nxbest/4,$ 
                   nybest/2-nybest/4:nybest/2+nybest/4]
;;    mima=minmax(histo_opt_rr(clipcut,0.3)) ; only medium range
    mima=minmax(clipcut) ;RR smear smears these 
    randomim=(randomu(seed,nxbest,nybest)*(mima[1]-mima[0]))+mima[0]

; change excess px to random in this greyscale range
    immuck=imbest
    wheremuck=where(imbest gt imbestmax-1) ;; or imbest lt imbestmin+1)
    if (wheremuck[0] ne -1) then immuck[wheremuck]=randomim[wheremuck]
    ;; ; inspect
    ;; showex,imbest,immuck,/blink
    
; mucked result
    if (ibest eq 1) then im304=immuck
    if (ibest eq 2) then imnisp=immuck
  endfor ; end ibest loop over the two "best" full-disk images
endif    ; end muckrandom=1

; reverse NISP greycale = RR magic
reformimage,imnisp,nisprev,/intreverse

; TRICKY heightdiff: shift Halpha to "higher-formed?" 304
; only for excentric trimboxcoal, following findimshift_tiles.pro
shiftnisp=nisprev
if (heightdiff gt 1 $
    and (abs(xcentrim) gt 200 or abs(ycentrim) gt 200)) then begin
  print,' ----- apply heightdiff correction (tricky)'
  solvec=sqrt(xcentrim^2+ycentrim^2)
  sintheta=solvec/pxnisprad
  delradkm=heightdiff*sintheta ; component away from limb
  delradpx=delradkm*pxnisprad/696340.
  gamma=atan(abs(soly)/(abs(solx)>0.01))   ; 0.01 avoids y/0=NaN
  delradx=delradpx*cos(gamma)
  delrady=delradpx*sin(gamma)
; correct quadrant signs
  if (solx lt 0) then delradx=-delradx
  if (soly lt 0) then delrady=-delrady
; shift Halpha outward over radial height-difference component
  if (max(abs([delradx,delrady])) gt 0.01) then $
    shiftnisp=shift_img(imnisp,+[delradx,delrady],$
                        missing=meanha) ; borders grey
endif ; end heightdiff correction for trimboxcoal

; write these images for align show while skipping coal
writefits,aiabestim,im304
writefits,nispbestim,shiftnisp

; ===== NISP-304 co-alignment per Metcalf for either tmode 

im1=im304
im2=shiftnisp
px1=0.6
px2=pxnisp
angle=anglenisp
flattencoal=100 ;RR judicious 
smearcoal=-10   ;RR judicious 
finalrinse=40   ; tile size in px (plm arcsec); full disk permitted 
;RR finalrinse heightdiff negative for Halpha below 304

; run findalignimages
print,' ----- start findalignimages on reversed NISP to AIA 304'
findalignimages,im1,im2,px1,px2,angle,$ 
  px2asym,shiftfor,shiftrev,nxfor,nyfor,nxrev,nyrev,$
  nxmuckfor=nxmuckfor,nymuckfor=nymuckfor,$
  nxmuckrev=nxmuckrev,nymuckrev=nymuckrev,$
  skipmetcalf=skipmetcalf,maxmetcalf=maxcometcalf,minmetqual=minmetqual,$
  applypx2asym=applypx2asym,trimboxim2=trimboxcoal,inshift12=inshift12,$
  flatten=flattencoal,smearim1=smearcoal,smearim2=smearcoal,$
  histopim1=histopim1,histopim2=histopim2,$
  muckdarkim1=muckdarkim1,muckdarkim2=muckdarkim2,$
  muckbrightim1=muckbrightim1,muckbrightim2=muckbrightim2,$
  metqual=metqual,nmet=nmet,finalrinse=finalrinse,heightdiff=-heightdiff,$
  checkmetcalf=checkmetcalf,blink=blink,show=(show eq 2),verbose=verbose

; write results into temp file for future skipcoal skips
writecol,coalignfile,px2,angle,px2asym[0],px2asym[1],$
  shiftfor[0],shiftfor[1],shiftrev[0],shiftrev[1],$
  nxfor,nyfor,nxrev,nyrev,fmt='(8F13.4,4I6)'

; print results
if (verbose ne 0) then print,' ##### results:'+$
  '  nmet ='+trimd(nmet)+$
  '  metqual = '+trim(metqual,'(E8.2)')+$
  '  angle ='+trimd(angle,2)+$
  '  px2 ='+trimd(px2,4)+$
  '  pxnispnominal ='+trimd(pxnispnominal,4)+$
  '  shiftfor ='+trimd(shiftfor,1)+$
  '  shiftrev ='+trimd(shiftrev,1)+$
  '  nxfor, nyfor ='+trimd([nxfor,nyfor])+$
  '  nxrev, nyrev ='+trimd([nxrev,nyrev])

; print warning at large shift (shiftrev is in AIA px)
if (max(abs(shiftrev)) gt 20) then begin
  print,' ##### large shiftrev ='+trimd(shiftrev)+$
    ' suggests derailed co-align; better try better trimboxcoal'
  return
endif

SKIPCOALIGN: ; =========================================== SKIP 

; get co-alignment results if skipped
if (skipcoal eq 1) then begin
  readcol,coalignfile,px2,angle,xpx2asym,ypx2asym,$
    xshiftfor,yshiftfor,xshiftrev,yshiftrev,nxfor,nyfor,nxrev,nyrev
  px2=px2[0]
  angle=angle[0]
  px2asym=[xpx2asym[0],ypx2asym[0]]
  shiftfor=[xshiftfor[0],yshiftfor[0]]
  shiftrev=[xshiftrev[0],yshiftrev[0]]
  nxfor=fix(nxfor[0])
  nyfor=fix(nyfor[0])
  nxrev=fix(nxrev[0])
  nyrev=fix(nyrev[0])
endif

; set scale factors (follow @@@@@ demo = emulate content findalignimages.pro)
if (fixscale eq -1) then px2=pxnispnominal
if (fixscale eq -2) then begin  ; px2 too large from radial expansion
  rescale=1.-heightdiff/695700. ; heightdiff = 304 higher limb extension (km)
  px2=px2*rescale
endif
; ?? add other fixscale options (scale from limbfit?) here
pxratio=px_aia/px2                          ; about 0.6
pxscale2=1./[pxratio,pxratio]

; ========== produce aligned center/cubesxal cubefile 

if (file_test(sdodir+'/center/cubesxal') ne 1) then begin
  print,' ##### nisp_sdo.pro abort: no sdodir center/cubesxal dir'
  return
endif

print,' ----- make nispaiacenterfile'

; transform NISP full-disk file to AIA-aligned AIA pixels
if (notrackcenter eq 0) then $
  reformcubefile,norotsalfile,nispaiacenterfile,$
  congridfactor=pxscale2,shift=shiftrev,rotate=-angle,$
  missingvalue=missing_nisp,splinip=1
if (notrackcenter eq 1) then $
  reformcubefile,rotsalfile,nispaiacenterfile,$
  congridfactor=pxscale2,shift=shiftrev,rotate=-angle,$
  missingvalue=missing_nisp,splinip=1

; cut out center area, resample to AIA timing, write center/cubesxal file
headfull=headfits(nispaiacenterfile)
nxfull=fxpar(headfull,'naxis1')
nyfull=fxpar(headfull,'naxis2')
cutx=fix([nxfull/2.-nx_cen/2.,nxfull/2.+nx_cen/2.])
cuty=fix([nyfull/2.-ny_cen/2.,nyfull/2.+ny_cen/2.])
cutx[1]=cutx[1]-1
cuty[1]=cuty[1]-1
reformcubefile,nispaiacenterfile,nispcenterfile,$
  xrange=[cutx[0],cutx[1]],yrange=[cuty[0],cuty[1]],$
  timearr_in=selnisptai,timearr_out=aiacentai,/splinip

; fix the center NISP header to SDO content
headnisp=headcen
if (nisp eq 1) then $
  sxaddpar,headnisp,'channel','GONG Halpha on 304','SDO diagnostic name'
if (nisp eq 2) then $
  sxaddpar,headnisp,'channel','SOLIS Halpha on 304','SDO diagnostic name'
modfits,nispcenterfile,0,headnisp

; check: start showex center cubesxal at best it
; NB: very difficult to see whether the match is good
;  - off-center 304 prominences and bright activity have large heightdiff
;  - compare at it=itcenbest
;  - maybe better to apply mean(t), blink 304 against reverse nispha
if (checksdocenter eq 1) then begin
  dummy=min(abs(aiacentai-selnisptai[itselbest]),itcencoal)
  print,' ===== check blinks SDO center, it coalign ='+trimd(itcencoal)
  print,' ===== reverse B = NISP; if shifts then improve trimboxcoal'
  if (nisp eq 1) then wavindB=11 else wavindB=14
  showex,/allfits,fitsdirs=sdodir+'/center/cubesxal',/addlabels,$
    wavindA=7,wavindB=wavindB,itthis=itcencoal,/revB,/blink
endif else if (verbose ne 0) then print,' ----- skip checksdocenter'

; ===== finally produce aligned target NISP cubefile = desired product

print,' ----- finally make sdodir/target file'

; ?? OK?
; add shifts to derotate NISP target from its (X,Y) when notracktarget=0
; [usual JSOC cutout tracking with (XCEN,YCEN) = location at SDO start]
if (notracktarget eq 0) then begin
  ; get (XCEN,YCEN) of the rotating target at itbest time
  xybest=rot_xy(xcen_tar,ycen_tar,$
                tstart=aiatargettai[0],tend=selnisptai[itselbest])
  xcen_best=xybest[0]
  ycen_best=xybest[1]
  ; get shifts to rotate all NISPs to itbest location at this (X,Y)
  nsel=n_elements(selnisptai)
  rotarcsec=fltarr(2,nsel)
  for it=1,nsel-1 do rotarcsec[*,it]=$
    rot_xy(xcen_best,ycen_best,$
;;;   tstart=selnisptai[itselbest],tend=selnisptai[it])$     ; original 
           tstart=selnisptai[it],tend=selnisptai[itselbest])$
    -[xcen_best,ycen_best]
  shiftrotbest=transpose(rotarcsec)/px_aia ; shifts in AIA px after rescale
  ; combine backrotation shifts with shiftrev
  shiftrottarget=shiftrotbest
  shiftrottarget[*,0]=shiftrev[0]-shiftrotbest[*,0] ; shiftrev finest = AIA
  shiftrottarget[*,1]=shiftrev[1]-shiftrotbest[*,1]
endif else begin
  xcen_best=xcen_tar
  ycen_best=ycen_tar
endelse

; transform NISP full-disk file to AIA-target-aligned on AIA pixel size
if (notracktarget eq 0) then $
  reformcubefile,norotsalfile,nispaiatargetfile,$
  congridfactor=pxscale2,shift=shiftrottarget,rotate=-angle,$
  missingvalue=missing_nisp,splinip=1
if (notracktarget eq 1) then $
  reformcubefile,rotsalfile,nispaiatargetfile,$
  congridfactor=pxscale2,shift=shiftrev,rotate=-angle,$
  missingvalue=missing_nisp,splinip=1

; target dimensions
headfull=headfits(nispaiatargetfile)
nxfull=fxpar(headfull,'naxis1')
nyfull=fxpar(headfull,'naxis2')
xcenpx=xcen_best/px_aia+nxfull/2. 
ycenpx=ycen_best/px_aia+nyfull/2. 

; get heightdiff shift for off-center target if small trimboxcoal
; NB: physically wrong, only suited for morphology tests
;     small trimboxcoal may have produced non-expansion-increased px scale
delradx=0
delrady=0
nxtrimcoal=trimboxcoal[2]-trimboxcoal[0]
nytrimcoal=trimboxcoal[3]-trimboxcoal[1]
if (fixscale eq 0 and heightdiff gt 1 and $
    (nxtrimcoal lt 400 and nytrimcoal lt 400) and $ 
    (abs(xcenpx) gt 200 or abs(ycenpx) gt 200)) then begin
  print,' ----- apply heightdiff correction (tricky)'
  solvec=sqrt(xcenpx^2+ycenpx^2)
  sintheta=solvec/pxnisprad
  delradkm=heightdiff*sintheta ; component out from limb
  delradpx=delradkm*pxnisprad/696340.
  gamma=atan(abs(ycen_tar)/(abs(xcen_tar)>0.01))   ; 0.01 avoids y/0=NaN
  delradx=delradpx*cos(gamma)
  delrady=delradpx*sin(gamma)
; correct quadrant signs
  if (xcen_tar lt 0) then delradx=-delradx
  if (ycen_tar lt 0) then delrady=-delrady
endif ; end get heightdiff correction for target

; cut out target area
headfull=headfits(nispaiatargetfile)
nxfull=fxpar(headfull,'naxis1')
nyfull=fxpar(headfull,'naxis2')
xcenpx=xcen_best/px_aia+nxfull/2. 
ycenpx=ycen_best/px_aia+nyfull/2. 
cuttarx=fix([xcenpx-nx_tar/2.+delradx,xcenpx+nx_tar/2.+delradx])
cuttary=fix([ycenpx-ny_tar/2.+delrady,ycenpx+ny_tar/2.+delrady])
cuttarx[1]=cuttarx[1]-1 ;RR IDL counts fingers 0-9
cuttary[1]=cuttary[1]-1
reformcubefile,nispaiatargetfile,nisptarcutfile,$
  xrange=[cuttarx[0],cuttarx[1]],yrange=[cuttary[0],cuttary[1]]

; shift-selfalign tracked nisp target to itbest (rubber sheet?) (NB: AIA px)
; only for tracked SDO, otherwise scene rotates; not when fixscale 
if (notracktarget eq 0 and fixscale ne 0) then $
  selfalignfitscube,nisptarcutfile,nisptarselffile,$
  naverref=1,itbest=itselbest,tselect=-1,$
  flatten=flattencoal,smear=smearcoal,applypx2asym=0,$
  alignshift=1,alignmetcalf=0,$
  trimbox=-1,plotpath=nispdir+'/nisp_target_selfalign',show=0 $
else spawn,'cp '+nisptarcutfile+' '+nisptarselffile

; resample to AIA timing = final result
reformcubefile,nisptarselffile,nisptargetfile,$
  timearr_in=selnisptai,timearr_out=aiatargettai,/splinip

; fix the target NISP header to SDO content
headnisp=headtar
if (nisp eq 1) then $
  sxaddpar,headnisp,'channel','GONG Halpha on 304','SDO diagnostic name'
if (nisp eq 2) then $
  sxaddpar,headnisp,'channel','SOLIS Halpha on 304','SDO diagnostic name'
modfits,nisptargetfile,0,headnisp

; final check: showex-blink target at align it (SOLIS wavindB=14 if GONG) 
print,' +++++ done: blink target 304 + rev(<nisp>aia) at align it'
dummy=min(abs(aiatargettai-selnisptai[itselbest]),ittarbest)
if (checksdotarget eq 1) then begin
  if (nisp eq 1) then wavindB=13 else wavindB=14
  showex,/allfits,fitsdirs=sdodir+'/target/cubes',/addlabels,$
    wavindA=7,wavindB=wavindB,itthis=ittarbest,/revB,/blink
endif

; print and touch align it for use in checks
print,' ===== target align it ='+trimd(ittarbest)
spawn,' touch '+nispdir+'/nisp-align_it_target_'+trim(ittarbest)

; print elapsed time
timedone=systime(1)
timelaps=ntostr((timedone-timestartnispsdo)/60.,format='(F11.1)')
print,' ===== nisp_sdo took '+timelaps+' minutes'

; cleanup reminder
print,' ===== if all done remove dir '+nispdir+'/temp'

;; STOP ; maybe do something?

end

; =============== main for test/call per IDLWAVE H-c ======================

; NISP trial quiet center nr 2 (lousy SOLIS, better near end)

cd,'/home/rutten/data/NISP/2012-08-15-quiet'
sdodir='sdo_limb_notrack'
notrackcenter=1
notracktarget=1

; which NISP telescope ## choose by flipping between the two
whichnisp='SOLIS'
whichnisp='GONG'

selstring='120815'

; rms, trimbox, tmode, delay etc below per whichnisp

; parameters
muckrandom=1
averspan=6   ; duration in min for tmode=2, same for both

; improvements from initial tmode=1 Metcalf
if (whichnisp eq 'GONG') then begin
  gongstation='B'
  rmsmin=500
  itbestnisp=11         ; amidst 10 good min at start (itselbest=5)
;; trimboxcoal=[738,804,1278,1275]  ; GONG near center
;; trimboxcoal=[824,824,1224,1224] ; SOLIS around center, smaller
;; trimboxcoal=[1306,932,1693,1183] ; near target
;; trimboxcoal=[1192,864,1635,1689] ; target inside ## DERAILS
;; trimboxcoal=[695,689,1382,1416]  ; larger center trimbox
;; trimboxcoal=-1
;; trimboxcoal=[1100,1500,1522,1760] ; near limb target
  trimboxcoal=-800
  fixscale=0
  heightdiff=0  ; correct?  304 just as high as GONG Halpha?
  timedelay=0
  anglenisp=0.29
  pxnisp=1.0556   ; for trimboxcoal=-800
  outfile='gong800.fits'
endif

if (whichnisp eq 'SOLIS') then begin
  rmsmin=0
  rmsmin=400
  itbestnisp=9     ; not highest rms but good period around it
  trimboxcoal=-800 ; large center field as trimbox (SDO center 750 arcsec) 
  fixscale=-2
  heightdiff=2000    ; for tiled finalrinse and target cutout
  timedelay=0
  anglenisp=0
  pxnisp=1.05
  outfile='solis800.fits'
endif

; set skips (set 0 first time; reset all to 0 for new rmsmin choice)
skipselbest=1
skipshiftselfal=1
skipmetselfal=1 ; if GONG only one station selected this is not done
skipcoal=0

; coalign Metcalf choices
show=1
blink=4
verbose=1

; result showex choices
chekcmetcalf=1
checksdocenter=1
checkmetcalf=1
checksdotarget=1

nisp_sdo,whichnisp,selstring,rmsmin,$
  nispdir=nispdir,sdodir=sdodir,aiafile=aiafile,outfile=outfile,$
  gongstation=gongstation,itbestnisp=itbestnisp,$
  fixscale=fixscale,muckrandom=muckrandom,$
  averspan=averspan,timedelay=timedelay,$
  notrackcenter=notrackcenter,notracktarget=notracktarget,$
  pxnisp=pxnisp,anglenisp=anglenisp,$
  maxselfmetcalf=maxselfmetcalf,cutselfmetqual=cutselfmetqual,$
  trimboxcoal=trimboxcoal,heightdiff=heightdiff,inshift12=inshift12,$
  skipselbest=skipselbest,$
  skipshiftselfal=skipshiftselfal,skipmetselfal=skipmetselfal,$
  skipcoal=skipcoal,checkmetcalf=checkmetcalf,$
  checksdocenter=checksdocenter,checksdotarget=checksdotarget,$
  blink=blink,show=show,verbose=verbose

end

