; file: gong_getimage.pro
; init: Mar 27 2017  Rob Rutten  Deil
; last: Aug  4 2020  Rob Rutten  Deil
; note: cumbersome from trying to keep macOS compatible

;RR GONG Halpha info: https://www.nso.edu/data/nisp-data/h-alpha/
;RR (B: Big Bear; C: CTIO; L: Learmonth; M: Mauna Loa, T: El Teide;
;RR  U: Udaipur.
;RR file name format:
;RR   ftp://gong2.nso.edu/HA/haf/YYYYMM/YYYYMMDDHHMMSSXh.fits.fz
;RR   whereYY... is date-time UT, X = 1-letter station identifier, h = Halpha
;RR in 2017 switches in file-per-station SS second values (last 2 digits)
;RR taken care per double obsidents specifation below
;RR would be better to have SS as wildcards but curl doesn't permit such
;RR use lftp instead?  Lynx?
;RR lftp on command line:
;RR  > lftp  ftp://gong2.nso.edu/HA/haf/201210/20121020
;RR  > mget 201210200704*.fits.fz
;RR  > exit
;RR OK, gets 20121020070434Lh.fits.fz, is only one (only L, 1 per minute)
;RR but usually there are more station files per minutes
;RR maybe get all and select the best?  rms around mean for center part?
;RR OK - can be done - @ do if I need GONG Halpha myself
;RR other issue = Koza problem '2012.10.20_07:04':
;RR   older L = Learmonth images wrong, bandwidth problems, overwritten 
;RR   repairs planned (Andrew Marble <amarble@nso.edu> email Dec 2 2018) 

;+
function gong_getimage,specifier,outdir=outdir

 ; get a GONG Halpha fitsfile taken during the specified minute
 ; (available from October 2009)
 ;
 ; INPUTS:
 ;   specifier: 3 options
 ;     archive mode: specifier='YYYY.MM.DD_HH:MM' datetime string 
 ;       mandatory RR format, eg: '2013.12.21_12:00'
 ;       get older GONG Halpha image taken at that time
 ;     latest mode: specifier='latest'
 ;       get latest GONG Halpha image
 ;     file mode: specifier = string path/filename of GONG Halpha fits image
 ;
 ; KEYWORD IPUT:
 ;   outdir: for fits file, default '/tmp'
 ;
 ; OUTPUT:
 ;   string = name fitsfile in outdir, as 'gong_ha_2020.01.01:00:00_Lh.fits'
 ; 
 ; RESTRICTIONS:
 ;   spawns curl, no longer wget, for the Mac-challenged
 ;   needs my rridl IDL library
 ;     https://robrutten.nl/rridl/rridl.zip
 ;   needs David Fanning's coyote IDL library
 ;     http://www.idlcoyote.com/documents/programs.php
 ;   needs unix/linux curl and lynx for latest
 ;   works fine in my IDL 6.4 but may act differently in newer IDL
 ;
 ; HISTORY:
 ;   Mar 27 2017 RR: start
 ;   Dec  3 2018 RR: rework after failure for Julius Koza
 ;   Jun  4 2019 RR: datetime > specifier options 
;-

; answer no-parameter query 
if (n_params(0) lt 1) then begin
  sp,gong_getimage
  retall
endif

; keyword default
if (n_elements(outdir) eq 0) then outdir='/tmp'

; interpret specifier
latest=(specifier eq 'latest')
archive=(strmatch(specifier,'20*.*.*_*:*'))
file=(latest eq 0 and archive eq 0)

if (latest+archive+file ne 1) then begin
  print, " ##### gong_featurelocator abort:"+$
    " specifier not YYYY.MM.DD_HH:MM or 'latest' or GONG file"
  sp,gong_getimage
  return,-1
endif

; ----- latest mode
; example: http://halpha.nso.edu/keep/hag/201906/20190604/
if (latest) then begin
  spawn,"date --rfc-3339='date' > /tmp/gongdate"
  readcol,'/tmp/gongdate',gongdatestring,format='A'
  aa=strsplit(gongdatestring[0],'-',/extract)
  yyyy=aa[0]
  momo=aa[1]
  dd=aa[2]
  datetime=yyyy+'.'+momo+'.'+dd+'_00:00' ; fake time
  
; find URL of latest GONG image file per lynx
;   following https://stackoverflow.com/questions/22510705
  spawn,"lynx -dump ftp://gong2.nso.edu/HA/haf/"+yyyy+momo+"/"$
    +yyyy+momo+dd+" | awk '/20*/{url=$2}END{print url}' > /tmp/gonglateststr"
  readcol,'/tmp/gonglateststr',gongstring,format='A'

  ; get GONG file per curl
  hafile=outdir+'/gong_ha_'+datetime+'_latest.fits'   
  spawn,'curl -s '+gongstring+' -o '+hafile+'.fz'+$
    ' && echo "1" > /tmp/curlsuccess || echo "0"  > /tmp/curlsuccess'
  readcol,'/tmp/curlsuccess',success,format='I',/silent 
  if (success eq 0) then begin
    print,' ##### gong get latest abort: curl failure gongstring '+gongstring
    return,-1
  endif

  ; uncompress GONG file
  spawn,'rm -f '+hafile
  spawn,'funpack -D '+hafile+'.fz'
  spawn,'rm -f /tmp/wgetsuccess /tmp/gongdate /tmp/gonglateststr '+$
    '/tmp/curlsuccess'
  return,hafile
  
endif ; end of latest mode

; ----- archive mode

; ftp://gong2.nso.edu/HA/haf/201906/20190604/20190604000730Lh.fits.fz

if (archive) then begin
; convert datestring to GONG URL notation
  datetime=specifier
  aa=strsplit(specifier,'_',/extract)
  bb=strsplit(aa[0],'.',/extract)
  yyyy=bb[0]
  momo=bb[1]
  dd=bb[2]
  cc=strsplit(aa[1],':',/extract)
  hh=cc[0]
  mimi=cc[1]
  if (yyyy lt 2009 or (yyyy eq 2009 and momo lt 10)) then begin
    print,' ###### gong_getimage abort: no images before 2009-10'
    retall
  endif

; get an image
; observations may not contain this minute, then search around
  thismimi=mimi
  startmin=fix(thismimi)

REPEATMIMI:
  print,' ----- try datetime = '$
    +trim(yyyy)+trim(momo)+trim(dd)+trim(hh)+trim(thismimi)
  urlstart='ftp://gong2.nso.edu/HA/haf/'+yyyy+momo+'/'+yyyy+momo+dd+$
    '/'+yyyy+momo+dd+hh+thismimi
  
; curl: {alpha,beta,gamma}.com or mysite.com/files[0001-0100].txt
;; urlend='[00-99]{Uh,Th}.fits.fz'
;; urlend={14Th,34Ch,54Bh,..}.fits.fz'
;RR should work but they don't, don't know why
;RR therefore split out into loop over the station idents in filenames
;RR this can go away by using lftp with wget but can Macs handle that?
;RR how improve on this while keeping macOS compatible?
  obsidents=['10Th','30Ch','50Bh','10Mh','30Lh','50Uh','50Zh','34Ah',$
             '14Th','34Ch','54Bh','14Mh','34Lh','54Uh','54Zh','34Ah']
  nobs=n_elements(obsidents)
  for iobs=0,nobs-1 do begin
    hafile=outdir+'/gong_ha_'+datetime+'_'+obsidents[iobs]+'.fits'   
    print,' ----- try GONG stationident: '+obsidents[iobs]+$
      ' with '+urlstart+obsidents[iobs]+'.fits.fz'
    spawn,'rm -f '+hafile+'.fz'
    spawn,'curl -s '+urlstart+obsidents[iobs]+'.fits.fz -o '+hafile+'.fz'+$ 
      ' && echo "1" > /tmp/wgetsuccess || echo "0"  > /tmp/wgetsuccess'
    if (obsidents[iobs] eq '50Bh') then STOP
    readcol,'/tmp/wgetsuccess',success,format='I',/silent 
    if (success eq 1) then begin
      print,' ----- success'
      break
    endif else print,' ----- nope'
  endfor
  if (success eq 0 and thismimi eq mimi) then begin
    thismin=startmin-1
    if (thismin eq -1) then thismin=startmin+1
    thismimi=strmid(string(fix(thismin+100),format='(I3)'),1,4) ; f**k IDL
    goto, REPEATMIMI
  endif
  if (success eq 0) then begin
    thismin=thismin+1
    if (thismin eq startmin) then thismin=startmin+1 ;RR wrong above 60
    thismimi=strmid(string(fix(thismin+100),format='(I3)'),1,4) ; f**k IDL
    if (thismin le startmin+1) then goto, REPEATMIMI
  endif

  if (success eq 0) then begin 
    print,' ##### gong_getimage.pro abort: no GONG Ha image for '+specifier
    return,-1
  endif else begin
    spawn,' rm -f '+hafile
    spawn,' funpack -D '+hafile+'.fz'
    spawn,'rm -f /tmp/wgetsuccess'
    return,hafile
  endelse

endif ; end of archive mode

; ----- file mode

if (file) then return,specifier

end      


; =============== main for testing per IDLWAVE H-c ======================
; ftp://gong2.nso.edu/HA/haf/201406/20140619/20140619095654Uh.fits.fz
; one per minute per active station at fixed seconds values

specifier='2014.06.19_09:56'  ; original test
;; specifier='2012.10.20_07:04'  ; Koza complaint; different keywords, angle
;; specifier='2018.06.01_12:00'  
;; specifier='2019.04.29_03:00' ; Kees de Jager birthdatetime
;; specifier='2020.01.01_00:00'
;; specifier='latest'
specifier='2020.05.30_14:58:46' ; SolO press-release first image 

hafile=gong_getimage(specifier,outdir='/home/rutten/data/trial')
ha=readfits(hafile,haheader)
showex,ha   

end
