; file: sv.pro = "show variable" = quick inspection (also per H-mouse3)
; init: Sep 27 2013  Rob Rutten  Deil
; last: Jul  7 2020  Rob Rutten  Deil

;+
pro sv,var

 ; PURPOSE:
 ;   show variable of any type, also for use as IDLWAVE mouseclick response
 ;   (for me HYPER+mouse3 with HYPER the Microsoft flag key set in .emacs)
 ;
 ; INPUT:
 ;   var = any type of variable
 ;
 ; OUTPUT:
 ;   depends on variable type:
 ;     string => print
 ;     string array => print first element
 ;     structure => help
 ;     pointer => help
 ;     0-D => only print the constant
 ;     1-D => cgplot against index unless constant, set /ylog when useful
 ;     2-D => cgimage or multicurve plot or stagger plot depending on size
 ;     3-D => showex
 ;     M-D => only print the dimensions
 ;
 ; HISTORY:
 ;   Sep 27 2013 RR: start at my first use of emacs IDLWAVE shell
 ;   Oct 13 2013 RR: this comment block
 ;   Oct 14 2013 RR: autoscale ximovie size for movie
 ;   Jan 23 2014 RR: string, structure, pointer
 ;   Nov  5 2017 RR: showex for movie 
;-

; no-parameter query reply
if (n_params() lt 1) then begin
  print,' usage: sv,var'
  return
endif

; return to main at error
on_error,1

; set 2D plot mode choice parameters
npixcut=20
ncurvecut=5

; get variable dimensions and type
sizevar=size(var)
ndim=sizevar[0]

; string or string array
if (sizevar[ndim+1] eq 7) then begin
  if (ndim lt 2) then print,'  sv: var is string = '+var
  if (ndim gt 1) then print,'  sv: var is string array; first = '+var[0]
  return
endif

; structure
if (sizevar[ndim+1] eq 8) then begin
  print,'  sv: var is structure: '
  help,var,/str
  return
endif

; pointer
if  (sizevar[ndim+1] eq 10) then begin
  print,'  sv: var is pointer: '
  help,*var_ptr 
  return
endif

; something numerical, get extrema
minv=min(var)
maxv=max(var)

; large dynamic range: set /ylog
if ((minv gt 0) and (alog10(maxv)-alog10(minv) gt 2)) $
then ylog=1 else ylog=0

; single value; print
if (ndim eq 0) then begin
  print,'  sv: var is scalar = ',ntostr(var) 
  return
endif

; 1D constant: print
if ((ndim eq 1) and (minv eq maxv)) then begin
  print,'  sv: var is constant = ',ntostr(var[0])
  return
endif

; 1D array: plot against array index
if (ndim eq 1) then begin
    ; only few points: add psym markers to the curve
  if (sizevar[1] lt 11) then psym=-4 else psym=0
  window,/free
  cgplot,var,yrange=[minv,maxv],ystyle=1,psym=psym,ylog=ylog,$
    xtitle='index',ytitle=scope_varname(var,level=-1)
  return
endif

;; ; (nx,ny,2) array: flickrr two images
;; if (ndim eq 3 and sizevar[3] eq 2) then begin
;;   flickrr,var[*,*,0],var[*,*,1]
;;   return
;; endif

; 3D array: movie (blink when nz=2) 
if (ndim eq 3) then begin
  showex,var
  return
endif

; yet larger array: just print dimensions
if (ndim gt 3) then begin
  print,'  sv: var is multi-D, dimensions: ',ntostr(sizevar)
  return
endif

; === what is left is 2D array: multicurve plot or stagger plot or image

; set decision values  
var2=var
if (sizevar[2] lt sizevar[1]) then var2=rotate(var,4) ; dim 1 = smallest
sizevar2=size(var2)

  ; both are large: probably an image, use showex for options?
if (sizevar2[1] gt npixcut) then begin
  cgimage,bytscl(var),/keep_aspect,charsize=1.5,$
    /display,position=[0.10,0.10,0.99,0.94],$
    /axes,axkeywords={font:1,ticklen:-0.02,$
    xtitle:'x  [px]',ytitle:'y  [px]'},$
    title=scope_varname(var,level=-1)
;;;;  showex,var ; alternative but the options are not so worthwhile
  return
endif

; === what is left now is a [small nx, larger ny] array

ncurve=sizevar2[1]

; small ncurve: multi-curve overlay plot
if (ncurve lt ncurvecut) then begin
  if (sizevar2[2] lt 11) then psym=-4 else psym=0
  cgplot,var2[0,*],yrange=[minv,maxv],ylog=ylog,psym=psym,/window
  for i=1,ncurve-1 do cgplot,var2[i,*],/overplot,/add,psym=psym
  return
endif

; and now we are left with many curves so do a stagger plot
delcurve=(float(maxv)-minv)/ncurve 
ymax=maxv+(ncurve-1)*delcurve
cgplot,var2[0,*],yrange=[minv,ymax],ylog=ylog,/window
for i=1,ncurve-1 do cgplot,var2[i,*]+delcurve*i,/overplot,/add

end 

; =============== try per Hyper-C ============================

;; ; two graphs
;; x=findgen(100)
;; y=sin(x/10)
;; sv,y
;; y3=y^3
;; sv,y3

; image
k=indgen(100)          
f=sin(k/5.)/exp(k/50.) 
g=cos(k/5.)*exp(k/50.) 
im1=f#g                  
sv,im1

;; ; multi-curve plot
;; k=indgen(100)          
;; f=sin(k/5.)/exp(k/50.) 
;; g=cos(k/5.)*exp(k/50.) 
;; s=f#g                  
;; s=s[*,0:10]
;; sv,s

;; ; stagger plot
;; k=indgen(100)          
;; f=sin(k/5.)/exp(k/50.) 
;; g=cos(k/5.)*exp(k/50.) 
;; s=f#g                  
;; s=s[*,0:15]
;; sv,s

end
