; file: splin2diter.pro
; init: Apr  8 2020  Rob Rutten  Deil
; last: May 23 2020  Rob Rutten  Deil

;+

pro splin2diter,tarr,xarr,yarr,tarrleft,$
  splax,splbx,splcx,spldx,$
  splay,splby,splcy,spldy,$
  smooth1=smooth1,smooth2=smooth2,niter=niter,cutsdev=cutsdev,$
  verbose=verbose

 ; PURPOSE:
 ;  iterate cubic spline fit successively removing outliers for 2 variables
 ;   
 ; INPUTS:
 ;   tarr: ordinate values, eg time
 ;   xarr, yarr: 2 functions, eg xshift and yshift
 ;   
 ; KEYWORD INPUTS:
 ;   smooth1: spline smoothing first fit, default 1E6 (0=interpolation)
 ;   smooth2: spline smoothing further fits, default 1E8 (smoother)
 ;   niter: number iterations, default 2
 ;   cutsdev: cut outliers beyond this, default 3
 ;   
 ; OUTPUTS:
 ;   tarrleft = remaining abcissa values
 ;   spla,splb,splc,spld: Vitas spline coefficients at these abscissae
 ;   
 ; HISTORY:
 ;    Apr  9 2020 RR: start from splin1diter.pro
;-

; answer no-parameter query 
if (n_params(0) lt 12) then begin
  sp,splin2diter
  return
endif

; defaults for keywords
if (n_elements(smooth1) eq 0) then smooth1=1E6
if (n_elements(smooth2) eq 0) then smooth2=1E8
if (n_elements(niter) eq 0) then niter=2
if (n_elements(cutsdev) eq 0) then cutsdev=3
if (n_elements(verbose) eq 0) then verbose=0

; initialize shrinkarrays (get smaller)
thistarr=tarr
thisxarr=xarr
thisyarr=yarr

; start iteration
for iter=0,niter-1 do begin

; first finer to cut out single outliers, then smoother
  if (iter eq 0) then thissmooth=smooth1 else thissmooth=smooth2

; get new array size, with check > 1 left
  nleft=n_elements(thistarr)
  if (nleft lt 3) then begin
    print,' ##### splin2diter: too few values left; killed'
    tarrleft=-1
    return
  endif

; make current spline approximations (RR-modified Vitas program)
  splinapp_set,thistarr,thisxarr,splax,splbx,splcx,spldx,smooth=thissmooth
  splinapp_set,thistarr,thisyarr,splay,splby,splcy,spldy,smooth=thissmooth

; shrink both arrays by removing outliers for next iteration
  if (iter lt niter-1) then begin

; get solutions at current samples (whole-arr calls)
    splinx=splinapp_get(thistarr,thistarr,splax,splbx,splcx,spldx)
    spliny=splinapp_get(thistarr,thistarr,splay,splby,splcy,spldy)
    
; get standard deviations
    diffx=abs(thisxarr-splinx)
    diffy=abs(thisyarr-spliny)
    momx=moment(diffx,sdev=sdevx)
    momy=moment(diffy,sdev=sdevy)

; find x outliers 
    thisgo=-1
    thisxgo=where(diffx gt cutsdev*sdevx)
    if (thisxgo[0] ne -1) then begin
      if (verbose ne 0) then print, $
        ' ----- iter'+trimd(iter)+' x outlier indices ',thisxgo
    endif

; find y outliers 
    thisygo=where(diffy gt cutsdev*sdevy)
    if (thisygo[0] ne -1) then begin
      thisgo=[thisxgo,thisygo]
      if (verbose ne 0) then print, $
        ' ----- iter'+trimd(iter)+' y outlier indices ',thisygo
    endif

; remove new outliers for next iteration if not too many
    if (n_elements(thisgo) lt 0.5*n_elements(thistarr) and thisgo[0] ne -1) $
       then remove,thisgo,thistarr,thisxarr,thisyarr
    
  endif

  ; output remaining ordinates where Vitas coefficients are given
  tarrleft=thistarr
  
; check on all gone
  if (n_elements(thistarr) lt 1) then begin
    print,' ##### splin1diter removed all remaining elements, killed'
    tarrleft=-1
    return
  endif

endfor ; end of iteration loop

; specify removal fraction
if (verbose) then begin
    nstart=n_elements(tarr)
    nleft=n_elements(tarrleft)
    print, ' ----- '+trimd(nleft)+' left out of'+trimd(nstart) 
  endif

end     ; end of program


; =============== main for testing per IDLWAVE H-c ======================

nval=50

tarr=indgen(nval)*1.1
sinxarr=sin(indgen(nval)*!pi/90.)
xarr=sinxarr+randomn(1,nval)*2
outlier=randomn(2,nval)*5
for it=3,nval-1,nval/20 do xarr[it]=xarr[it]+outlier[it]

cosyarr=2.-cos(indgen(nval)*!pi/90.)
yarr=cosyarr+randomn(1,nval)/5.
outlier=randomn(2,nval)/5.
for it=1,nval-1,nval/11 do yarr[it]=yarr[it]+outlier[it]

;; smooth1=1E2*nval/100. ; OK
;; smooth2=1E4*nval/100.
smooth1=1E4
smooth2=1E5*nval/100.
niter=4
cutsdev=3
verbose=1

splin2diter,tarr,xarr,yarr,tarrleft,$
  splax,splbx,splcx,spldx,$
  splay,splby,splcy,spldy,$
  smooth1=smooth1,smooth2=smooth2,niter=niter,cutsdev=cutsdev,$
  verbose=verbose

; get solution at remaing points
splinx=splinapp_get(tarrleft,tarrleft,splax,splbx,splcx,spldx)
spliny=splinapp_get(tarrleft,tarrleft,splay,splby,splcy,spldy)

plot,tarr,xarr,psym=2,symsize=20./sqrt(nval),$
  xrange=[tarr[0],tarr[nval-1]],xstyle=1,yrange=[-1,4],ystyle=1
oplot,tarr,yarr,psym=4,symsize=20./sqrt(nval)
oplot,tarr,sinxarr
oplot,tarr,cosyarr
oplot,tarrleft,splinx,psym=2,symsize=10./sqrt(nval)    ; gaps show outliers
oplot,tarrleft,spliny,psym=4,symsize=10./sqrt(nval)    ; gaps show outliers

end
