; file: splin1diter.pro
; init: Apr  7 2020  Rob Rutten  Deil
; last: May 23 2020  Rob Rutten  Deil

;+

pro splin1diter,xarr,yarr,xarrleft,spla,splb,splc,spld,$
  smooth1=smooth1,smooth2=smooth2,niter=niter,cutsdev=cutsdev,$
  verbose=verbose

 ; PURPOSE:
 ;  iterate cubic spline fit successively removing outliers for 1 variable
 ;   
 ; CALL:
 ;   see above
 ;   
 ; INPUTS:
 ;   xarr, yarr: ordinate (eg time) and function values   
 ;   
 ; KEYWORD INPUTS:
 ;   smooth1: spline smoothing first fit, default 1E6 (0=interpolation)
 ;   smooth2: spline smoothing further fits, default 1E8 (near linear)
 ;   niter: number iterations, default 2
 ;   cutsdev: cut outliers beyond this, default 3
 ;   
 ; OUTPUTS:
 ;   xarrleft = remaining abcissa values
 ;   spla,splb,splc,spld: Vitas spline coefficients at these abscissae
 ;   
 ; HISTORY:
 ;    Apr  8 2020 RR: start excerpt previous sdo_writepairspline.pro
;-

; answer no-parameter query 
if (n_params(0) lt 7) then begin
  sp,splin1diter
  return
endif

; defaults for keywords
if (n_elements(smooth1) eq 0) then smooth1=1E6
if (n_elements(smooth2) eq 0) then smooth2=1E8
if (n_elements(niter) eq 0) then niter=2
if (n_elements(cutsdev) eq 0) then cutsdev=3
if (n_elements(verbose) eq 0) then verbose=0

; initialize shrinkarrays (get smaller)
thisxarr=xarr
thisyarr=yarr

; start iteration 
for iter=0,niter-1 do begin

; first finer to cut out single outliers, then smoother
  if (iter eq 0) then thissmooth=smooth1 else thissmooth=smooth2

; get new array size, with check > 1 left
  nleft=n_elements(thisyarr)
  if (nleft lt 3) then begin
    print,' ##### splin1diter: too few values left; killed'
    xarrleft=-1
    return
  endif

; make spline approximations (RR-modified Vitas program)
  splinapp_set,thisxarr,thisyarr,spla,splb,splc,spld,smooth=thissmooth

; shrink both arrays removing outliers for next iteration
  if (iter lt niter-1) then begin

; get solution at current samples (whole-arr call)
    spliny=splinapp_get(thisxarr,thisxarr,spla,splb,splc,spld)
    
; get standard deviation
    diffy=abs(thisyarr-spliny)
    momy=moment(diffy,sdev=sdev)

; find outliers in y 
    thisindexgone=where(diffy gt cutsdev*sdev)
    if (thisindexgone[0] ne -1) then begin
      if (verbose ne 0) then print, $
        ' ----- iter'+trimd(iter)+' Y outliers at current indices ',$
        thisindexgone
    endif
    
; remove new outliers for next iteration if not too many
    if (n_elements(thisindexgo) lt 0.5*n_elements(thisxarr) and $
      thisindexgone[0] ne -1) then remove,thisindexgone,thisxarr,thisyarr
    
  endif

  ; output remaining ordinates where Vitas coefficients are given
  xarrleft=thisxarr
  
; check on all gone
  if (n_elements(thisxarr) lt 1) then begin
    print,' ##### splin1diter removed all remaining elements, killed'
    xarrleft=-1
    return
  endif
  
endfor ; end of iteration loop

end


; =============== main for testing per IDLWAVE H-c ======================


xarr=indgen(100)*1.1
sinyarr=sin(indgen(100)*!pi/90.)
yarr=sinyarr+randomn(1,100)/5.
outlier=randomn(2,100)
for ix=0,99,10 do yarr[ix]=yarr[ix]+outlier[ix]

smooth1=1E2
smooth2=1E4
niter=2
cutsdev=2
verbose=1

splin1diter,xarr,yarr,xarrleft,spla,splb,splc,spld,$
  smooth1=smooth1,smooth2=smooth2,niter=niter,cutsdev=cutsdev,$
  verbose=verbose

; get solution at remaining points (whole arr call)
spliny=splinapp_get(xarrleft,xarrleft,spla,splb,splc,spld)

; show 
plot,xarr,yarr,psym=2
oplot,xarr,sinyarr
oplot,xarrleft,spliny,psym=4  ; gaps show outliers

end
