; file: plotftsint.pro
; init: Feb  7 2012 
; last: Jul 13 2022  Rob Rutten  Deil
; note: run/try/examples per Hyper-C at end

;+
pro plotftsint,wav1,wav2,$
  outfilename=outfilename,hz=hz,tb=tb,ymin=ymin,ymax=ymax,$
  label=label,segwidth=segwidth,segcolor=segcolor,nopsclose=nopsclose,$
  _extra=plotkeywords

 ;   plot segment of "Neckel" solar disk-center optical-part intensity atlas
 ; 
 ; INPUTS:
 ;   Neckel atlas as ascii file or as IDL savefile, switch under ?? below
 ;   wav1, wav2 = desired segment, in Angstrom, within 3290 - 12510 AA
 ; 
 ; OPTIONAL KEYWORD INPUTS:
 ;   outfilename: string, default '/tmp/neckel_wav1-wav2', no extension
 ;   hz = 1/0: spectrum interval unit Hz^-1
 ;   tb = 1/0: plot brightness temperature instead of intensity 
 ;   ymin = minimum plotted y axis
 ;   ymax = maximum plotted y axis
 ;   label = 0: none
 ;         = 1: default 'FTS + Neckel calibration'
 ;         = string: that string
 ;   segwidth = add means over wavelength segments segwidth AA wide 
 ;   segcolor = color for this means histogram (default 'white')
 ;   nopsclose= 1/0: don't psclose to enable adding annotation to the ps plot 
 ;
 ; OUTPUTS:
 ;   graph in outfilename.ps, optional mean histogram in outfilename.dat  
 ; 
 ; REMARKS:
 ;   unfair to call this "Neckel atlas" since the data came from Brault;
 ;   a better name is "FTS atlas" and a proper cite string:
 ;     \citeads{1984SoPh...90..205N} % Neckel+Labs
 ;     \citeads{1999SoPh..184..421N} % Neckel announcement I put on ADS
 ;   ADS has 1972kpsa.book.....B but my traditional citation used to be
 ;    Brault & Testerman 1972 - not on ADS, something wrong? 
 ;
 ; HISTORY:
 ;   very long ago RR: start using original split-up Neckel files
 ;   less long ago Luc Rouppe van der Voort: asssembly into IDL savefile
 ;   May 24 2016 RR: cleanup, nicer axes, this header, name change
 ;   Aug  6 2018 RR: segwidth, nopsclose for review 2019SoPh..294..165R
;-

; no-parameter query answer
if (n_params() lt 2) then begin
  sp,plotftsint
  return
endif

; keyword defaults 
if (n_elements(outfilename) eq 0) then $
  outfilename='/tmp/neckel_'+trim(wav1)+'-'+trim(wav2) $
else outfilename=outfilename
if (n_elements(hz) eq 0) then hz=0
if (n_elements(tb) eq 0) then tb=0
if (n_elements(ymin) eq 0) then ymin=-1
if (n_elements(ymax) eq 0) then ymax=-1
if (n_elements(label) eq 0) then label=0
if (n_elements(nopsclose) eq 0) then nopsclose=0
if (n_elements(segwidth) eq 0) then segwidth=0
if (n_elements(segcolor) eq 0) then segcolor='white'

; open Neckel intensity atlas

; ?? next lines for idlsave file (faster)
neckelpath='/home/rutten/rr/wrk/atlases/neckel/'
neckelsave='neckel_intens_atlas.idlsave'
restore,neckelpath+neckelsave

;; ; ?? next lines for the file in my solar archive
;; ; https://robrutten.nl/rrweb/rjr-archive/atlases/neckel-intatlas.txt
;; neckelpath='/home/rutten/rr/www/rrweb/rjr-archive/atlases/'
;; neckelfile='neckel-intatlas.txt'
;; readcol,neckelpath+neckelfile,neckelwav,neckelint,neckelcont

yy=neckelint
nneckel=n_elements(neckelwav)

; optionally convert into brightness temperature
if (tb ne 0) then yy=brighttemp(neckelwav*1.E-4,neckelint*1.E7*1E4)

; optionally convert into per Hz when desired
clight=299792458.*1D10 ; in AA/s
if (hz ne 0) then yy=(neckelwav*neckelwav/clight)*neckelint*1E4

; make plot
xrange=[wav1,wav2]
if (ymin ne -1) then ymin=ymin else ymin=min(yy)
if (ymax ne -1) then ymax=ymax else ymax=max(yy)
yrange=[ymin,ymax]
if (yrange[0] lt 0) then yrange[0]=0.0
openpsplot,outfilename+'.ps',thick=2,fontsize=7
;; !p.thick=0.1                  ; thin curve
angstrom=cgsymbol("angstrom") ; Coyote cg, probably most robust
xtitle='wavelength  ['+angstrom+']'
ytitle=textoidl("intensity  [Watt cm^{-2} ster^{-1} ")+angstrom+'!u-1!n]'
if (tb ne 0) then ytitle='brightness temperature  [K]'
if (hz ne 0) then ytitle='intensity   [J m!U-2!N s!U-1!N Hz!U-1!N sr!U-1!N]'
if (wav2-wav1 ge 5) then tickformat='(I8)' else $ 
  if (wav2-wav1 ge 1) then tickformat='(f8.1)' else $
    if (wav2-wav1 ge 0.1) then tickformat='(f8.2)' else $
      if (wav2-wav1 ge 0.01) then tickformat='(f8.3)'
plot,neckelwav,yy,$
  position=[0.15,0.15,0.95,0.95],/normal,$     ; set margins around plot
  xticklen=0.03,yticklen=0.03/1.6,$            ; same-length ticks
  xtitle=xtitle,ytitle=ytitle,$
  xrange=xrange,xstyle=1,$
  xtickformat=tickformat,$
  xtickinterval=(wav2-wav1 ge 5)*!x.tickinterval + $ 
    (wav2-wav1 lt 5)*fix((wav2-wav1)/5.),$
  yrange=yrange,ystyle=1,_extra=plotkeywords

; optional label
if (label eq 0) then thislabel=''
if (label eq 1) then thislabel='FTS + Neckel calibration'
szlabel=size(label)
if (szlabel[1] eq 7) then thislabel=label
xyouts,0.55,0.84,/norm,alignment=0.5,charsize=1.3,label

; optionally add segwidth averaged rectangular curve (as histogram)
; used in Fig. 10 Vienna irradiance NLTE review 2019SoPh..294..165R
if (segwidth ne 0) then begin
  startseg=fix(wav1/segwidth)*segwidth
  endseg=fix(wav2/segwidth)*segwidth
  nseg=(endseg-startseg)/segwidth
  segwavs=fltarr(nseg)
  segmeans=fltarr(nseg)
  segwav1=indgen(nseg)*segwidth+startseg
  segwav2=segwav1+segwidth
  for iw1=0l,nneckel-1 do if (neckelwav[iw1] ge segwav1[0]) then break
  for iseg=0,nseg-1 do begin  
    for iw1=iw1,nneckel-1 do if (neckelwav[iw1] ge segwav1[iseg]) then break
    for iw2=iw1,nneckel-1 do if (neckelwav[iw2] ge segwav2[iseg]) then break
    segwavs[iseg]=(neckelwav[iw2]-neckelwav[iw1])/2.+neckelwav[iw1]
    segmeans[iseg]=avg(yy[iw1:iw2])
  endfor

; extend end segments
  segwavs=[segwav1[0],segwavs,segwav2[nseg-1]]
  segmeans=[segmeans[0],segmeans,segmeans[nseg-1]]

; overplot - AHA: psym=10 histogram mode
; color 255 = white for compresssed-atlas plots that are black with lines     
  oplot,segwavs,segmeans,psym=10,color=cgcolor(segcolor),thick=3

; write segwavs and segmans to file for overlay in RH plots
 writecol,outfilename+'.dat',segwavs,segmeans,fmt='(2f15.3)'
endif

; close ps plot unless nopsclose set
if (nopsclose ne 1) then closepsplot,outfilename+'.ps',opengv=1

end 


; =============== test/run per IDLWAVE Hyper-C ==========================

;; plotftsint,3290,12510,/Tb  ; full Neckel atlas

;; ; optical for vienna.tex
;; outfilename='/home/rutten/wrt_Tosh2/vienna/paper/robfigs/neckel-optical' 
;; plotftsint,3700,6700,/Tb,ymin=3200,ymax=7200,$   ; Dunn et al range
;;   outfilename=outfilename,label=1,nopsclose=1,segwidth=100,segcolor="yellow"
;; xyouts,3940,6800,textoidl("HK"),alignment=0.5
;; xyouts,6563,6300,textoidl("H\alpha"),alignment=0.5
;; xyouts,4861,6500,textoidl("H\beta"),alignment=0.5
;; xyouts,4340,6600,textoidl("H\gamma"),alignment=0.5
;; xyouts,4102,6600,textoidl("H\delta"),alignment=0.5
;; xyouts,3970,6600,textoidl("H\epsilon"),alignment=0.5
;; xyouts,5170,6450,textoidl("Mg b"),alignment=0.5
;; xyouts,5893,6350,textoidl("Na D"),alignment=0.5
;; xyouts,4310,6800,textoidl("CH"),alignment=0.5
;; xyouts,3888,6440,textoidl("CN"),alignment=1.0 ;; edge so leftsided 
;; closepsplot,outfilename+'.ps',opengv=0

;; ; ultraviolet vienna.tex
;; outfilename='/home/rutten/wrt_Tosh2/vienna/paper/robfigs/neckel-ultraviolet'
;; plotftsint,3300,3750,/Tb,ymin=3200,ymax=7200,$  
;;    outfilename=outfilename,label=1,nopsclose=0,segwidth=10,segcolor="yellow"

;; plotftsint,3290,3700,/Tb,ymin=3200,ymax=7200,$  ; shortest Neckel wide
;;            label=1,nopsclose=0,segwidth=100

;; plotftsint,3858,3861,/Tb,ymin=3200,ymax=7200,$  ; FeI mult 4
;;            label=1,nopsclose=0,segwidth=0

;; plotftsint,3858,3858.3,/Tb,ymin=3200,ymax=7200,$  ; FeI mult 4
;;            label=1,nopsclose=0,segwidth=0

; H-gamma 4340.5 no wav numbers beyond frame
plotftsint,4338.05,4342.95,outfilename='/tmp/neckel-H-gamma',label="H-gamma"

; HeI D3 5876.0 no wav numbers beyond frame
plotftsint,5873.55,5878.45,outfilename='/tmp/neckel-HeID3',label="He I D3"

end
