; file: limbangle.pro
; init: Aug 10 2014  Rob Rutten  Deil
; last: Aug 19 2020  Rob Rutten  Deil

;+
function limbangle,x,y

 ; return CW position angle of limbward direction for solar (X,Y) 
 ; 
 ; INPUTS:
 ;   x = solar X coordinate
 ;   y = solar Y coordinate
 ;
 ;     orthogonal coordinates of a solar location in arcsec on the
 ;     apparent solar disk seen from at- or near-Earth telescope,
 ;     from disk center: solar Y = along line from disk center though the
 ;     the solar North pole, solar X = perpendicular to Y, positive to
 ;     East
 ;
 ; OUTPUT:
 ;   angle NWSE (clockwise from North) of limbward direction in degrees
 ;
 ; USAGE example for plot:
 ;   ; params from record.idl: (X,Y)=-202,206   angle_gbo=62.6380
 ;   limbang=limbangle(-202,206)
 ;   centerangle=limbang-180.-62.6380
 ;   x0=0.83  ; trial-and-error location and size selection 
 ;   y0=0.87
 ;   x1=0.92
 ;   y1=y0+(x1-x0)*tan((90.-centerangle)*!pi/180.) 
 ;   arrow,x0,y0,x1,y1,/norm,hsize=-0.3,thick=3,color=255
 ;
 ; HISTORY:
 ;   Aug 10 2014 RR: start
 ;   Aug 19 2020 RR: it said NESW but it is NWSE = clockwise from North
;-

; answer no-parameter query
if (n_params() lt 2) then begin
  print,'  angle=limbangle(x,y)'
  return, -1
endif

ang1=atan(abs(float(y))/abs(float(x)))*180./!pi
if (x ge 0 and y ge 0) then limbangle=90.-ang1
if (x ge 0 and y lt 0) then limbangle=90.+ang1
if (x lt 0 and y lt 0) then limbangle=270.-ang1
if (x lt 0 and y ge 0) then limbangle=270.+ang1
if (x eq 0 and y eq 0) then begin
  print, ' ##### limbangle.pro: disk center, limb angle set to 0'
  limbangle=0
endif

return,limbangle

end

;  ============= test IDLWAVE ====================

X=-682.5
y=-186.3

x=-202
y=206
print,' limb angle = '+ntostr(limbangle(x,y))
print,' center direction = '+ntostr(limbangle(x,y)-180.)
end
