; file: doallfiles.pro
; site: idl/rridl/misclib
; init: Jul 27 2014  Rob Rutten  Deil
; last: Jun  8 2020  Rob Rutten  Deil

;+
pro doallfiles,indirpath,outdirpath,inselstring,outselstring,$
  procedure,_extra=doallkeywords

 ; NAME:
 ;   doallfiles.pro  
 ;
 ; PURPOSE:
 ;   run an IDL pro treating one file at a time for multiple files
 ;
 ; INPUTS:
 ;   indirpath = string with directory where the input files are,
 ;     or string array specifying inputfiles,  with inselstring=''
 ;   outdirpath = string with directory for the output files
 ;   inselstring = string with part of filename as file selector, eg '.fits'
 ;     or '' for all files in dir indirpath instead
 ;   outselstring = string with new last part for output files, eg '_rot.fits'
 ;     (may also not have an extension, then out extension same as input file)
 ;   procedure = string with name of the IDL procedure
 ;   keywords = required or optional keywords for that procedure
 ;
 ; RESTRICTION:
 ;   called procedure must have call: procedure,infile,outfile,keywords
 ;
 ; EXAMPLE:
 ;    doallfiles, '.', '.', '.fits', '_rot.fits', 'rotatefitscube', angle=30
 ;
 ; HISTORY:
 ;   Jul 27 2014 RR: start  
 ;   Sep 26 2015 RR: permit space in dirpath 
 ;   Jan  3 2018 RR: filelist as alternative input
;-

; answer no-parameter query
if (n_params() lt 5) then begin
  sp,doallfiles
  return
endif

; set wall-clock timer (seconds)
timestart=systime(1)

; get list of input files
if (inselstring eq '') then infilelist=findfile(indirpath+'/*.*') $ 
else infilelist=findfile(indirpath+'/*'+inselstring+'*')
if (infilelist[0] eq '') then begin
  print,' ##### warning: doallfiles found no file in '+indirpath+$
    ' with inselstring '+inselstring
  nfiles=-1
endif else nfiles=n_elements(infilelist)

; make outdir if not present
spawn, 'mkdir -p '+outdirpath

; process each file
for ifile=0,nfiles-1 do begin
  infile=infilelist[ifile] 

  ; add input file extension if regular and outselstring has no .ext
  indummy=cgrootname(infile,extension=inext)  ;RR needs coyotelib
  bareinfile=file_basename(infile,'.'+inext)
  outfile=outdirpath+'/'+bareinfile+outselstring
  outdummy=cgrootname(outfile,extension=outext)
  ; test for familiar ext because SST filenames can have multiple periods 
  if (not(outext eq 'fits' or outext eq 'bcube' or outext eq 'icube' $
          or outext eq 'fcube')) then outfile=outfile+'.'+inext
  
; print pacifier
  print,' ===== start on '+infile+' > '+outfile

; do this file
  call_procedure,procedure,infile,outfile,_extra=doallkeywords

endfor

; print elapsed time
timedone=systime(1)
timelaps=ntostr((timedone-timestart)/60.,format='(F11.3)')
print,' ===== doallfiles took '+timelaps+' minutes'

end

; =================== test for IDLWAVE Super-C ==========================

;; ; standard test with data on Tosh disk
;; cd,'/home/rutten/data/SDO/2014-06-14-small/target/cubes/' 
;; indirpath='.'
;; ;; indirpath=['aia304.fits','aia171.fits']
;; inselstring='aia*00'
;; ;; inselstring='' ; test all-file option
;; outdirpath='/tmp/rotcubes'
;; outselstring='_rot'
;; doallfiles,indirpath,outdirpath,inselstring,outselstring,'rotatefitscube',$
;;   angle=30

;; showex,'/home/rutten/data/SDO/2014-06-14-small/target/cubes/aia304.fits',$
;;         '/tmp/rotcubes/aia304_rot.fits'

; Vasco CHROMIS+CRISP
cd,'/media/rutten/SSTNEW/alldata/SST/2018-06-10-vasco'
indirpath='cair2rotsdo'
inselstring='crispex'
outdirpath='/tmp'
outselstring='_test'
doallfiles,indirpath,outdirpath,inselstring,outselstring,'rotatefitscube',$
   angle=30

end
