; file: croprotbox.pro
; init: Dec 17 2017  Rob Rutten  Deil
; last: Dec 25 2017  Rob Rutten  Deil

;+

pro croprotbox,nx,ny,angle,nxcrop,nycrop

 ; NAME:
 ;   croprotbox   
 ;
 ; PURPOSE:
 ;   Given a rectangle of size nx * ny that will get rotated over 'angle' 
 ;   degrees, compute the nxcrop and nycrop size of the largest
 ;   axis-aligned rectangle that fits within the rotated rectangle.
 ;
 ;   Reversely, for a just-contained rotated rectangle (image) enter
 ;   its bounding outer frame size to obtain the original non-rotated
 ;   rectangle size, then use that in the above forward manner.  If
 ;   needed make the outer frame just-containing by using
 ;   /cropborders in reformimage.pro.  This is all done within the
 ;   /greytriangles and /croptriangles options of reformmimage.pro.
 ;
 ; DESCRIPTION:
 ;   translations of various Python recipes at stackoverflow
 ;     
 ; CALL:
 ;   see above
 ;   
 ; INPUTS:
 ;   nx = x-size not-rotated original box/image (ususally image pixels)  
 ;   ny = y-size idem
 ;   angle = rotation angle in degrees (angle doesn't matter)
 ;  
 ; OUTPUTS:
 ;   nxcrop = x-size best-fit rectangle 
 ;   nycrop = y-size best-fit rectangle  
 ;   
 ; RESTRICTIONS:
 ;   failure around 45 degrees for reverse operation, then the trimbox
 ;   becomes square instead of having the non-rotated aspect ratio;
 ;   shared by all three versions below (which seem highly similar)
 ;   
 ; HISTORY:
 ;   Dec 25 2017 RR: start 
;-

; answer no-parameter query 
if (n_params(0) lt 5) then begin
  print,' croprotbox,nx,ny,angle,nxcrop,nycrop'
  sp,croprotbox
  return
endif

; convert angle to radians
angrad=angle*(!pi/180.)

goto,VERSION1  ; this is the best one (with my fix for 45-degree singularity)
;; goto,VERSION2
;; goto,VERSION3

VERSION3:
; ----------------------------------------- croprotbox3 
;   https://stackoverflow.com/questions/5789239/calculate-largest-rectangle-in-a-rotated-rectangle => version Ivan Korchurkin Aug 23 2013

if (nx le ny) then begin
  w0=nx
  h0=ny
endif else begin
  w0=ny
  h0=nx
endelse
ang=angrad-fix((angrad+!pi)/(2*!pi))*2*!pi
ang=abs(ang)
if (ang gt !pi/2) then ang=!pi-ang
sina=sin(ang)
cosa=cos(ang)
sincosa=sina*cosa
w1=w0*cosa+h0*sina
h1=w0*sina+h0*cosa
c=h0*sincosa/(2*h0*sincosa+w0)
x=w1*c
y=w1*c
if (nx le ny) then begin
  w=w1-2*x
  h=h1-2*y
endif else begin
  w=h1-2*y
  h=w1-2*x
endelse
nxcrop=w
nycrop=h
goto,DONE

VERSION2:
; --------------------------------------------------- croprotbox2
;   translation of Python recipe by Magnus Hoff at
;   https://stackoverflow.com/questions/5789239/calculate-largest-rectangle-in-a-rotated-rectangle

alpha=angrad
w=float(nx)*cos(alpha)+ny*sin(alpha)
h=float(ny)*sin(alpha)+ny*cos(alpha)

if (w lt h) then gamma=atan(w/h) else gamma=atan(h/w)
delta=!pi-alpha-gamma

if (w lt h) then length=h else length=w
d=length*cos(alpha)
a=d*sin(alpha)/sin(delta)

y=a*cos(gamma)
x=y*tan(gamma)

nxcrop=w-2*x
nycrop=h-2*y

nxcrop=x
nycrop=h

goto, DONE

VERSION1:
; --------------------------------------------------  croprotbox1
;   https://stackoverflow.com/questions/16702966/rotate-image-and-crop-out-black-borders/16778797#16778797
;   answer given May 27 2013 at 18:40 by member "coproc".

width_longer=(nx ge ny) 
if (width_longer) then begin
  side_long=nx 
  side_short=ny
endif else begin
  side_short=nx 
  side_long=ny
endelse
cosa=abs(cos(angrad))
sina=abs(sin(angrad))
if (side_short le 2.*sina*cosa*side_long $
   or abs(cosa-sina) lt 1.E-3)$   ;RR this one was not in the Python code 
then begin 
  x=0.5*side_short
  if (width_longer) then begin
    nxcrop=x/sina
    nycrop=x/cosa
  endif else begin
    nxcrop=x/cosa
    nycrop=x/sina
  endelse
endif else begin
  cos2a=cosa^2-sina^2
  nxcrop=(nx*cosa-ny*sina)/cos2a
  nycrop=(ny*cosa-nx*sina)/cos2a
endelse
goto,DONE

; --------------------------------- end for whatever version

DONE:
end


; =============== main for testing per IDLWAVE H-c ======================
; input 
path='/home/rutten/data/SST/tests/'
image=readfits(path+'sdo-2016-09-05-demo.fits')  
reformimage,image,image,yrange=[0,120] ; rectangular cutout
;; reformimage,image,image,xrange=[0,120],yrange=[0,120] ; square cutout

szimage=size(image)
nx=szimage[1]
ny=szimage[2]

; run program
angle=45
croprotbox,nx,ny,angle,nxcrop,nycrop

print,' ----- nx,ny,nxcrop,nycrop ='+trimd([nx,ny,nxcrop,nycrop])
if (finite(nxcrop) eq 0) then begin
  print,' ##### croprotbox abort: nxcrop = NaN'
  return
endif

; display result
reformimage,image,rotimage,rotate=angle,nxlarge=250,nylarge=250 ; "full"
;; reformimage,image,rotimage,rotate=angle,nxlarge=200,nylarge=200  ; 45 just
;; reformimage,image,rotimage,rotate=angle ; "partial"
;; reformimage,image,rotimage,rotate=angle,nxlarge=190,nylarge=150 ; corners

reformimage,rotimage,bordercropimage,/cropborders,cropbox=borderbox
; first show border-strip box
while !d.window ne -1 do wdelete,!d.window
sv,rotimage
cgplots,[borderbox[0],borderbox[2]],[borderbox[1],borderbox[1]],color='green'
cgplots,[borderbox[0],borderbox[2]],[borderbox[3],borderbox[3]],color='green'
cgplots,[borderbox[0],borderbox[0]],[borderbox[1],borderbox[3]],color='green'
cgplots,[borderbox[2],borderbox[2]],[borderbox[1],borderbox[3]],color='green'

; get nxrot,nyrot of rotated bordercropped image
szcroprot=size(bordercropimage)
nxrot=szcroprot[1]
nyrot=szcroprot[2]

; get the same for full-field-containing case = directly from (nx,ny)
rotvec1=rotatevec([nx/2.,ny/2.],angle)
rotvec2=rotatevec([nx/2.,-ny/2.],angle)
nxrot2=2*max([rotvec1[0],rotvec2[0]])
nyrot2=2*max([rotvec1[1],rotvec2[1]])
print,' ----- nxrot,nyrot,nyxrot2,nyrot2 ='+trimd([nxrot,nyrot,nyrot2,nyrot2])

; test how to get (nx,ny) from nxrot,nyrot for full-containment frame
croprotbox,nxrot,nyrot,angle,trynxorig,trynyorig
print,' ----- nx,ny,trynxorig,trynyorig ='+trimd([nx,ny,trynxorig,trynyorig])
; Eureka! ???  ornot?  but only for fully contained  

; now show recipe result
sv,bordercropimage
cropbox=[fix(nxrot/2.-nxcrop/2.),$
          fix(nyrot/2.-nycrop/2.),$
          fix(nxrot/2.+nxcrop/2.),$
          fix(nyrot/2.+nycrop/2.)]
cgplots,[cropbox[0],cropbox[2]],[cropbox[1],cropbox[1]],color='red'
cgplots,[cropbox[0],cropbox[2]],[cropbox[3],cropbox[3]],color='red'
cgplots,[cropbox[0],cropbox[0]],[cropbox[1],cropbox[3]],color='red'
cgplots,[cropbox[2],cropbox[2]],[cropbox[1],cropbox[3]],color='red'

end
