; file: brighttemp.pro
; last: Jun 20 2022  Rob Rutten  Deil

;+
function brighttemp,wav,intensity,Hz=Hz,AA=AA
 ;    compute brightness temperature from absolute intensity
 ;    straightforward but take care with units
 ; CALL:
 ;   see above
 ; INPUTS:
 ;    wav = wavelength in micron
 ;      nm if /Hz
 ;      Angstrom if /AA 
 ;    intensity = ergs / cm2 s ster [micron bandwidth] 
 ;      intensity = ergs / cm2 s ster [AA bandwidth] if /AA
 ;      intensity = J / m2 s Hz ster [Hzbandwidth] if /Hz 
 ;    wav and intensity may be single values or equal-size arrays
 ; OPTIONAL KEYWORD INPUTS
 ;    Hz = 1/0: wav in nm, intensity SI units or micron and cgs
 ;    AA = 1/0: wav in Angstrom, intensity cgs with AA bandwidth 
 ; OUTPUTS:
 ;   brightness temperature (array) in Kelvin
 ; HISTORY:
 ;   sometime ?    RR: start
 ;   May 22 2016   RR: this header     
;-

; answer no-parameter query
if (n_params() lt 2) then begin    ; N = nr required parameters
  print,'  usage: x=brighttemp(wav,intensity,Hz=hz,AA=AA)'
  print,'         wav in micron, in AA if /AA, in nm if /Hz'
  print,'         intensity in ergs / cm2 s ster [micron bandwidth]'
  print,'         intensity in ergs / cm2 s ster [AA bandwidth]  if /AA'
  print,'         intensity in J / m2 s Hz ster   if /Hz'
  return,0
endif

; default keywords
if (n_elements(Hz) eq 0) then Hz=0
if (n_elements(AA) eq 0) then AA=0

; error check
if (AA eq 1 and Hz eq 1) then begin
  print,' #### brighttemp error: do not set Hz and AA together'
  return,0
endif

; physics constants in cgs
k=1.380650D-16        ; Boltzmann constant (erg K; double precision)
h=6.626068D-27        ; Planck constant (erg s)
c=2.99792458D10       ; speed of light (cm/s)

if (Hz eq 0) then begin
  wavcm=wav*1D-4                   ; wav in micron to cm
  if (AA) then wavcm=wav*1D-8      ; wav in AA to cm 
  intcm=intensity*1D4              ; change into per cm bandwidth
  if (AA) then intcm=intensity*1D8 ; change into per cm bandwidth
  tempbright=h*c/(wavcm*k)/alog(2D0*h*c^2/(intcm*wavcm^5)+1D0)
endif

if (Hz eq 1) then begin 
  ; physics constants in SI
  k=k*1D-7   ; Boltzmann constant (J K)
  h=h*1D-7   ; Planck constant (J s)
  c=c*1D-2   ; speed of light (m/s)
  freq=c/(wav*1D-9)     ; freq in Hz  
;RR Mar  5 2012 trouble, 1D0 for long wavelengths
  tempbright=h*freq/(k*alog(2D0*h*(freq^3)/(intensity*(c^2))+1D0))
endif

return,tempbright
end

; ============== run per IDLWAVE Hyper-C ================================


print,brighttemp(363.,6.E-9,/Hz) ; ALC7 Ba cont(RH analyze): T_b = 5480
print,brighttemp([363.,364.],[6.E-9,7.E-9],/Hz) ; try arrays

end
