; file: saha_rr.pro = Saha for three successive stages
; note: after saha_mc.pro = saha.pro from the MULTI lib of Mats Carlsson
; init: Sep 27 2013  Rob Rutten
; last: Apr  2 2015  Rob Rutten  Deil

pro saha_rr,temp,eldens,xi,xii,u0,u1,u2,n0_ntot,n1_ntot,n2_ntot

;+
 ; NAME:
 ;   saha_rr
 ; PURPOSE:
 ;   compute Saha ratios N0/Ntot, N1/Ntot, N2/Ntot for first 3 stages
 ; DESCRIPTION:
 ;   see Gray III (1.22) p17; also Carlsson MULTI (2 stages only
 ; CALL:
 ;   saha,temp,eldens,xi,xii,u0,u1,u2,n0_ntot,n1_ntot,n2_ntot
 ; INPUTS:
 ;   temp    temperature [K] (single value or array)
 ;   eldens  electron density [particles/cm^3] (single or same-size array)
 ;   xi      ionization energy first stage [eV]
 ;   xii     ionization energy second stage [eV]
 ;   u0      partition function of neutral element
 ;   u1      partition function of once ionized element
 ;   u2      partition function of twice ionized element
 ; OUTPUTS:
 ;   n0_ntot  neutral fraction of total (single or array)
 ;   n1_ntot  once-ionized fraction of total (single or array)
 ;   n2_ntot  twice-ionized fraction of total (single or array)
 ; RESTRICTIONS:
 ;   all of the element must sit in these three stages
 ; HISTORY:
 ;   sometime 2013 RR: start
;-   Apr  2 2015 RR: cleanup, check array size equality

; answer no-parameter query 
if(n_params(0) lt 10) then begin
  print,'saha_rr,t,eldens,xi,xii,u0,u1,u2,n0_ntot,n1_ntot,n2_ntot'
  return
endif

; check array sizes
ntemp=n_elements(temp)
neldens=n_elements(eldens)
if (ntemp gt 1 and neldens gt 1 and ntemp ne neldens) then begin
  print,' #### ABORT saha_rr: temp and eldens inequal array sizes' 
  n0_ntot=-1
  n1_ntot=-1
  n2_ntot=-1
  return
endif

; physics constants
bk=1.38066D-16         ; Boltzann constant cgs; double precision

; Saha ratios after Gray III (1.22) p17; underscore = divide
theta=5039.77D0/temp       
n1_n0=0.6665D0*u1/u0*temp^2.5*10.^(-theta*xi)/(eldens*bk*temp)
n2_n1=0.6665D0*u2/u1*temp^2.5*10.^(-theta*xii)/(eldens*bk*temp) 

; rework into fractions of total
ntot_n1=(1./n1_n0)+1.+n2_n1
n0_ntot=(1./n1_n0)/ntot_n1
n1_ntot=1./ntot_n1
n2_ntot=n2_n1/ntot_n1

end

; =============== main for testing per IDLWAVE H-c ======================

temp=5000.
temp=[5000.,6000.,7000.] ; test array usage
eldens=1.E14 
eldens=[1.E14,1.E13,1.E12] ; test array usage
elemnr=8  ; O
elemnr=26 ; Fe  # oops Chianti aonle _1 files for OI and NI
ionstage=1     ; remember: ratio Fe+++ / Fe++ = FeIV / FeIII

atomdata_wsa,elemnr,atomwgt,abund,ion1,ion2,ion3,elemsymbol
u0=partfunc_rr(temp,elemnr,0)
u1=partfunc_rr(temp,elemnr,1)
u2=partfunc_rr(temp,elemnr,2)
saha_rr,temp,eldens,ion1,ion2,u0,u1,u2,n0_ntot,n1_ntot,n2_ntot
if (ionstage eq 0) then $
    print,' ==== saha_rr: ',ntostr(n1_ntot/n0_ntot,format='(G15.3)')
if (ionstage eq 1) then $
    print,' ==== saha_rr: ',ntostr(n2_ntot/n1_ntot,format='(G15.3)')

; compare with saha_ratio (but may use the _wsa routines)
 print,' ==== saha_ratio ',ntostr(saha_ratio(elemnr,temp,eldens,ionstage),$
  format='(G15.3)')

end
