; file: saha_ratio.pro 
; init: Mar 28 2015  Rob Rutten  Deil
; last: Jul 26 2018  Rob Rutten  Deil for Klimchuk
; note: startup chianti with "setssw chianti" in /home/rutten/bin/idl
;       then run per Hyper C

function saha_ratio,elemnr,temp,eldens,ionstage,verbose=verbose

;+
 ; NAME:
 ;   saha_ratio   
 ; PURPOSE:
 ;   compute Saha population ratio between two successive ionization stages
 ; CALL:
 ;   ratio=saha_ratio(elemnr,temp,eldens,ionstage,verbose=verbose)
 ; INPUTS:
 ;   elemnr = element number (hydrogen=1)
 ;   temp = temperature [K] (scalar or array)
 ;   eldens = electron density [cm^-3] (scalar or same-size array)
 ;   ionstage = lower of the two ionization stages (neutral=0)
 ; OPTIONAL KEYWORD INPUT
;    verbose 0/1: print not too fatal error messages too (default 0)
 ; OUTPUTS:
 ;   Saha ratio N_ion+1 / N_ion (array if temp and/or eldens is array)  
 ; RESTRICTIONS:
 ;   needs SWW Chianti and other Rob Rutten ltelib programs
 ;   for low stages at missing Chianti file it switches to partfunc_rr
 ; HISTORY:
 ;   Mar 28 2015 RR: start
;-

; answer no-parameter query 
if (n_params(0) lt 4) then begin
  print,'ratio=saha_ratio(elemnr,temp,eldens,ionstage,verbose=verbose)'
  return,-1  
endif

; defaults for keywords
if (not keyword_set(verbose)) then verbose=0

; various checks
if (elemnr gt 30) then begin
  print,' #### ABORT saha_ratio: element not in Chianti (only 1-30)'
  return,-2
endif
if (ionstage gt elemnr-1)  then begin
  print,' #### ABORT saha_ratio: ionstage > elemnr'
  return,-3
endif
ntemp=n_elements(temp)
neldens=n_elements(eldens)
if (ntemp gt 1 and neldens gt 1 and ntemp ne neldens) then begin
  print,' #### ABORT saha_ratio: temp and eldens inequal array sizes' 
  return,-4
endif

; get ionization energy from Chianti file
read_ip,!xuvtop+'/ip/chianti.ip',ip,ref
ionerg=ip[elemnr-1,ionstage]
; convert ionization energy from cm^-1 into eV
ergpercmwav=1.98649E-16   ; erg per cm-1 wave number
ergperev=1.60217657E-12   ; erg per electronvolt
ionerg=ionerg*ergpercmwav/ergperev

; compile partfunc_rr which may be eeded (stupid IDL parentheses issue)
forward_function partfunc_rr

; get partition function lower stage
u1=partfunc_chianti(elemnr,ionstage,temp)
if (min(u1) lt 0 and ionstage lt 3) then u1=partfunc_rr(temp,elemnr,ionstage)
if (min(u1) lt 0) then begin 
  if (verbose) then print,$
    ' ##### ABORT saha_ratio: no partition function for lower = '+$
    ntostr(ionstage)
  return,-5
endif

; get partition function next higher stage
u2=partfunc_chianti(elemnr,ionstage+1,temp)
if (min(u2) lt 0 and ionstage lt 2) then u2=partfunc_rr(temp,elemnr,ionstage+1)
if (min(u2) lt 0) then begin 
  if (verbose) then print,$
   ' ##### ABORT saha_ratio: no partition function for upper = '+$
    ntostr(ionstage+1)
  return,-6
endif

; Saha ratio after Gray III (1.22) p17
kcgs=1.380650D-16         ; Boltzmann constant (erg/deg; real men use cgs)
theta=5039.77D0/temp       
ratio=0.6665D0*u2/u1*temp^2.5*10.^(-theta*ionerg)/(eldens*kcgs*temp)

return,ratio
end


; =============== main for testing per IDLWAVE H-c ======================

temp=5000.
temp=[5000.,6000.,7000.] ; test array usage
eldens=1.E14 ; mid-photosphere (not H_tot density!)
eldens=[1.E14,1.E13,1.E12] ; test array usage
elemnr=26    ; Fe
ionstage=1     ; remember: ratio Fe+++ / Fe++ = FeIV / FeIII

rat=saha_ratio(elemnr,temp,eldens,ionstage)
print,' ==== saha_ratio: ',ntostr(rat)

; check against saha_rr.pro (only first three stages = first two ratios)
if (ionstage lt 2) then begin
  atomdata_wsa,elemnr,atomwgt,abund,ion1,ion2,ion3,elemsymbol
  u0=partfunc_rr(temp,elemnr,0)
  u1=partfunc_rr(temp,elemnr,1)
  u2=partfunc_rr(temp,elemnr,2)
  saha_rr,temp,eldens,ion1,ion2,u0,u1,u2,n0_ntot,n1_ntot,n2_ntot
  if (ionstage eq 0) then $
    print,' ==== saha_rr:  ',ntostr(n1_ntot/n0_ntot,format='(G15.3)')
  if (ionstage eq 1) then $
    print,' ==== saha_rr:    ',ntostr(n2_ntot/n1_ntot,format='(G15.3)')
endif

end
